/*
 * Decompiled with CFR 0.152.
 */
package de.bsvrz.ars.ars.mgmt.tasks;

import com.google.common.base.Joiner;
import com.google.common.collect.Multimap;
import com.google.common.collect.MultimapBuilder;
import com.google.common.collect.Multimaps;
import de.bsvrz.ars.ars.mgmt.ArchiveManager;
import de.bsvrz.ars.ars.mgmt.LongTermTaskStatePublisher;
import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SyncKey;
import de.bsvrz.ars.ars.mgmt.tasks.CronTask;
import de.bsvrz.ars.ars.mgmt.tasks.MirrorDefinition;
import de.bsvrz.ars.ars.mgmt.tasks.base.Task;
import de.bsvrz.ars.ars.mgmt.tasks.base.TaskManager;
import de.bsvrz.ars.ars.mgmt.tasks.base.TaskState;
import de.bsvrz.ars.ars.persistence.CacheManager;
import de.bsvrz.ars.ars.persistence.DataIdentificationDir;
import de.bsvrz.ars.ars.persistence.IdDataIdentification;
import de.bsvrz.ars.ars.persistence.LockedContainerDirectory;
import de.bsvrz.ars.ars.persistence.PersistenceException;
import de.bsvrz.ars.ars.persistence.StartupProperties;
import de.bsvrz.ars.ars.persistence.directories.ActivePersistenceDirectory;
import de.bsvrz.ars.ars.persistence.directories.PersistenceDirectory;
import de.bsvrz.ars.ars.persistence.directories.mgmt.lock.DirectoryStatus;
import de.bsvrz.ars.ars.persistence.directories.mgmt.lock.LockFileManager;
import de.bsvrz.ars.ars.persistence.util.FileSync;
import de.bsvrz.ars.ars.persistence.util.SyncResult;
import de.bsvrz.ars.ars.persistence.walk.DataIdentificationDirAction;
import de.bsvrz.ars.ars.persistence.walk.DataIdentificationDirWalk;
import de.bsvrz.ars.ars.persistence.walk.internal.StatusPrinter;
import de.bsvrz.dav.daf.util.cron.CronDefinition;
import de.bsvrz.dav.daf.util.cron.CronScheduler;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Properties;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public final class MirrorTask
extends CronTask {
    private static final Debug _debug = Debug.getLogger();
    private static final String[] BYTE_UNIT = new String[]{" Byte", " KiB", " MiB", " GiB", " TiB", " PiB", " EiB"};
    private final CronScheduler _refreshScheduler = new CronScheduler();
    private final Task _longTermTaskStatePublisherTask;
    private final LockFileManager lockFileManager = new LockFileManager();

    public MirrorTask(ArchiveManager archiveMgr, LongTermTaskStatePublisher longTermTaskStatePublisher) {
        super(archiveMgr);
        this.setName("Datenkonsistentes Backup");
        this._refreshScheduler.schedule(this::refreshTasks, CronDefinition.EVERY_HOUR);
        this.refreshTasks();
        this._longTermTaskStatePublisherTask = longTermTaskStatePublisher.registerTask("Datenkonsistentes Backup");
        this._longTermTaskStatePublisherTask.setTaskState(TaskState.IDLE);
    }

    private void refreshTasks() {
        this.setCronTaskDefinitions(this.refreshCronTasks());
    }

    @Override
    public void terminateTask() {
        super.terminateTask();
        this._refreshScheduler.shutdownNow();
        this._longTermTaskStatePublisherTask.setTaskState(TaskState.TERMINATED);
    }

    private List<CronTask.CronTaskDefinition> refreshCronTasks() {
        try {
            Path backupDir = this.getPersistenceManager().getBackupConfigurationDirectory();
            Files.createDirectories(backupDir, new FileAttribute[0]);
            final ArrayList definitions = new ArrayList();
            try (Stream<Path> list = Files.list(backupDir);){
                list.forEach(new Consumer<Path>(this){

                    @Override
                    public void accept(Path path) {
                        if (path.getFileName().toString().toLowerCase(Locale.ROOT).endsWith(".properties")) {
                            CronDefinition cronDefinition;
                            Path targetBaseDir;
                            Properties properties = new Properties();
                            try (InputStream inStream = Files.newInputStream(path, new OpenOption[0]);){
                                properties.load(inStream);
                            }
                            catch (IOException e) {
                                _debug.warning("Datei " + String.valueOf(path) + " konnte nicht gelesen werden.", (Throwable)e);
                                return;
                            }
                            String cronString = properties.getProperty("cron");
                            String targetString = properties.getProperty("ziel");
                            if (cronString == null) {
                                _debug.warning("Datei " + String.valueOf(path) + " konnte nicht gelesen werden: Sie hat keinen Parameter \"cron\".");
                                return;
                            }
                            if (targetString == null) {
                                _debug.warning("Datei " + String.valueOf(path) + " konnte nicht gelesen werden: Sie hat keinen Parameter \"ziel\".");
                                return;
                            }
                            try {
                                targetBaseDir = Paths.get(targetString.trim(), new String[0]);
                            }
                            catch (InvalidPathException e) {
                                _debug.warning("Das in Datei \"" + String.valueOf(path) + "\" angegebene Ziel \"" + targetString + "\" ist kein g\u00fcltiger Pfadname. Die Backupdefinition wird ignoriert.", (Throwable)e);
                                return;
                            }
                            String commandString = properties.getProperty("cmd");
                            String[] commandLine = this.getCommand(commandString);
                            try {
                                cronDefinition = new CronDefinition(cronString);
                            }
                            catch (IllegalArgumentException e) {
                                _debug.warning("Der in Datei \"" + String.valueOf(path) + "\" angegebene Cron-String \"" + cronString + "\" ist ung\u00fcltig. Die Backupdefinition wird ignoriert.", (Throwable)e);
                                return;
                            }
                            MirrorDefinition e = new MirrorDefinition(cronDefinition, targetBaseDir, commandLine);
                            definitions.add(e);
                        }
                    }

                    private String[] getCommand(String command) {
                        if (command == null) {
                            return new String[0];
                        }
                        if (System.getProperty("os.name", "unknown").startsWith("Windows")) {
                            return new String[]{"cmd", "/C", command};
                        }
                        return new String[]{"sh", "-c", command};
                    }
                });
            }
            return definitions.stream().map(this::createMirrorCronTask).collect(Collectors.toList());
        }
        catch (IOException e) {
            _debug.warning("Kann Backup-Definitionen nicht einlesen", (Throwable)e);
            return Collections.emptyList();
        }
    }

    private CronTask.CronTaskDefinition createMirrorCronTask(MirrorDefinition mirrorDefinition) {
        return new CronTask.CronTaskDefinition(mirrorDefinition.getCronDefinition(), () -> {
            TaskManager.run(this._longTermTaskStatePublisherTask, tpi -> this.doMirror(mirrorDefinition.getTargetDirectory(), mirrorDefinition.getCommandLine()));
            this._longTermTaskStatePublisherTask.setTaskState(TaskState.IDLE);
        });
    }

    public void doMirror(Path targetBaseDir, String[] commandLine) {
        if (this.shouldTerminate()) {
            return;
        }
        try {
            Thread.currentThread().setName("Datenkonsistentes Backup nach " + String.valueOf(targetBaseDir));
            Files.createDirectories(targetBaseDir, new FileAttribute[0]);
            Path startUpInfoFile = targetBaseDir.resolve("_startUpInfo.property");
            Path restartTimeFile = targetBaseDir.resolve("_restartTime.property");
            Files.deleteIfExists(startUpInfoFile);
            Files.deleteIfExists(restartTimeFile);
            Path sourceRootPath = this.getPersistenceManager().getRootPath();
            MirrorAction lastResult = null;
            List<? extends PersistenceDirectory> persistenceDirectories = this.getPersistenceManager().getPersistenceDirectories(0);
            for (PersistenceDirectory persistenceDirectory : persistenceDirectories) {
                Path relativePath;
                Path targetPath;
                MirrorAction result = this.doPersistenceDirectoryMirror(persistenceDirectory, targetPath = targetBaseDir.resolve(relativePath = sourceRootPath.relativize(persistenceDirectory.getBasePath())));
                lastResult = result != null && result.isSuccessful() ? result : null;
            }
            new FileSync().syncDirectories(sourceRootPath.resolve("backup"), targetBaseDir.resolve("backup"));
            if (this.shouldTerminate()) {
                return;
            }
            if (lastResult != null) {
                this.writeStartUpInfo(lastResult, targetBaseDir);
                this.writeRestartTime(lastResult, targetBaseDir);
            }
            if (commandLine.length > 0) {
                MirrorTask.runExternalProcess(targetBaseDir, commandLine);
            }
        }
        catch (Exception e) {
            _debug.error("Unerwarteter Fehler beim konsistenten Backup", (Throwable)e);
        }
    }

    private MirrorAction doPersistenceDirectoryMirror(PersistenceDirectory sourceDirectory, Path targetDirectory) throws Exception {
        DirectoryStatus status;
        if (Files.exists(targetDirectory, new LinkOption[0])) {
            try {
                status = this.lockFileManager.open(targetDirectory);
            }
            catch (FileAlreadyExistsException ex) {
                this.lockFileManager.forceWritable(targetDirectory);
                status = DirectoryStatus.Writable;
            }
        } else {
            this.lockFileManager.createWritable(targetDirectory);
            status = DirectoryStatus.Writable;
        }
        if (status == DirectoryStatus.Finalized) {
            return null;
        }
        final MirrorAction action = new MirrorAction(sourceDirectory, targetDirectory);
        DataIdentificationDirWalk walk = DataIdentificationDirWalk.allDirectories(sourceDirectory);
        walk.setPublisherTask(this._longTermTaskStatePublisherTask);
        walk.setStatusPrinter(new StatusPrinter(this){

            @Override
            public String getStatusMessage(String actionName, Duration runtime, StatusPrinter.ApproximationType approximationType, long allContainerDirs, long visitedContainerDirs, long visitedContainerFiles) {
                return super.getStatusMessage(actionName, runtime, approximationType, allContainerDirs, visitedContainerDirs, visitedContainerFiles) + MirrorTask.getMessage(action, runtime);
            }

            @Override
            public String getSuccessMessage(String actionName, Duration runtime, long visitedContainerDirs, long visitedContainerFiles) {
                return super.getSuccessMessage(actionName, runtime, visitedContainerDirs, visitedContainerFiles) + MirrorTask.getMessage(action, runtime);
            }
        });
        walk.execute("Datenkonsistentes Backup von " + String.valueOf(sourceDirectory) + " nach " + String.valueOf(targetDirectory), 1, action);
        if (action.isSuccessful()) {
            if (!(sourceDirectory instanceof ActivePersistenceDirectory)) {
                this.lockFileManager.writeProtectPermanently(targetDirectory);
            }
            this.lockFileManager.close(targetDirectory);
        }
        return action;
    }

    @NotNull
    private static String getMessage(MirrorAction action, Duration runtime) {
        String byteSize = MirrorTask.formatByteSize(action.getNumBytesCopied());
        String speed = MirrorTask.formatSpeed((double)action.getNumBytesCopied() / ((double)runtime.getSeconds() + (double)runtime.getNano() / 1.0E9));
        return "\n " + MirrorTask.getFileCountString(action.getNumFilesCopied()) + " (" + byteSize + "; " + speed + ") kopiert.\n " + MirrorTask.getFileCountString(action.getNumFilesDeleted()) + " gel\u00f6scht.";
    }

    private static String formatSpeed(double bytesPerSecond) {
        if (!Double.isFinite(bytesPerSecond)) {
            return "?";
        }
        double bytes_double = bytesPerSecond;
        for (String unit : BYTE_UNIT) {
            if (bytes_double < 1024.0) {
                return String.format("%.1f", bytes_double) + unit + "/s";
            }
            bytes_double /= 1024.0;
        }
        return bytesPerSecond + BYTE_UNIT[0] + "/s";
    }

    @NotNull
    private static String formatByteSize(long bytes) {
        double bytes_double = bytes;
        for (String unit : BYTE_UNIT) {
            if (bytes_double < 1024.0) {
                return String.format("%.1f", bytes_double) + unit;
            }
            bytes_double /= 1024.0;
        }
        return bytes + BYTE_UNIT[0];
    }

    @NotNull
    private static String getFileCountString(long numFiles) {
        if (numFiles == 0L) {
            return "Keine Dateien";
        }
        if (numFiles == 1L) {
            return "Eine Datei";
        }
        return numFiles + " Dateien";
    }

    private void writeStartUpInfo(MirrorAction action, Path targetBaseDir) throws PersistenceException {
        StartupProperties startupProperties = new StartupProperties(targetBaseDir);
        startupProperties.setVal(StartupProperties.STUP_LAST_ATIME, action.getLastATime());
        startupProperties.setVal(StartupProperties.STUP_MAX_CONT_ID, action.getMaxContainerId());
        startupProperties.writeStartUpProperties();
    }

    private void writeRestartTime(MirrorAction action, Path targetBaseDir) {
        this.getPersistenceManager().saveUnsubscriptionTime(action.getArchiveTimes(), targetBaseDir.resolve("_restartTime.property"));
    }

    private static void runExternalProcess(Path targetBaseDir, String[] commandLine) throws IOException, InterruptedException {
        _debug.info("F\u00fchre Befehl aus: '" + Joiner.on((String)"' '").join((Object[])commandLine) + "'");
        ProcessBuilder processBuilder = new ProcessBuilder(new String[0]).command(commandLine);
        processBuilder.directory(targetBaseDir.toFile());
        processBuilder.inheritIO();
        Process process = processBuilder.start();
        process.waitFor();
    }

    private final class MirrorAction
    implements DataIdentificationDirAction {
        private final Path _targetBaseDir;
        private final AtomicLong _lastATime = new AtomicLong();
        private final AtomicLong _maxContainerId = new AtomicLong();
        private final AtomicLong _numFilesDeleted = new AtomicLong();
        private final AtomicLong _numBytesCopied = new AtomicLong();
        private final AtomicLong _numFilesCopied = new AtomicLong();
        private boolean _successful = true;
        private final Multimap<Long, IdDataIdentification> _archiveTimes = Multimaps.synchronizedMultimap((Multimap)MultimapBuilder.treeKeys().arrayListValues().build());
        private final PersistenceDirectory _persistenceDirectory;

        private MirrorAction(PersistenceDirectory sourceDirectory, Path targetDirectory) {
            this._persistenceDirectory = sourceDirectory;
            this._targetBaseDir = targetDirectory;
        }

        @Override
        public void run(DataIdentificationDir dataIdentificationDir, DataIdentificationDirWalk walk) {
            if (MirrorTask.this.shouldTerminate()) {
                this._successful = false;
                walk.terminate();
                return;
            }
            MirrorTask.this.suspendTaskIfNecessary();
            this._archiveTimes.put((Object)System.currentTimeMillis(), (Object)dataIdentificationDir.getDataIdentification());
            try (SyncKey<IdDataIdentification> indexLock = MirrorTask.this.getPersistenceManager().lockIndex(dataIdentificationDir.getDataIdentification());){
                for (LockedContainerDirectory lockedContainerDirectory : dataIdentificationDir.getContainerDirectories(indexLock)) {
                    new MirrorDirectoryAction(this._persistenceDirectory, lockedContainerDirectory).execute();
                }
            }
            catch (IOException e) {
                this._successful = false;
                _debug.warning("Verzeichnis " + String.valueOf(dataIdentificationDir) + " konnte nicht vollst\u00e4ndig gesichert werden. Fehler beim Kopieren der Dateien", (Throwable)e);
            }
            catch (Exception e) {
                this._successful = false;
                _debug.warning("Verzeichnis " + String.valueOf(dataIdentificationDir) + " konnte nicht vollst\u00e4ndig gesichert werden.", (Throwable)e);
            }
        }

        public long getLastATime() {
            return this._lastATime.get();
        }

        public long getMaxContainerId() {
            return this._maxContainerId.get();
        }

        public Multimap<Long, IdDataIdentification> getArchiveTimes() {
            return this._archiveTimes;
        }

        public long getNumFilesCopied() {
            return this._numFilesCopied.get();
        }

        public long getNumFilesDeleted() {
            return this._numFilesDeleted.get();
        }

        public long getNumBytesCopied() {
            return this._numBytesCopied.get();
        }

        public boolean isSuccessful() {
            return this._successful;
        }

        private class MirrorDirectoryAction {
            private final LockedContainerDirectory containerDirectory;
            private final PersistenceDirectory persistenceDirectory;

            public MirrorDirectoryAction(PersistenceDirectory persistenceDirectory, LockedContainerDirectory containerDirectory) {
                this.persistenceDirectory = persistenceDirectory;
                this.containerDirectory = containerDirectory;
            }

            public void execute() throws Exception {
                PersistenceDirectory persistenceDirectory = this.persistenceDirectory;
                if (persistenceDirectory instanceof ActivePersistenceDirectory) {
                    ActivePersistenceDirectory activePersistenceDirectory = (ActivePersistenceDirectory)persistenceDirectory;
                    this.flushCache(activePersistenceDirectory);
                    this.refreshIndex(activePersistenceDirectory);
                    MirrorAction.this._lastATime.accumulateAndGet(activePersistenceDirectory.maxATime(this.containerDirectory), Math::max);
                    MirrorAction.this._maxContainerId.accumulateAndGet(this.persistenceDirectory.maxContainerId(this.containerDirectory), Math::max);
                }
                this.copyFiles();
            }

            private void flushCache(ActivePersistenceDirectory persistenceDirectory) throws Exception {
                CacheManager.Cache cache;
                long openContID = persistenceDirectory.getOpenContID(this.containerDirectory);
                if (openContID != -1L && (cache = CacheManager.getInstance().getCache(openContID)) != null) {
                    cache.flush();
                }
            }

            private void refreshIndex(ActivePersistenceDirectory persistenceDirectory) {
                persistenceDirectory.flushIndexes(this.containerDirectory);
            }

            private void copyFiles() throws IOException {
                Path path = this.persistenceDirectory.getPath(this.containerDirectory);
                Path targetPath = MirrorAction.this._targetBaseDir.resolve(this.persistenceDirectory.getBasePath().relativize(path));
                Files.createDirectories(targetPath, new FileAttribute[0]);
                Path rebuildFlagFile = targetPath.resolve("_rebuildIndex.flag");
                if (!Files.exists(rebuildFlagFile, new LinkOption[0])) {
                    Files.createFile(rebuildFlagFile, new FileAttribute[0]);
                }
                SyncResult syncResult = new FileSync(this){

                    @Override
                    protected boolean shouldIgnore(String fileName) {
                        if ("_rebuildIndex.flag".equals(fileName) || "_backupindex.idx".equals(fileName)) {
                            return true;
                        }
                        return fileName.endsWith(".defect");
                    }
                }.syncDirectories(path, targetPath);
                MirrorAction.this._numFilesCopied.addAndGet(syncResult.getFilesCopied());
                MirrorAction.this._numFilesDeleted.addAndGet(syncResult.getFilesRemoved());
                MirrorAction.this._numBytesCopied.addAndGet(syncResult.getBytesCopied());
                Files.deleteIfExists(rebuildFlagFile);
            }
        }
    }
}

