/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import de.bsvrz.ars.ars.mgmt.ArchivConfig.SubscrMarker.SubscrInfo;
import de.bsvrz.ars.ars.mgmt.datatree.DataIdentNode;
import de.bsvrz.ars.ars.mgmt.datatree.DataIdentTree;
import de.bsvrz.ars.ars.mgmt.simulation.SimulationResultData;
import de.bsvrz.ars.ars.mgmt.tasks.QueueTask;
import de.bsvrz.ars.ars.mgmt.tasks.RepeatingTask;
import de.bsvrz.ars.ars.persistence.CacheManager;
import de.bsvrz.ars.ars.persistence.IdDataIdentification;
import de.bsvrz.ars.ars.persistence.util.SignalingQueue;
import de.bsvrz.ars.ars.persistence.writer.CloseContainerObject;
import de.bsvrz.dav.daf.main.*;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.dataIdentificationSettings.DataIdentification;
import de.bsvrz.sys.funclib.dataIdentificationSettings.EndOfSettingsListener;
import de.bsvrz.sys.funclib.dataIdentificationSettings.SettingsManager;
import de.bsvrz.sys.funclib.dataIdentificationSettings.UpdateListener;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.bsvrz.sys.funclib.losb.ringbuffer.RingBuffer;
import de.bsvrz.sys.funclib.losb.util.K;
import de.bsvrz.sys.funclib.losb.util.Util;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.util.*;
import java.util.concurrent.TimeoutException;

/**
 * Konfiguration des Archivsystems. Die Klasse ist als Thread implementiert, damit die Anmeldungen nicht im Thread des Datenverteilers (in den Callback-Methoden
 * des SettingsManager) laufen müssen. Stattdessen wird ein einfacher Ping-Pong-Mechanismus implementiert, mit dem die Parameter abgearbeitet werden, sobald
 * der EndOfSettingsListener aufgerufen wurde. Das Ende eines Parametrierungszyklus kann durch {@link #createEndOfParamListener()} abgewartet werden.
 * <p>
 * In einem Parametrierungszyklus werden alle An-/Abmeldungen von zu archivierenden Datenidentifikationen und Quittierungs-Datenidentifikationen vorgenommen.
 * <p>
 * Diese Klasse ist nicht mithilfe von {@link RepeatingTask} und {@link RingBuffer} implementiert, da sie einen von der Kernsoftware zur Verfuegung
 * gestellten Mechanismus zur Parameterübernahme verwendet.
 *
 * @author beck et al. projects GmbH
 * @author Alexander Schmidt
 * @version $Revision$ / $Date$ / ($Author$)
 */
public class ArchivConfig extends QueueTask<ArchivConfig.SubscrMarker> {

	private static final String ARS_PARAM_ATG_PID = "atg.archiv";

	private static final String ARS_PARAM_DELETE_ATG_PID = "atg.archivDatenEndgültigLöschen";

	private static final String ARS_PARAM_ASP_PID = "asp.parameterSoll";

	private static final String DEFAULT_PARAMS_FILE = "ArchivsystemParameter.txt";

	/** Bei Ausgeben der Parametrierung wird diese Zeitspanne auf den Parameter-Datensatz gewartet */
	private static final long PARAM_TIMEOUT = 10000;

	private boolean paramsAvailable;

	private InQueuesMgr inQueuesMgr;

	private int totalDIDs;

	@Nullable
	private SettingsManager settingsMgr;

	private EndOfSettingsListener eofSettingsLstnr;

	private UpdateListener updateSettingsLstnr;

	@Nullable
	private FileOutputStream paramDumpStream;

	private boolean printParams;

	private boolean noAnswer;

	private boolean gotFirstPrintParam;

	/** Listeners die benachrichtigt werden, wenn eine Parametrierung beendet ist. Diese Liste wird nach jedem Durchlauf geleert. */
	private final List<EndOfParamListener> endOfParamListeners = new ArrayList<>();

	/** Gibt an, ob schon eine Applikationsfertigmeldung versendet wurde */
	private boolean isReady;

	/** Hilfs-Task, der Simulations-parametrierungen übernimmt */
	private SimParamTask simParamWorker;

	/**
	 * Empfänger für die Parameterdaten fürs endgültige Löschen
	 */
	private PermanentDeleteReceiver _permanentDeleteReceiver;

	/**
	 * Erstellt einen neuen ArchivConfig.
	 *
	 * @param aMgr Der {@link ArchiveManager}
	 */
	public ArchivConfig(ArchiveManager aMgr) {
		super(aMgr, new SignalingQueue<>());
	}

	public void setSimConfigQueue(SignalingQueue<ResultData> queue) {
		simParamWorker = new SimParamTask(getArchMgr(), queue);
		simParamWorker.start();
	}

	private SubscrMarker markerTmp = new SubscrMarker();

	private SettingsManager createSettingsMgr() {
		SettingsManager sMgr = new SettingsManager(
				getArchMgr().getDavCon(), new DataIdentification(
				getArchMgr().getArchiveObject(), getArchMgr().getDD(ARS_PARAM_ATG_PID, ARS_PARAM_ASP_PID)
		)
		);

		// Callback-Methode: Wird vom DAV aufgerufen, wenn eine an/abzumeldende DID identifiziert wurde
		updateSettingsLstnr = (did, oldD, newD) -> {
			SubscrMarker subscrMarker = ArchivConfig.this.markerTmp;
			try {
				// Falls das An/Abmelden noch nicht fertig ist, blockieren wir die update()-Methode ein wenig.
				synchronized (ArchivConfig.this) {
					while (paramsAvailable) {
						try {
							ArchivConfig.this.wait();
						} catch (InterruptedException e) {
							return;
						}
					}
				}
				totalDIDs++;
				if (printParams) {
					printParam(paramDumpStream, String.valueOf(totalDIDs), did, newD);
				} else {
					// Erst Parametrierung übernehmen, dann an/abmelden. Dadurch kann man falls nötig
					// DIDs zu schwungweiser An-/Abmeldung zusammenfassen (wird derzeit nicht gemacht).

					if (oldD == null && newD != null)        // Fall 1: Neu parametriert
					{
						prepareNewParam(subscrMarker, did, newD);
					} else if (oldD != null && newD == null)    // Fall 2: Nicht mehr parametriert
					{
						prepareUnparam(subscrMarker, did);
					} else if (oldD != null) //  && newD != null  Fall 3: Umparametriert
					{
						prepareReparam(subscrMarker, did, newD);
					} else {
						_debug.warning("Archivparametrierung: ungueltige Parameter (oldD==newD==null)");
					}
				}
			} catch (Exception npe) {
				Debug.getLogger().warning("Fehler beim Auswerten der eingehenden Daten. Daten werden ignoriert! " + did + ", " + oldD + ", " + newD, npe);
			}
		};
		sMgr.addUpdateListener(updateSettingsLstnr);

		// Callback-Methode: Wird vom DAV aufgerufen, sobald alle An/Abzumeldungen beendet sind
		eofSettingsLstnr = () -> {
			submit(markerTmp);
			markerTmp = new SubscrMarker();
		};
		sMgr.addEndOfSettingsListener(eofSettingsLstnr);
		return sMgr;
	}

	@Override
	protected void work(final SubscrMarker step) throws InterruptedException {
		processParams(step);
		synchronized(endOfParamListeners) {
			for(EndOfParamListener eopl : endOfParamListeners) {
				eopl.gotParams();
			}
			endOfParamListeners.clear();
		}

		synchronized(this) {
			paramsAvailable = false;

			notifyAll();		// den ggf. wartenden UpdateListener weitermachen lassen
		}
	}

	/**
	 * Bestimmt die zusätzliche Vorhaltezeit für das endgültige Löschen für die angegebene Datenidentifikation.
	 * @param dataIdentNode DataIdentNode der Datenidentifikation
	 * @return zusätzliche Vorhaltezeit in Sekunden (&gt; 0). -1 falls nicht endgültig gelöscht werden soll.
	 * 0 und andere negative Werte werden nicht zurückgegeben.
	 */
	public long getPermanentDeleteTimeSeconds(final DataIdentNode dataIdentNode) {

		PermanentDeleteReceiver permanentDeleteReceiver = _permanentDeleteReceiver;

		if(permanentDeleteReceiver == null) return -1;

		IdDataIdentification dataIdentification = dataIdentNode.getDataIdentification();

		// Normal parametrierte Zeit ermitteln
		long parameterizedTime = permanentDeleteReceiver.getParameterizedDeleteExtraTimeSeconds(dataIdentification);

		// Es ist ein parametrierter Wert vorhanden
		if(parameterizedTime > 0) {
			return parameterizedTime;
		}

		// Die angegebene Datenidentifikation wurde nicht parametriert. Es gibt 2 Möglichkeiten:
		// * Die Daten sollen nicht gelöscht werden
		// * Das Objekt, zu denen die Daten gehören wurde endgültig gelöscht (z.B. transientes dynamisches Objekt oder Funktion endgültiges Löschen der Konf.)
		//
		// Im letzten Fall kann der Objekttyp nicht mehr ermittelt werden und es wird das Standard-Löschintervall für nicht bekannte benutzt

		if(isPermanentlyDeleted(dataIdentification)) {

			// Objekt ist nicht bekannt

			return permanentDeleteReceiver.getDeleteExtraTimeSecondsForUnresolvable();
		}

		return -1;
	}

	/**
	 * Prüft, ob mindestens ein Bestandteil einer Datenidentifikation endgültig gelöscht wurde.
	 * @param dataIdentification Datenidentifikation
	 * @return true falls nicht auflösbar, sonst false.
	 */
	private boolean isPermanentlyDeleted(final IdDataIdentification dataIdentification) {
		// In der Regel sollten nur Objekte vollständig gelöscht werden, aber die Prüfung für
		// Attributgruppen und Aspekte schadet auch nicht, um obsolete Daten zu bereinigen.
		return isPermanentlyDeleted(dataIdentification.getObjectId())
				|| isPermanentlyDeleted(dataIdentification.getAtgId())
				|| isPermanentlyDeleted(dataIdentification.getAspectId());
	}

	private boolean isPermanentlyDeleted(final long objectId) {
		DataModel dataModel = getArchMgr().getDataModel();
		return dataModel != null && dataModel.getObject(objectId) == null;
	}

	private class PermanentDeleteReceiver implements ClientReceiverInterface {
		private final DataDescription _dataDescription;
		private List<ParameterizedBlock> _parameterizedBlocks = ImmutableList.of();
		private long _deleteExtraTimeSecondsForUnresolvable = -1;

		public PermanentDeleteReceiver(final DataDescription dataDescription) {
			_dataDescription = dataDescription;
		}

		public DataDescription getDataDescription() {
			return _dataDescription;
		}

		@Override
		public void update(final ResultData[] results) {
			for(ResultData result : results) {
				Data data = result.getData();
				if(data != null) {
					getArchMgr().getTaskScheduler().setDeletePermanentlyInterval(
							data.getTextValue("Löschintervall").getText(),
							"Ja".equals(data.getTextValue("Aktiv").getText())
					);
					_parameterizedBlocks = parseParameterizedBlocks(data);
					_deleteExtraTimeSecondsForUnresolvable = data.getTimeValue("VorhaltezeitEndgültigGelöschteObjekte").getSeconds();
				}
			}
		}

		private List<ParameterizedBlock> parseParameterizedBlocks(final Data data) {
			final List<ParameterizedBlock> result = new ArrayList<>();
			for(Data datum : data.getItem("ParameterSatz")) {
				Set<String> areaPids = getPids(datum.getReferenceArray("Bereich"));
				List<ParameterizedBlockTuple> blocks = new ArrayList<>();
				Data dataSpecificationArray = datum.getItem("DatenSpezifikation");
				for(Data dataSpecBlock : dataSpecificationArray) {
					blocks.add(parseParameterizedBlockTuple(dataSpecBlock));
				}
				long timeSeconds = datum.getTimeValue("ZusätzlicheVorhaltezeit").getSeconds();
				result.add(new ParameterizedBlock(areaPids, blocks, timeSeconds));
			}
			return result;
		}

		private ParameterizedBlockTuple parseParameterizedBlockTuple(final Data datum) {
			Set<String> objectPids = getPids(datum.getReferenceArray("Objekt"));
			Set<String> attributeGroupPids = getPids(datum.getReferenceArray("AttributGruppe"));
			Set<String> aspectPids = getPids(datum.getReferenceArray("Aspekt"));
			int simVar = datum.getUnscaledValue("SimulationsVariante").intValue();
			return new ParameterizedBlockTuple(objectPids, attributeGroupPids, aspectPids, simVar);
		}

		private Set<String> getPids(final Data.ReferenceArray array) {
			return Arrays.stream(array.getSystemObjectArray()).map(it -> it != null ? it.getPidOrId() : "").collect(ImmutableSet.toImmutableSet());
		}

		public long getParameterizedDeleteExtraTimeSeconds(final IdDataIdentification dataIdentification) {
			DataModel dataModel = getArchMgr().getDataModel();
			if (dataModel == null) return -1;
			
			SystemObject object = dataModel.getObject(dataIdentification.getObjectId());
			SystemObject attributeGroup = dataModel.getObject(dataIdentification.getAtgId());
			SystemObject aspect = dataModel.getObject(dataIdentification.getAspectId());
			if(object == null || attributeGroup == null || aspect == null) return -1;

			String objectPid = object.getPidOrId();
			String attributeGroupPid = attributeGroup.getPidOrId();
			String aspectPid = aspect.getPidOrId();
			String areaPid = object.getConfigurationArea().getPidOrId();

			long result = -1;

			for(ParameterizedBlock parameterizedBlock : _parameterizedBlocks) {
				if(parameterizedBlock.matches(areaPid, objectPid, attributeGroupPid, aspectPid, dataIdentification.getSimVariant())) {
					// Nicht return hier, wir wollen den letzten passenden parametrierten Wert zurückgeben, nicht den ersten.
					// (Analog zu SettingsManager)
					result = parameterizedBlock.getDeleteTimeSeconds();
				}
			}

			return result;
		}

		public long getDeleteExtraTimeSecondsForUnresolvable() {
			if(_deleteExtraTimeSecondsForUnresolvable < 1) return -1;
			return _deleteExtraTimeSecondsForUnresolvable;
		}

		private class ParameterizedBlock {
			private final Set<String> _areaPids;
			private final List<ParameterizedBlockTuple> _blocks;
			private final long _timeSeconds;

			public ParameterizedBlock(final Set<String> areaPids, final List<ParameterizedBlockTuple> blocks, final long timeSeconds) {
				_areaPids = areaPids;
				_blocks = blocks;
				_timeSeconds = timeSeconds;
			}

			public boolean matches(final String areaPid, final String objectPid, final String attributeGroupPid, final String aspectPid, final int simVariant) {
				return matchesSetWithWildcard(_areaPids, areaPid) && _blocks.stream().anyMatch(it -> it.matches(objectPid, attributeGroupPid, aspectPid, simVariant));
			}

			public long getDeleteTimeSeconds() {
				return _timeSeconds;
			}

			@Override
			public String toString() {
				return "ParameterizedBlock{" +
						"_areaPids=" + _areaPids +
						", _blocks=" + _blocks +
						", _timeSeconds=" + _timeSeconds +
						'}';
			}
		}

		private boolean matchesSetWithWildcard(final Set<String> set, final String item) {
			return set.isEmpty() || set.contains(item);
		}

		private class ParameterizedBlockTuple {
			private final Set<String> _objectPids;
			private final Set<String> _attributeGroupPids;
			private final Set<String> _aspectPids;
			private final int _simVar;

			public ParameterizedBlockTuple(final Set<String> objectPids, final Set<String> attributeGroupPids, final Set<String> aspectPids, final int simVar) {
				_objectPids = objectPids;
				_attributeGroupPids = attributeGroupPids;
				_aspectPids = aspectPids;
				_simVar = simVar;
			}

			public boolean matches(final String objectPid, final String attributeGroupPid, final String aspectPid, final int simVariant) {
				return matchesSetWithWildcard(_objectPids, objectPid)
						&& matchesSetWithWildcard(_attributeGroupPids, attributeGroupPid)
						&& matchesSetWithWildcard(_aspectPids, aspectPid)
						&& simVariant == _simVar;
			}

			@Override
			public String toString() {
				return "ParameterizedBlockTuple{" +
						"_objectPids=" + _objectPids +
						", _attributeGroupPids=" + _attributeGroupPids +
						", _aspectPids=" + _aspectPids +
						", _simVar=" + _simVar +
						'}';
			}
		}
	}

	private class SimParamTask extends QueueTask<ResultData> {

		protected SimParamTask(final ArchiveManager archiveMgr, final SignalingQueue<ResultData> queue) {
			super(archiveMgr, queue);
		}

		@Override
		protected void work(final ResultData step) throws InterruptedException {
			processSimParams(step);
		}
	}

	@Override
	public void terminateTask() {
		super.terminateTask();
		if(simParamWorker != null) {
			simParamWorker.terminateTask();
		}
	}

	@Override
	public void join() throws InterruptedException {
		super.join();
		if(simParamWorker != null) {
			simParamWorker.join();
		}
	}

	/**
	 * Meldet Daten einer Simulation zum Archivieren (und Quittieren) an. {@code rd} muss vom Typ {@link SimulationResultData} sein, sonst wird keine
	 * Anmeldung vorgenommen.
	 *
	 * @param rd Spezifikation der anzumeldenden Daten. Muss vom Typ {@link SimulationResultData} sein.
	 */
	private void processSimParams(ResultData rd) throws InterruptedException {
		if (rd instanceof SimulationResultData ard) {
			Map<DataIdentification, Data> settingsMap = new HashMap<>();
			SimVarParamExtracter.extractSettings(ard, getArchMgr().getDataModel(), settingsMap, ard.getSimVar());
			int i = 0;
			SubscrMarker sm = new SubscrMarker();	// zum Mitzählen der Quittierungen eigenen Marker anlegen
			for(final Map.Entry<DataIdentification, Data> dataIdentificationDataEntry : settingsMap.entrySet()) {
				i++;
				DataIdentification dataIdentification = dataIdentificationDataEntry.getKey();
				DataDescription dd = dataIdentification.getDataDescription();
				DataIdentNode din;
				din = getDidTree().get(dataIdentification);
				din.arsParamSetParams(dataIdentificationDataEntry.getValue());				// Teil der aktuellen Parametrierung

				// Beim ersten DS nach Anmeldung darf der DIdx ausnahmsweise gleich dem letzten DIdx sein (wenn sich die DID nicht
				// geändert hat). Um dies feststellen zu können, mussen folgende Flags gesetzt werden:
				din.setFirstDataAfterSubscription(ArchiveDataKind.ONLINE, true);
				din.setFirstDataAfterSubscription(ArchiveDataKind.ONLINE_DELAYED, true);

				// erst die Quittung, damit sofort quittiert werden kann
				if(din.arSParamIsQuittieren()) {
					final SystemObject so = dataIdentification.getObject();
					final AttributeGroup atg = dd.getAttributeGroup();
					final Aspect asp = dd.getAspect();
					final short simulationVariant = dd.getSimulationVariant();
					CacheManager.getInstance().setCachingEnabled(so.getId(), atg.getId(), asp.getId(), simulationVariant, false);
					subscrAck(so, atg, asp, simulationVariant, din, sm);
				}

				// Hier sollte es keinen Fehler geben, denn der SettingsManager sollte nur gueltige DIDs liefern:
				getArchMgr().getInQueuesMgr().subscribe(getArchMgr().getDavCon(), dataIdentification.getObject(), dd);
			}
			_debug.info(
					"Anmeldungen fuer Simulationsvariante " + ard.getSimVar() + " abgeschlossen:" + Debug.NEWLINE + Util.leadBlank(i, 6)
					+ " Datenidentifikationen angemeldet davon " + Util.leadBlank(sm.subscrFaults, 6) + " Fehler" + Debug.NEWLINE + Util.leadBlank(i, 6)
					+ " Quittierungen angemeldet davon " + Util.leadBlank(sm.qSubscrFault, 6) + " Fehler"
			);
			ard.doneSubscribing(true);
		}
	}

	private void processParams(final SubscrMarker subscrMarker) throws InterruptedException {
		if(printParams) {
			if(paramDumpStream != null) closeParamDumpStream(paramDumpStream);
		}
		else {
			_debug.info("EndOfSettingsListener: Archivparametrierung für " + totalDIDs + " Datenidentifikationen erhalten. Starte An/Abmeldungen...");
			processSubscriptions(subscrMarker);
			if (!isReady) {
				getArchMgr().archiveIsReady();
				isReady = true;
			}
			subscrMarker.printStatistics();
			subscrMarker.reset();	// Speicher sparen...
			totalDIDs = 0;
		}
	}


	/**
	 * Erzeugt einen Listener der benachrichtigt wird, wenn ein Parametrierungszyklus durchlaufen ist.
	 *
	 * @return Listener
	 */
	public EndOfParamListener createEndOfParamListener() {
		EndOfParamListener eopl = new EndOfParamListener();
		synchronized(endOfParamListeners) {
			endOfParamListeners.add(eopl);
		}
		return eopl;
	}

	/**
	 * Initialisiert den Thread und startet durch {@link #restartSubscribeArchiveParams()} die Übernahme der Archivparametrierung.
	 *
	 * @param iqMgr Empfaenger der Datenanmeldung
	 * @param dTree Baum, in den die Datenidentifikationen zusammen mit der Parametrierung eingetragen werden
	 *
 	 */
	public void startSubscribeArchiveParams(InQueuesMgr iqMgr, DataIdentTree dTree) {
		printParams = false;
		inQueuesMgr = iqMgr;
		if (getDidTree() != dTree) {
			throw new AssertionError();
		}
		start();
		restartSubscribeArchiveParams();
	}

	/**
	 * Startet die Übernahme der Archivparametrierung und meldet die zu archivierenden Datenidentifikation an. Diese Methode wird auch nach einem
	 * Verbindungsabbruch zum DAV aufgerufen.
	 *
 	 */
	public void restartSubscribeArchiveParams() {
		settingsMgr = createSettingsMgr();
		settingsMgr.start();

		DataDescription dataDescription = getArchMgr().getDD(ARS_PARAM_DELETE_ATG_PID, ARS_PARAM_ASP_PID);
		if(dataDescription.getAttributeGroup() == null) {
			_debug.warning("Endgültiges Löschen deaktiviert, da " + ARS_PARAM_DELETE_ATG_PID + " nicht vorhanden ist.");
			return;
		}
		_permanentDeleteReceiver = new PermanentDeleteReceiver(dataDescription);
		getArchMgr().getDavCon().subscribeReceiver(_permanentDeleteReceiver, getArchMgr().getArchiveObject(), dataDescription, ReceiveOptions.normal(), ReceiverRole.receiver());
	}

	/** Beendet den SettingsManager und meldet die Parametrierung des Archivsystems ab. */
	public void unsubscribeArchiveParams() {
		if(settingsMgr != null) {
			settingsMgr.removeEndOfSettingsListener(eofSettingsLstnr);
			settingsMgr.removeUpdateListener(updateSettingsLstnr);
			settingsMgr.stop();
			settingsMgr = null;
		}
		if(_permanentDeleteReceiver != null) {
			getArchMgr().getDavCon().unsubscribeReceiver(_permanentDeleteReceiver, getArchMgr().getArchiveObject(), _permanentDeleteReceiver.getDataDescription() );
		}
	}


	/**
	 * Meldet die zu archivierenden Daten der angegebenen Simulationsvariante ab.
	 *
	 * @param simVar Simulationsvariante. Falls < 0 werden alle zu archivierenden Daten abgemeldet
	 */
	private void unsubscribeArchiveData(int simVar) {
		_debug.info("Start Abmeldung aller zu archivierenden Datenidentifikationen" + (simVar < 0 ? "" : (" der Simulationsvariante " + simVar)));
		int numDID = 0;
		if (getDidTree() != null) {
			for (DataIdentNode din : getDidTree()) {
				IdDataIdentification id = din.getDataIdentification();
				if((simVar < 0 || id.getSimVariant() == simVar) && din.isArSParameterized() && din.arSParamIsArchivieren()) {
					SystemObject so = getArchMgr().getObj(id.getObjectId());
					DataDescription dd = new DataDescription(getArchMgr().getAtg(id.getAtgId()), getArchMgr().getAsp(id.getAspectId()), (short) id.getSimVariant());
					getArchMgr().getDavCon().unsubscribeReceiver(inQueuesMgr.getArchiveDataReceiver(), so, dd);
					_debug.finest("Abmeldung fuer " + so + ", " + dd + " erfolgreich");
					numDID++;
				}
			}
		}
		_debug.info(numDID + " zu archivierende Datenidentifikationen" + (simVar != -1 ? (" fuer SimVar " + simVar) : "") + " abgemeldet");
	}

	/** Meldet alle zu archivierenden Daten ab */
	public void unsubscribeArchiveData() {
		unsubscribeArchiveData(-1);
	}

	/**
	 * Meldet alle zu einer Simulation gehoerenden Daten (= zu archivierende Daten + Quittungen) ab. Da es nicht auf einfache Weise möglich ist, festzustellen,
	 * wann keine Datensätze einer bestimmten SimVar mehr in der Archiv-Queue sind, kann man nicht bis dahin warten, bevor die Quittungen abgemeldet werden. Es
	 * kann also sein, dass einige Datensätze der Simulation nicht mehr quittiert werden.
	 */
	public void unsubscribeSimVarData(int simVar) {
		unsubscribeArchiveData(simVar);
		unsubscribeArchiveDataAck(simVar);

		for (DataIdentNode dataIdentNode : getDidTree()) {
			if(dataIdentNode.getDataIdentification().getSimVariant() == simVar) dataIdentNode.arsParamSetParams(null);
		} 
	}


	/**
	 * Meldet die Quittierungsaspekte der genannte Simulationsvariante ab. Dies muss nach {@link #unsubscribeArchiveData()} geschehen, damit die Datensätze,
	 * die noch in der Queue sind, nach der Archivierung quittiert werden können.
	 */
	private void unsubscribeArchiveDataAck(int simVar) {
		_debug.info("Start Abmeldung aller Quittierungsaspekte" + (simVar < 0 ? "" : (" der Simulationsvariante " + simVar)));
		ClientDavInterface davCon = getArchMgr().getDavCon();
		int numQ = 0;
		for (DataIdentNode din : getDidTree()) {
			IdDataIdentification id = din.getDataIdentification();
			if((simVar < 0 || id.getSimVariant() == simVar) && din.isArSParameterized() && din.arSParamIsQuittierenValid()) {
				SystemObject so = davCon.getDataModel().getObject(id.getObjectId());
				AttributeGroup atg = (AttributeGroup) davCon.getDataModel().getObject(id.getAtgId());
				Aspect asp = (Aspect) davCon.getDataModel().getObject(din.arSParamGetQuittieren());
				CacheManager.getInstance().setCachingEnabled(id.getObjectId(), id.getAtgId(), id.getAspectId(), id.getSimVariant(), true);
				davCon.unsubscribeSender(inQueuesMgr.getDataAckSender(), so, new DataDescription(atg, asp, (short) id.getSimVariant()));
				numQ++;
			}
		}
		_debug.info(numQ + " Quittierungsaspekte" + (simVar != -1 ? (" fuer SimVar " + simVar) : "") + " abgemeldet");
	}

	/**
	 * Meldet alle Quittierungsaspekte ab. Dies muss nach {@link #unsubscribeArchiveData()} geschehen, damit die Datensätze, die noch in der Queue sind, nach
	 * der Archivierung quittiert werden können.
	 */
	public void unsubscribeArchiveDataAck() {
		unsubscribeArchiveDataAck(-1);
	}


	/**
	 * Gibt die Parametrierung auf dem übergebenen Stream aus.
	 *
	 * @param dumpFile Ausgabestream. Falls null, wird auf stdout ausgegeben
	 *
 	 * @throws InterruptedException Thread unterbrochen
	 */
	public void printArchiveParams(File dumpFile) throws TimeoutException, InterruptedException {
		printParams = true;
		paramDumpStream = dumpFile == null ? null : createParamDumpStream(dumpFile);
		noAnswer = false;
		gotFirstPrintParam = false;
		final EndOfParamListener eopl = createEndOfParamListener();

		Thread watchdog = new Thread(() -> {
			try {
				Thread.sleep(PARAM_TIMEOUT);
				if(!gotFirstPrintParam) {
					// Wenn schon angefangen wurde nicht unterbrechen
					noAnswer = true;
					eopl.gotParams();
				}
			}
			catch(InterruptedException ignored) {
			}
		});
		watchdog.start();
		start();
		settingsMgr = createSettingsMgr();
		settingsMgr.start();
		eopl.waitForParams();
		watchdog.interrupt();
		if(noAnswer) throw new TimeoutException("Kein Parameter-Datensatz nach " + (PARAM_TIMEOUT / 1000) + " Sekunden erhalten.");
	}

	@Nullable
	private FileOutputStream createParamDumpStream(File dumpFile) {
		try {
			if(dumpFile.isDirectory()) dumpFile = new File(dumpFile, DEFAULT_PARAMS_FILE);
			if (dumpFile.exists())
				getArchMgr().quitError("Ausgabedatei '" + dumpFile.getAbsolutePath() + "' für die Archivparametrierung existiert bereits.");
			if(!dumpFile.createNewFile()) {
				getArchMgr().quitError("Ausgabedatei '" + dumpFile.getAbsolutePath() + "' für die Archivparametrierung konnte nicht angelegt werden.");
			}
			if (!dumpFile.canWrite())
				getArchMgr().quitError("Ausgabedatei '" + dumpFile.getAbsolutePath() + "' kann nicht geschrieben werden.");
			return new FileOutputStream(dumpFile);
		}
		catch(IOException e) {
			getArchMgr().quitError("Ausgabedatei '" + dumpFile.getAbsolutePath() + "' für die Archivparametrierung konnte nicht angelegt werden: " + e.getMessage());
		}
		return null;
	}

	private static void closeParamDumpStream(FileOutputStream ds) {
		try {
			ds.close();
		}
		catch(IOException e) {
			_debug.error("Fehler beim Schreiben der Ausgabedatei für die Archivparametrierung" + e.getMessage());
		}
	}

	private void unsubscrAck(SystemObject so, AttributeGroup atg, int sv, long qAspId, final SubscrMarker subscrMarker) {
		subscrMarker.qUnsubscrTotal++;
		try {
			Aspect qAsp = getArchMgr().getAsp(qAspId);
			getArchMgr().getDavCon().unsubscribeSender(inQueuesMgr.getDataAckSender(), so, new DataDescription(atg, qAsp, (short)sv));
		}
		catch(IllegalArgumentException e) {
			// das sollte mit den gueltigen/ungueltigen Q-Asp vermieden werden:
			_debug.error("Sendeabmeldung eines illegalen Quittierungsaspektes: " + e.getMessage());
		}
	}

	/**
	 * Meldet die Quittung an. Falls nicht erfolgreich, wird dies im DataIdentNode vermerkt (durch negative ID-1).
	 *
	 * @param so  Systemobjekt
	 * @param atg Attributgruppe
	 * @param sv  SimVar
	 * @param din DataIdentnode
	 */
	private void subscrAck(SystemObject so, AttributeGroup atg, Aspect asp, int sv, DataIdentNode din, SubscrMarker sm) {
		Exception ex;
		long qAspId = din.arSParamGetQuittieren();
		if(sm != null) sm.qSubscrTotal++;
		try {
			if(asp.getId() == qAspId) throw new IllegalStateException("AspektID==QuittungsaspektID");
			Aspect qAsp = getArchMgr().getAsp(qAspId);
			getArchMgr().getDavCon().subscribeSender(
					getArchMgr().getInQueuesMgr().getDataAckSender(), so, new DataDescription(atg, qAsp, (short) sv), SenderRole.source()
			);
			din.arSParamMarkQuittierenInvalid(false);
			return;
		}
		catch(Exception e) {
			ex = e;
		}

		if(sm != null) sm.qSubscrFault++;
		din.arSParamMarkQuittierenInvalid(true);
		_debug.fine("Sendeanmeldung fuer Quittierungsaspekt 'id=" + qAspId + "' fuer " + so + ", " + atg + "' fehlgeschlagen: " + ex.getMessage());
	}

	/**
	 * Diese Methode arbeitet den {@link SubscrMarker} ab, der in den Callback-Methoden des {@link SettingsManager} gefuellt wurde. Es werden gemäß
	 * Parametrierung alle An- und Abmeldungen für die Datensätze als auch für die Quittungen durchgefuehrt. Die {@link DataIdentNode}s wurden bereits mit der
	 * aktuellen Parametrierung besetzt.
	 * @param subscrMarker Anmeldungen
	 */
	private void processSubscriptions(final SubscrMarker subscrMarker) throws InterruptedException {
		ClientDavInterface davCon = getArchMgr().getDavCon();

		long i = 0, t = Util.startTimer();
		for (DataIdentNode din : getDidTree()) {
			if(shouldTerminate()) return;

// Sollen Simulationen hier mit aufgenommen werden?
//			if (it.getSV() != 0)
//				continue;

			IdDataIdentification id = din.getDataIdentification();
			SubscrInfo si = subscrMarker.getSubscrInfo(din);
			if(si != null) {
				// für diesen DataIdentNode hat sich im letzten Parametrierungslauf etwas geändert
				i++;
				SystemObject so = getArchMgr().getObj(id.getObjectId());
				AttributeGroup atg = getArchMgr().getAtg(id.getAtgId());
				Aspect asp = getArchMgr().getAsp(id.getAspectId());
				DataDescription dd = new DataDescription(atg, asp, (short) id.getSimVariant());

				// Objekte erfolgreich erhalten, nun gibt es 6 Fälle, in denen An/Abmeldungen notwendig sein können:

				switch (si.kind) {
					case SubscrInfo.NEW_PARAM_ARCH, SubscrInfo.RE_PARAM_ARCH -> {
						// *** Anmelden wegen Neuparametrierung (mit arch=ja) oder Umparametrierung (mit arch=ja) ***

						// laut TAnfArS 5.1.2.5 müssen bei Änderung des Attributes "archivieren" offene Cont. abgeschlossen werden.
						if (si.kind == SubscrInfo.RE_PARAM_ARCH)
							inQueuesMgr.insertCloseContainer(new CloseContainerObject(new IdDataIdentification(so, dd)));
						try {
							// Beim ersten DS nach Anmeldung darf der DIdx ausnahmsweise gleich dem letzten DIdx sein (wenn sich die DID nicht
							// geändert hat). Um dies feststellen zu können, mussen folgende Flags gesetzt werden:
							din.setFirstDataAfterSubscription(ArchiveDataKind.ONLINE, true);
							din.setFirstDataAfterSubscription(ArchiveDataKind.ONLINE_DELAYED, true);

							boolean subscribed = getArchMgr().getInQueuesMgr().subscribe(davCon, so, dd);
							if (subscribed) {

								// wenn das erfolgreich war, Quittung anmelden:
								if (din.arSParamIsQuittieren()) {
									CacheManager.getInstance().setCachingEnabled(so.getId(), atg.getId(), asp.getId(), id.getSimVariant(), false);
									subscrAck(so, atg, asp, id.getSimVariant(), din, subscrMarker);       // setzt Quittung valid/invalid
								} else {
									din.arSParamMarkQuittierenInvalid(false);
								}

							} else {
								subscrMarker.subscrFaults++;
							}
						} catch (IllegalArgumentException e) {
							subscrMarker.subscrFaults++;
							_debug.error(i + ".: Empfangsanmeldung fuer " + so + ", " + dd + " fehlgeschlagen: " + e.getMessage());
						}
					}
					case SubscrInfo.UN_PARAM, SubscrInfo.RE_PARAM_NOARCH -> {
						// *** Abmelden (erst DID, wenn erfolgreich die Quittung (ist die konsistentere Loesung))
						davCon.unsubscribeReceiver(inQueuesMgr.getArchiveDataReceiver(), so, dd);

						// laut TAnfArS 5.1.2.5 müssen bei Änderung des Attributes "archivieren" oder Abmeldung offene Cont. abgeschlossen werden.
						inQueuesMgr.insertCloseContainer(new CloseContainerObject(new IdDataIdentification(so, dd)));

						// Abmeldung der Quittungsaspekte an dieser Stelle:
						// Dies bedeutet, dass bei DID, die vorher mit Quittierung
						// parametriert waren, und von denen noch welche in der Queue sind, keine Quittungen
						// mehr gesendet werden können. Siehe ArchivTask.sendReceipt()

						if (si.kind == SubscrInfo.UN_PARAM && din.arSParamIsQuittierenValid()) {
							CacheManager.getInstance().setCachingEnabled(so.getId(), atg.getId(), asp.getId(), id.getSimVariant(), true);
							final long qAsp = din.arSParamGetQuittieren();
							unsubscrAck(so, atg, id.getSimVariant(), qAsp, subscrMarker);
						} else if (si.kind == SubscrInfo.RE_PARAM_NOARCH && si.hasOldAspect()) {
							CacheManager.getInstance().setCachingEnabled(so.getId(), atg.getId(), asp.getId(), id.getSimVariant(), true);
							final long qAsp = si.oldQAspectId;
							unsubscrAck(so, atg, id.getSimVariant(), qAsp, subscrMarker);
						}

						// wenn nicht archiviert wird, kann auch nicht quittiert werden
						din.arSParamMarkQuittierenInvalid(true);
						if (si.kind == SubscrInfo.UN_PARAM)
							din.arsParamSetParams(null);    // nicht mehr Teil der aktuellen Parametrierung
					}
					case SubscrInfo.RE_PARAM_PARAM -> {
						// *** Ggf. Abmelden des alten und Anmelden des neuen Q-Aspekts:
						CacheManager.getInstance().setCachingEnabled(so.getId(), atg.getId(), asp.getId(), id.getSimVariant(), true);
						if (si.hasOldAspect()) unsubscrAck(so, atg, id.getSimVariant(), si.oldQAspectId, subscrMarker);
						if (din.arSParamIsArchivieren()) din.arSParamMarkQuittierenInvalid(false);
						if (din.arSParamIsArchivieren() && din.arSParamIsQuittieren()) {
							CacheManager.getInstance().setCachingEnabled(so.getId(), atg.getId(), asp.getId(), id.getSimVariant(), false);
							subscrAck(so, atg, asp, id.getSimVariant(), din, subscrMarker);
						}
					}
					case SubscrInfo.NEW_PARAM_NOARCH ->
						// wenn nicht archiviert wird, kann auch nicht quittiert werden
							din.arSParamMarkQuittierenInvalid(true);
				}
			}
			else	// keine aktuelle Parametrierung für diesen DIN vorhanden
			{
				din.arsParamSetParams(null);	// nach DAV-Absturz notwendig, wenn nicht abgemeldet wurde
			}
		}
		_debug.info(i + " An/Abmeldungen durchgeführt in " + Util.stopTimer(t));
	}

	/**
	 * Vermerkt einen DataIdentNode als reparametriert.
	 *
	 * @param subscrMarker Anmeldungen
	 * @param did  DataIdentNode
	 * @param newD Neue Parameterdaten
	 */
	private void prepareReparam(final SubscrMarker subscrMarker, DataIdentification did, Data newD) {
		DataIdentNode din = getDidTree().get(did);
		boolean archivieren = getBool(newD, K.ARCHIVIEREN);

		if(!din.arSParamIsArchivieren() && archivieren) {
			subscrMarker.reParamArch(din);        // Ab jetzt archivieren (Anmeldung vermerken)
		}
		else if(din.arSParamIsArchivieren() && !archivieren) {
			// Ab jetzt nicht mehr archivieren (Abmeldung u. ggf. alten Q-Aspekt vermerken)
			if(din.arSParamIsQuittierenValid()) {
				subscrMarker.reParamNoArch(din, din.arSParamGetQuittieren());
			}
			else {
				subscrMarker.reParamNoArch(din);
			}
		}
		else {
			// Sonstige Änderung der Parametrierung (z.B. der Quittierung)
			if(din.arSParamIsQuittierenValid() && din.arSParamIsQuittieren()) {
				subscrMarker.reParam(din, din.arSParamGetQuittieren()); // alte Quittierung zum Abmelden vermerken
			} else {
				subscrMarker.reParam(din);
			}
		}
		din.arsParamSetParams(newD);                // Parametrierung überschreiben (deshalb alten Quittungsaspekt merken)
	}

	/**
	 * Vermerkt einen DataIdentNode als neu parametriert. Der Node wird erzeugt, falls er nicht existiert. Dies passiert in der Callback-Methode. Die
	 * DataIdentNodes sind also immer vor {@link #processParams(SubscrMarker)} angelegt. D.h. sobald dort das erste Datum angemeldet wird, kann es archiviert werden.
	 *
	 * @param subscrMarker Anmeldungen
	 * @param did  Datenidentifikation
	 * @param newD Neue Parameterdaten
	 */
	private void prepareNewParam(final SubscrMarker subscrMarker, DataIdentification did, Data newD) {
		DataIdentNode din;
		din = getDidTree().get(did);
		din.arsParamSetParams(newD);				// Parametrierung vermerken
		if(din.arSParamIsArchivieren()) {
			subscrMarker.newParamArch(din);		// Als zu archivieren vermerken
		}
		else {
			subscrMarker.newParamNoArch(din);	// Als nicht zu archiveren zu vermerken
		}
	}

	/**
	 * Vermerkt einen DataIdentNode als nicht mehr parametriert.
	 *
	 * @param subscrMarker Anmeldungen
	 * @param did Datenidentifikation
	 */
	private void prepareUnparam(final SubscrMarker subscrMarker, DataIdentification did) {
		subscrMarker.unParam(getDidTree().get(did));    // Parametrierung wird spaeter auf null gesetzt
	}

	/**
	 * Schreibt die empfangene Parametrierung auf den übergebenen Ausgabestrom oder auf stdout.
	 *
	 * @param dumpStream    Ausgabestrom. Falls null, wird auf stdout geschrieben.
	 * @param separatorLine Zeile, die zwischen Datensätzen eingefügt wird (\n wird angehängt).
	 * @param did           Datenidentifikation
	 * @param data          Parametrierung
	 */
	@SuppressWarnings("StringConcatenationInsideStringBufferAppend")
	private void printParam(@Nullable OutputStream dumpStream, String separatorLine, DataIdentification did, Data data) {
		gotFirstPrintParam = true;
		StringBuilder sb = new StringBuilder();
		DataDescription dd = did.getDataDescription();
		sb.append(
				separatorLine + "\nobj   " + did.getObject().getPid() + " (id=" + did.getObject().getId() + ") '" + did.getObject().getName() + "'\natg   "
						+ dd.getAttributeGroup().getPid() + " (" + dd.getAttributeGroup().getId() + ") '" + dd.getAttributeGroup().getName() + "'\nasp   "
		);
		sb.append(dd.getAspect().getPid() + " (" + dd.getAspect().getId() + ") '" + dd.getAspect().getName() + "'\nsv    " + dd.getSimulationVariant() + "\n");
		if(data != null) {
			sb.append("einst archiv.: " + (getBool(data, K.ARCHIVIEREN) ? "ja" : "nein"));
			sb.append(" / sich.: " + (getBool(data, K.SICHERN) ? "ja" : "nein"));
			sb.append(
					" / quitt.: " + (data.getArray(K.QUITTIEREN).getLength() > 0
							? data.getArray(K.QUITTIEREN).getItem(0).asReferenceValue().getValueText()
							: "---")
			);
			sb.append(" / vorh.: " + data.getTimeValue(K.VORHALTEN).getSeconds());
			int nachf = data.getArray(K.NACHFORDERN).getLength();
			if(nachf == 0) {
				sb.append(" / nachf.: ---");
			} else {
				sb.append("\n      nachf.:\n");
				for (int j = 0; j < nachf; j++) {
					sb.append("      " + data.getArray(K.NACHFORDERN).getItem(j).asTextValue().getValueText());
				}
			}
			sb.append("\n");
		}
		else {
			sb.append("   ## Fehler: Data==null");
		}

		if(dumpStream != null) {
			try {
				dumpStream.write(sb.toString().getBytes(Charset.defaultCharset()));
				dumpStream.flush();
			}
			catch(IOException e) {
				_debug.error("Fehler beim Schreiben der Ausgabedatei für die Archivparametrierung: " + e.getMessage());
			}
		} else {
			System.out.print(sb);
		}
	}

	private static boolean getBool(Data data, String key) {
		return data.getUnscaledValue(key).intValue() > 0;
	}

	/**
	 * Mit dieser Klasse wird während der Parameterübernahme in den Callback-Methoden vermerkt für welche Datenidentifikation (= {@link DataIdentNode}) was
	 * parametriert wurde. Dadurch ist eine Trennung von Parameterübernahme und An-/Abmeldung möglich. Datenidentifikationen können dadurch falls notwendig für
	 * blockweise An-/Abmeldung zusammengefasst werden. Die Parametrierung, die im {@link DataIdentNode} vermerkt ist, ist allein nicht ausreichend, da auch die
	 * Parameteränderung relevant ist. Gespeichert wird dies durch eine Map von {@link DataIdentNode} nach {@link SubscrInfo}. Da die neuen Parameter sofort im
	 * {@link DataIdentNode} gespeichert werden, ist es in einigen Fällen erforderlich, den alten Quittierungsaspket hier mitzuspeichern.
	 * <p>
	 * Unterschieden wird nach:
	 * <p>
	 * -- Neu parametriert, zu archivieren -- Neu parametriert, nicht zu archivieren -- Nicht mehr parametriert -- Umparametriert, zu archivieren --
	 * Umparametriert, nicht zu archivieren (der ehemalige Quittierungsaspekt wird gespeichert, falls vorhanden) -- Umparametriert, Archivierung nicht geändert,
	 * jedoch ggf. Quittierungsaspekt
	 *
	 * @author AlexanderS
	 */
	protected static class SubscrMarker {

		// Platz sparen beim Aufbau der Markierungen, daher DAG-Struktur soweit wie möglich:
		private static final SubscrInfo SI_NEW_PARAM_ARCH = new SubscrInfo(SubscrInfo.NEW_PARAM_ARCH);
		private static final SubscrInfo SI_NEW_PARAM_NOARCH = new SubscrInfo(SubscrInfo.NEW_PARAM_NOARCH);
		private static final SubscrInfo SI_UN_PARAM = new SubscrInfo(
				SubscrInfo.UN_PARAM
		);
		private static final SubscrInfo SI_RE_PARAM_ARCH = new SubscrInfo(SubscrInfo.RE_PARAM_ARCH);
		private static final SubscrInfo SI_RE_PARAM_NOARCH = new SubscrInfo(SubscrInfo.RE_PARAM_NOARCH);
		private static final SubscrInfo SI_RE_PARAM_PARAM = new SubscrInfo(
				SubscrInfo.RE_PARAM_PARAM
		);

		protected static final class SubscrInfo {

			public static final int NEW_PARAM_ARCH = 1;
			public static final int NEW_PARAM_NOARCH = 2;
			public static final int UN_PARAM = 3;
			public static final int RE_PARAM_ARCH = 4;
			public static final int RE_PARAM_NOARCH = 5;
			public static final int RE_PARAM_PARAM = 6;

			/** Was ist geschehen (Neu-/Ent-/Umparametrierung) */
			public final int kind;

			/** In einigen Fällen braucht man den alten Quittungsaspekt */
			public long oldQAspectId = -1;

			public SubscrInfo(int kind) {
				this.kind = kind;
			}

			public SubscrInfo(int kind, long oldQAspectId) {
				this.kind = kind;
				this.oldQAspectId = oldQAspectId;
			}

			public boolean hasOldAspect() {
				return oldQAspectId != -1;
			}

			public SubscrInfo copy() {
				return new SubscrInfo(kind, oldQAspectId);
			}
		}

		private final Map<DataIdentNode, SubscrInfo> marker = new HashMap<>();

		// An- und Abmeldefehler mitzählen
		public int subscrFaults, unsubscrFaults, qSubscrTotal, qUnsubscrTotal, qUnsubscrFault, qSubscrFault;

		/**
		 * Vermerken, dass dieser DataIdentNode neu parametriert wurde und archiviert wird. Anmeldung der DID und ggf. der Quittierung erforderlich.
		 *
		 * @param din Der betreffende Datenknoten
		 */
		public void newParamArch(DataIdentNode din) {
 			marker.put(din, SI_NEW_PARAM_ARCH);
		}

		/**
		 * Vermerken, dass dieser DataIdentNode neu parametriert wurde und nicht archiviert wird. Keine An-/Abmeldungen notwendig, nur zur Statistik vermerken.
		 *
		 * @param din Der betreffende Datenknoten
		 */
		public void newParamNoArch(DataIdentNode din) {
			marker.put(din, SI_NEW_PARAM_NOARCH);
		}

		/**
		 * Vermerken, dass dieser DataIdentNode nicht mehr parametriert ist. Abmeldung der DID und ggf. der alten (noch im din enthaltenen) Quittierung erforderlich.
		 *
		 * @param din Der betreffende Datenknoten
		 */
		public void unParam(DataIdentNode din) {
			marker.put(din, SI_UN_PARAM);
		}

		/**
		 * Vermerken, dass dieser DataIdentNode auf archivieren umparametriert wurde. Anmeldung der DID und ggf. der Quittierung erforderlich.
		 *
		 * @param din Der betreffende Datenknoten
		 */
		public void reParamArch(DataIdentNode din) {
			marker.put(din, SI_RE_PARAM_ARCH);
		}

		/**
		 * Vermerken, dass dieser DataIdentNode auf nicht archivieren umparametriert wurde. Abmeldung der DID und der alten Quittierung erforderlich. Da die
		 * Quittierung im DataIdentNode überschrieben worden sein kann, muss sie mitgespeichert werden.
		 *
		 * @param din Der betreffende Datenknoten
		 */
		public void reParamNoArch(DataIdentNode din, long oldQAspectId) {
			SubscrInfo sc = SI_RE_PARAM_NOARCH.copy();
			sc.oldQAspectId = oldQAspectId;
			marker.put(din, sc);
		}

		/**
		 * Vermerken, dass dieser DataIdentNode auf nicht archivieren umparametriert wurde. Abmeldung der DID erforderlich, keine Quittierung abzumelden.
		 *
		 * @param din Der betreffende Datenknoten
		 */
		public void reParamNoArch(DataIdentNode din) {
			marker.put(din, SI_RE_PARAM_NOARCH);
		}

		/**
		 * Vermerken, dass dieser DataIdentNode umparametriert wurde, was keinen Einfluss auf die Archivierung hat. Abmeldung der Quittierung erforderlich. Wenn sich
		 * bestimmte Werte der Parametrierung geändert haben, mussen aktuelle Container abgeschlossen werden.
		 *
		 * @param din Der betreffende Datenknoten
		 */
		public void reParam(DataIdentNode din, long oldQAspectId) {
			SubscrInfo sc = SI_RE_PARAM_PARAM.copy();
			sc.oldQAspectId = oldQAspectId;
			marker.put(din, sc);
		}

		/**
		 * Vermerken, dass dieser DataIdentNode umparametriert wurde, jedoch nicht in seiner Quittierungseigenschaft. Wenn sich bestimmte Werte der Parametrierung
		 * geändert haben, mussen aktuelle Container abgeschlossen werden.
		 *
		 * @param din Der betreffende Datenknoten
		 */
		public void reParam(DataIdentNode din) {
			marker.put(din, SI_RE_PARAM_PARAM);
		}

		/**
		 * Liefert die vorgemerkte Markierung.
		 *
		 * @param din Der betreffende Datenknoten
		 */
		public SubscrInfo getSubscrInfo(DataIdentNode din) {
			return marker.get(din);
		}

		/** Löschen der gespeicherten Markierungen. */
		public void reset() {
			subscrFaults = 0;
			unsubscrFaults = 0;
			qSubscrTotal = 0;
			qUnsubscrTotal = 0;
			qUnsubscrFault = 0;
			qSubscrFault = 0;
			marker.clear();
		}

		/** Ausgabe der gespeicherten Markierungen. */
		private void printStatistics() {
			String text = toString();
			if(text.equals("[]")) return;
			_debug.info("An/Abmeldungen abgeschlossen:\n  " + text);
		}

		public String toString() {
			int newparam_arch = 0, newparam_noarch = 0, reparam_param = 0, reparam_subscr = 0, reparam_unsubscr = 0, unparam = 0, closeCont = 0;
			if(marker.isEmpty()) return "[]";

			for(SubscrInfo si : marker.values()) {
				if(si.kind == SubscrInfo.NEW_PARAM_ARCH) {
					newparam_arch++;
				}
				else if(si.kind == SubscrInfo.NEW_PARAM_NOARCH) {
					newparam_noarch++;
				}
				else if(si.kind == SubscrInfo.UN_PARAM) {
					unparam++;
					closeCont++;
				}
				else if(si.kind == SubscrInfo.RE_PARAM_ARCH) {
					reparam_subscr++;
					closeCont++;
				}
				else if(si.kind == SubscrInfo.RE_PARAM_NOARCH) {
					reparam_unsubscr++;
					closeCont++;
				}
				else if(si.kind == SubscrInfo.RE_PARAM_PARAM) {
					reparam_param++;
				}
			}

			return Util.leadBlank(newparam_arch + newparam_noarch, 6) + " Zugaenge (" + newparam_arch + " arch / "
					+ newparam_noarch + " noarch), " + unparam + " Abgaenge, " + (reparam_param + reparam_unsubscr + reparam_subscr) + " Aenderungen ("
					+ reparam_subscr + "->arch / " + reparam_unsubscr + "->noarch / " + reparam_param + " andere)" + ", insgesamt " + marker.size() + "\n  "
					+ Util.leadBlank(closeCont, 6) + " Datenidentifikationen abzuschliessen" + "\n  " + Util.leadBlank(subscrFaults, 6)
					+ " Fehler bei Anmeldungen von Datenidentifikationen" + "\n  " + Util.leadBlank(unsubscrFaults, 6)
					+ " Fehler bei Abmeldungen von Datenidentifikationen" + "\n  " + Util.leadBlank(qSubscrTotal, 6) + " Anmeldungen von Quittierungen, davon "
					+ qSubscrFault + " Fehler" + "\n  " + Util.leadBlank(qUnsubscrTotal, 6) + " Abmeldungen von Quittierungen, davon " + qUnsubscrFault
					+ " Fehler";
		}
	}

	/**
	 * Spezielle Klasse, mit der man auf das Ende der Übernahme der nächsten Archiveinstellungen warten kann. Dies ist u.a. für Testzwecke hilfreich. Listener
	 * dieser Art können nur einmal verwendet werden, weil die Liste der Listener in der Klasse {@link ArchivConfig} nach jedem Zyklus geleert wird.
	 */
	public static class EndOfParamListener {

		public boolean gotParams;

		private synchronized void gotParams() {
			gotParams = true;
			notifyAll();
		}

		public synchronized void waitForParams() throws InterruptedException {
			while(!gotParams) wait();
		}
	}
}
