/*
 *
 * Copyright 2010-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt;

import de.bsvrz.ars.ars.mgmt.tasks.base.Task;
import de.bsvrz.ars.ars.mgmt.tasks.base.TaskManager;
import de.bsvrz.ars.ars.mgmt.tasks.base.TaskState;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.ClientSenderInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.OneSubscriptionPerSendData;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.SenderRole;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.ConfigurationObject;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.debug.Debug;

import java.util.*;

/**
 * Verwaltet und dokumentiert die Zustände von Tasks und deren Fortschritt. Bei Zustandswechseln einzelner Tasks und in zyklischen Abständen werden die Zustände
 * aller Tasks ausgegeben und über den Datenverteiler publiziert. Der Fortschritt von Aktionen einzelner Tasks wird zyklisch ausgegeben.
 *
 * @author Kappich Systemberatung
 * @version $Revision: none $
 */
public class ArchiveLongTermTaskStatePublisher implements LongTermTaskStatePublisher {

	/** Intervalldauer zur Ausgabe und Publikation der Zustände aller Tasks */
	private static final long ALL_TASK_UPDATE_INTERVAL = 60000;

	/** Intervalldauer zur Ausgabe des Zustands einer Task */
	private static final long TASK_UPDATE_INTERVAL = 10000;

	private final List<ArchiveLongTermTask> _tasks = new ArrayList<>();

	private ClientDavInterface _davConnection;

	private long _nextAllTaskUpdateTime;

	private boolean _enableDataSource;

	private final Debug _debug = Debug.getLogger();

	private static final String NEWLINE = System.getProperty("line.separator");

	private final ArchiveManager _archMgr;

	private final String _minorChangeHeaderText;

	private final String _majorChangeHeaderText;

	private AttributeGroup _atg;

	private ConfigurationObject _archiveObject;

	private DataDescription _dataDescription;

	/**
	 * Erzeugt eine neue Instanz
	 * @param archMgr Archivmanager
	 * @param minorChangeHeaderText Text für die Kopfzeile bei Zustandsänderungen
	 * @param majorChangeHeaderText Text für die Kopfzeile bei Änderungen an den Tasks
	 */
	public ArchiveLongTermTaskStatePublisher(final ArchiveManager archMgr, final String minorChangeHeaderText, final String majorChangeHeaderText) {
		_archMgr = archMgr;
		_minorChangeHeaderText = minorChangeHeaderText;
		_majorChangeHeaderText = majorChangeHeaderText;
		_nextAllTaskUpdateTime = System.currentTimeMillis();
	}

	@Override
	public void setDavConnection(final ClientDavInterface davConnection) {
		_davConnection = davConnection;
		try {
			final DataModel dataModel = davConnection.getDataModel();
			_atg = dataModel.getAttributeGroup("atg.hintergrundTaskZustände");
			final Aspect asp = dataModel.getAspect("asp.archivSystem");
			if(_atg == null || asp == null)
			{
				_debug.info("Kann Taskzustände nicht veröffentlichen, benötigte Attributgruppe und/oder Aspekt sind nicht im Datenmodell vorhanden. Es wird mindestens Version 23 von kb.systemModellGlobal.config benötigt.");
				return;
			}
			_archiveObject = _archMgr.getArchiveObject();
			_dataDescription = new DataDescription(_atg, asp);
			davConnection.subscribeSender(new ClientSender(), _archiveObject, _dataDescription, SenderRole.source());
		}
		catch(OneSubscriptionPerSendData oneSubscriptionPerSendData) {
			_debug.warning("Es gibt bereits ein Objekt, dass Taskzustände veröffentlicht", oneSubscriptionPerSendData);
		}
		catch(Exception e){
			_debug.warning("Fehler beim Anmelden der Veröffentlichung der Taskzustände", e);
		}
	}

	private Data createDataObject() {
		final Data data = _davConnection.createData(_atg);
		final Data.Array array = data.getItem("HintergrundTaskZustände").asArray();
		int i = 0;
		Set<Task> allTasks = new LinkedHashSet<>(_tasks);
		allTasks.addAll(TaskManager.getActiveTasks());
		array.setLength(allTasks.size());
		for (Task task : allTasks) {
			insertData(task, array.getItem(i++));
		}
		return data;
	}

	private void insertData(final Task task, final Data item) {
		item.getItem("TaskName").asTextValue().setText(task.getName());
		item.getItem("TaskZustand").asTextValue().setText(task.getState().getStateName());
		item.getItem("TaskFortschrittsBeschreibung").asTextValue().setText(task.getProgressText());
		item.getItem("TaskFortschritt").asUnscaledValue().set(task.getProgress());
		item.getItem("TaskFortschrittMaximum").asUnscaledValue().set(task.getMaximumProgress());
		item.getItem("TaskFortschrittErgebnis").asUnscaledValue().set(task.getProgressResult());
	}


	@Override
	public Task registerTask(final String taskName) {
		synchronized(this) {
			ArchiveLongTermTask longTermTask = new ArchiveLongTermTask(taskName);
			_tasks.add(longTermTask);
			return longTermTask;
		}
	}

	private class ClientSender implements ClientSenderInterface{
		@Override
		public void dataRequest(final SystemObject object, final DataDescription dataDescription, final byte state) {
			synchronized(ArchiveLongTermTaskStatePublisher.this) {
				if(object.equals(_archiveObject) && dataDescription.equals(_dataDescription)) {
					_enableDataSource = state == START_SENDING;
					if(_enableDataSource) {
						try {
							final Data data = createDataObject();
							_davConnection.sendData(new ResultData(_archiveObject, _dataDescription, System.currentTimeMillis(), data));
							_debug.info("Taskzustände werden über Datenverteiler veröffentlicht");
						}
						catch(Exception e) {
							_debug.warning("Fehler beim Senden der Taskzustände", e);
						}
					}
					else {
						_debug.info("Veröffentlichung der Taskzustände gestoppt");
					}
				}
			}
		}

		@Override
		public boolean isRequestSupported(final SystemObject object, final DataDescription dataDescription) {
			return true;
		}
	}

	private class ArchiveLongTermTask implements Task {

		private volatile long _nextTaskUpdateTime;

		private final String _taskName;

		private volatile TaskState _taskState = TaskState.INIT;

		private volatile boolean _minorChanges;

		private volatile String _format = "";

		private volatile long _progress = -1;

		private volatile long _maximumProgress = -1;

		private volatile long _progressResult = -1;

		public ArchiveLongTermTask(final String taskName) {
			_taskName = taskName;
			_nextTaskUpdateTime = System.currentTimeMillis() + TASK_UPDATE_INTERVAL;
		}

		@Override
		public void setTaskState(final TaskState taskState) {
			synchronized(ArchiveLongTermTaskStatePublisher.this) {
				if (_taskState != taskState) {
					final long now = System.currentTimeMillis();
					if (_minorChanges) publishMinorChange(now);
					_taskState = taskState;
					_format = "";
					_progress = -1;
					_maximumProgress = -1;
					_progressResult = -1;
					_minorChanges = false;
					publishMajorChanges(now);
				}
			}
		}

		@Override
		public void setTaskProgress(final String format, final long progress, final long maximumProgress, final long progressResult) {
			// Die Werte werden ohne Prüfung auf Änderung übernommen, weil sie ohnehin nur in vorgegebenen Zeitabständen ausgegeben werden
			final long now = System.currentTimeMillis();
			boolean publish = false;
			_format = format;
			_progress = progress;
			_maximumProgress = maximumProgress;
			_progressResult = progressResult;
			_minorChanges = true;
			if (now > _nextTaskUpdateTime) {
				publish = true;
			}
			if(publish) {
				synchronized(ArchiveLongTermTaskStatePublisher.this) {
					notifyMinorChanges(now);
				}
				synchronized(this) {
					if(now > _nextTaskUpdateTime) {
						publishMinorChange(now);
					}
				}
			}
		}

		private void publishMinorChange(final long now) {
			_debug.info(this.toString());
			publishedMinorChanges(now);
		}

		@Override
		public String toString() {
			final StringBuilder stringBuilder = new StringBuilder();
			stringBuilder.append(_minorChangeHeaderText);
			stringBuilder.append(NEWLINE);
			appendTo(stringBuilder);
			return stringBuilder.toString();
		}

		private void appendTo(final StringBuilder stringBuilder) {
			stringBuilder.append(_taskName);
			stringBuilder.append(": ");
			stringBuilder.append(_taskState.getStateName());
			if(!_format.isEmpty()) {
				stringBuilder.append(": ");
				try(Formatter formatter = new Formatter(stringBuilder)) {
					formatter.format(_format, _progress, _maximumProgress, _progressResult);
				}
			}
		}

		@Override
		public synchronized String getProgressText() {
			return String.format(_format, _progress, _maximumProgress, _progressResult);
		}

		public void publishedMinorChanges(final long now) {
			_nextTaskUpdateTime = now + TASK_UPDATE_INTERVAL;
			_minorChanges = false;
		}

		@Override
		public String getName() {
			return _taskName;
		}

		@Override
		public TaskState getState() {
			return _taskState;
		}

		@Override
		public long getProgress() {
			return _progress;
		}

		@Override
		public long getMaximumProgress() {
			return _maximumProgress;
		}

		public synchronized long getProgressResult() {
			return _progressResult;
		}
	}

	private void publishMajorChanges(final long now) {
		final StringBuilder stringBuilder = new StringBuilder();
		stringBuilder.append(_majorChangeHeaderText);
		stringBuilder.append(NEWLINE);
		for(ArchiveLongTermTask longTermTask : _tasks) {
			longTermTask.appendTo(stringBuilder);
			stringBuilder.append(NEWLINE);
			longTermTask.publishedMinorChanges(now);
		}
		_debug.info(stringBuilder.toString());

		if(_enableDataSource)
		{
			try {
				final Data data = createDataObject();
				_davConnection.sendData(new ResultData(_archiveObject, _dataDescription, System.currentTimeMillis(), data));
			}
			catch(Exception e){
				_debug.warning("Fehler beim Senden der Taskzustände", e);
			}
		}

		_nextAllTaskUpdateTime = now + ALL_TASK_UPDATE_INTERVAL;
	}

	private void notifyMinorChanges(final long now) {
		if(now > _nextAllTaskUpdateTime) {
			publishMajorChanges(now);
		}
	}
}
