/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt;

import de.bsvrz.ars.ars.mgmt.commands.ArSCmdInterface;
import de.bsvrz.ars.ars.mgmt.simulation.SimulationManager;
import de.bsvrz.ars.ars.mgmt.tasks.Task;
import de.bsvrz.ars.ars.mgmt.tasks.scheduler.TaskScheduler;
import de.bsvrz.ars.ars.persistence.*;
import de.bsvrz.ars.ars.persistence.directories.ActivePersistenceDirectory;
import de.bsvrz.ars.ars.persistence.directories.mgmt.lock.LockFileManager;
import de.bsvrz.ars.ars.persistence.directories.mgmt.range.WeekDomain;
import de.bsvrz.ars.ars.repair.Defect;
import de.bsvrz.ars.ars.repair.PersistenceCheckRunner;
import de.bsvrz.ars.ars.repair.RescueResult;
import de.bsvrz.ars.ars.repair.ScanMode;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.config.ConfigurationObject;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.commandLineArgs.ArgumentList;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.bsvrz.sys.funclib.losb.DAVAppBase;
import de.bsvrz.sys.funclib.losb.kernsoftware.ConnectionManager;

import java.io.File;
import java.io.IOException;
import java.lang.ref.WeakReference;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Stream;

/**
 * Zentrale Archiv-Verwaltung.
 *
 * @author beck et al. projects GmbH
 * @author Thomas Schaefer
 * @author Alexander Schmidt
 * @version $Revision$ / $Date$ / ($Author$)
 */
public class ArchiveManager extends DAVAppBase implements TaskManager {

	private static final int DEFAULT_CMD_PORT = 4242;

	private static final String P_PRINT_PARAMS = "-nurParametrierungAusgeben";

	private static final String P_PRINT_PARAMS_FILE = "-parametrierungAusgabeDatei";

	private static final String P_PERS_PATH = "-persistenzVerzeichnis";

	private static final String P_CHECK_PERS_DIR = "-persistenzVerzeichnisPruefen";

	private static final String P_REMOTE_CONTROL = "-kommandoPort";

	private static final String P_MAX_QUERIES_PER_APPLICATION = "-maxAnfragenProApplikation";

	private static final int DEFAULT_MAX_QUERIES_PER_APPLICATION = 5;

	private static final String P_REMOTE_CONTROL_NORC = "deaktiviert";

	private static final String P_REMOTE_CONTROL_DEFAULT = "default";

	private PersistenceManager persistenceMgr;

	private InQueuesMgr inQueuesMgr;

	private ArchivConfig archivConfig;

	private TaskScheduler taskScheduler;

	private SimulationManager simulationMgr;

	// Aufrufparameter:
	private boolean printParamsAndExit;

	private File printParamsFile;

	private ScanModes checkPersDirMode;

	private Path archivePath;

	private int maximumQueriesPerApplication;

	private int cmdPortNum = -1;

	private boolean terminated;

	private boolean runningProperly;

	private final Object _terminatedLock = new Object();

	/**
	 * Enthält die Anzahl aktiver Archivanfragen je Applikation. Es sind nur für die Applikationen Einträge in der Map enthalten für die mindestens eine aktive Archivanfrage in der Verarbeitung ist.
	 */
	private final TreeMap<SystemObject, Integer> _applicationToArchiveQueryCount;

	/**
	 * Thread für Shutdown-Hook
	 */
	private Thread _shutDownHookThread;

	/**
	 * Anzahl Archivtasks
	 */
	private int _numOfArchTasks = -1;

	/**
	 * Anzahl Threads fürs Index-Schließen beim Beenden
	 */
	private int _numCloseIndexThreads = -1;

	/**
	 * Anzahl Threads für Persistenz-Prüfung und Wiederherstellung
	 */
	private int _numCheckPersistenceThreads = -1;

	/**
	 * Anzahl Threads fürs Löschen
	 */
	private int _numDeleteThreads = -1;

	/**
	 * Minimum Nanosekunden-Wartezeit zwischen Anmeldungen
	 */
	private long _minWaitNanosPerSubscription;

	/**
	 * Maximum Nanosekunden-Wartezeit zwischen Anmeldungen
	 */
	private long _maxWaitNanosPerSubscription;

	/**
	 * Anzahl gleichzeitiger Anmeldungen
	 */
	private int _slidingWindowSize;

	/**
	 * Gesamtkapazität Datensätze in Online-Queues
	 */
	private int _totalCapacityOfOnlineQueues = -1;

	/**
	 * System.exit() überspringen (für Unit-Tests)
	 */
	private boolean _skipSystemExit;

	/**
	 * Maximale Anzahl Bytes pro Index als Cache
	 */
	private int _indexCacheMaxSize;

	/**
	 * Pid des Objekts für das Archivsystem (vom Typ "typ.archiv", in der Regel die lokale AOE)
	 */
	private String _archiveObjectPid;

	/**
	 * Wiederherstellungsmodus
	 */
	private RebuildMode rebuildMode;

	/**
	 * Endgültiges Löschen nur simulieren?
	 */
	private final AtomicBoolean simulatePermanentDelete = new AtomicBoolean(true);

	/**
	 * Haupt-Konstruktor des Archivsystems
	 * @param args Kommandozeilen-Argumente
	 * @throws Exception Fehler bei Initialisierung
	 */
	public ArchiveManager(String[] args) throws Exception {
		// Initialer DisconnectHandler wird später in der run()-Methode ersetzt
		super(args, "Archivsystem", () -> {
			Debug.getLogger().error("Verbindung zum Datenverteiler wurde frühzeitig terminiert, Archivsystem wird beendet");
			System.exit(2);
		});
		try {
			// Abhängigkeiten prüfen, damit der Anwendet sofort merkt, dass etwas falsch läuft, wenn er eine zu alte Kernsoftware verwendet.
			Class.forName("de.bsvrz.dav.daf.util.FileAccess");
		} catch (ClassNotFoundException ignored) {
			Debug.getLogger().error("Das ArchivSystem benötigt die Kernsoftware in Mindestversion 3.9.6");
			System.exit(3);
		}
		Thread.setDefaultUncaughtExceptionHandler(new ExceptionHandler());
		_applicationToArchiveQueryCount = new TreeMap<>();
		logger = Debug.getLogger();
		try {
			parseArguments(argList);
		} catch (Exception e) {
			logger.error("Fehler beim Auswerten der Aufrufargumente: " + e.getMessage());
			e.printStackTrace(System.err);
			if (!_skipSystemExit) {
				System.exit(4);
			}
			throw e;
		}
		try {
			if (!printParamsAndExit) {
				persistenceMgr = new PersistenceManager(this, archivePath, detectDomain(archivePath, false));
				persistenceMgr.initialize();
			}
			archivConfig = new ArchivConfig(this);
		} catch (Exception e) {
			// Exception auf Debug ausgeben
			logger.error("Fehler bei Initialisierung des Archivsystems", e);
			// Exception weitergeben, z. B. für Tests
			throw e;
		}
	}

	/**
	 * Ermittelt, nach welcher Struktur bzw. welcher Zeit-Intervall-Aufteilung das persistenzverzeichnis aufgebaut ist.
	 *
	 * @param archivePath Wurzelverzeichnis des Archivsystems
	 * @param silent      Sollen hilfreiche Debug-Ausgaben gemacht werden?
	 * @return entweder ein {@link WeekDomain}-Objekt oder null, falls das Persistenzverzeichnis das alte Format hat
	 * @throws IOException Fehler bei Analyse
	 */
	@Nullable
	public static WeekDomain detectDomain(Path archivePath, boolean silent) throws IOException {
		Debug debug = Debug.getLogger();
		WeekDomain weekDomain = new WeekDomain();
		if (!Files.isDirectory(archivePath)) {
			if (!silent) debug.warning("Initialisiere neues Wurzelverzeichnis: " + archivePath.toAbsolutePath());
			Files.createDirectories(archivePath);
			return weekDomain;
		}
		try (Stream<Path> stream = Files.list(archivePath)) {
			List<Path> dirs = stream.filter(Files::isDirectory).filter(it -> !isBackupDir(it)).toList();
			if (dirs.isEmpty()) {
				if (!silent) debug.warning("Initialisiere leeres Wurzelverzeichnis: " + archivePath.toAbsolutePath());
				return weekDomain;
			}
			List<Path> validWeekDirs = dirs.stream().filter(WeekDomain::isYearDir).toList();
			if (!validWeekDirs.isEmpty()) {
				if (!silent)
					debug.info("Initialisiere bestehendes Wurzelverzeichnis mit " + validWeekDirs.size() + " Jahresverzeichnisse: " + archivePath.toAbsolutePath());
				return weekDomain;
			}
			if (dirs.stream().anyMatch(it -> it.getFileName().toString().startsWith("obj"))) {
				// Altes Layout

				// Verzeichnis zwangsweise als beschreibbar markieren.
				Path isOpenFile = archivePath.resolve(LockFileManager.NOT_PROTECTED_FLAG_FILE_NAME);
				if (!Files.exists(isOpenFile)) {
					Files.createFile(isOpenFile);
				}
				if (!silent) warnAboutMigration(archivePath, debug);
				return null;
			}
			if (!silent) debug.warning("Unbekannte Dateien im Wurzelverzeichnis: " + dirs);
			return weekDomain;
		}
	}

	private static boolean isBackupDir(Path it) {
		return it.getFileName().toString().equals(PersistenceManager.BACKUP_DIR_NAME);
	}

	private static void warnAboutMigration(Path archivePath, Debug debug) {
		debug.warning("""
				--------------------------------------------------------------------------------------------------------
				Das übergebene Persistenzverzeichnis wurde noch nicht in das Format von Version 6 migriert.

				Das Archivsystem Version 6 bietet eine kompakte, schnellere Verzeichnis-Struktur und verbesserte
				Wartbarkeit, jedoch werden die Funktionen Auslagern, Wiederherstellen und Löschen nicht mehr
				unterstützt.

				Sollten Sie diese Funktionen weiterhin benötigen, wechseln Sie bitte zurück auf Version 5.x.

				Ansonsten sollten Sie die Archivdaten mit z. B. folgendem Befehl in das neue Format konvertieren:

				    java -jar '%s' -src='%s' -dst='%s' -debugLevelStdErrText=INFO
								
				und zukünftig das Archivsystem mit -archiv='%3$s' starten.
								
				Die Konvertierung kann einige Zeit in Anspruch nehmen. Während der Konvertierung können alle
				ausgelagerten Daten wieder integriert werden. Dies ist nachträglich nicht mehr möglich.
								
				Die Konvertierung darf nicht durchgeführt werden, während das Archivsystem läuft,
				da dadurch Inkonsistenzen entstehen könnten. Bei Bedarf kann (z. B. mit der konsistenten Datensicherung)
				eine Kopie des Quellverzeichnisses erstellt werden.
				--------------------------------------------------------------------------------------------------------
				""".formatted(swePath(), archivePath.toAbsolutePath(), suggestNewArchivePath(archivePath))

		);
	}

	private static String swePath() {
		URL location = ArchiveManager.class.getProtectionDomain().getCodeSource().getLocation();
		if (location == null) {
			return "de.bsvrz.ars.migration/de.bsvrz.ars.migration-runtime.jar";
		}
		try {
			String string = Paths.get(location.toURI()).toString();
			if (string.endsWith("-runtime.jar")) {
				return string.replace("de.bsvrz.ars.ars", "de.bsvrz.ars.migration");
			}
		} catch (URISyntaxException ignored) {
		}
		return "de.bsvrz.ars.migration/de.bsvrz.ars.migration-runtime.jar";
	}

	private static Object suggestNewArchivePath(Path archivePath) {
		return archivePath.resolveSibling("archiv_neu");
	}

	/**
	 * Gibt das Archiv-Objekt vom Typ "typ.archiv" zurück
	 *
	 * @return das Archiv-Objekt
	 */
	public final ConfigurationObject getArchiveObject() {
		ConfigurationObject object = null;
		if (_archiveObjectPid.isEmpty()) {
			object = getConfigAuth();
		} else {
			DataModel dataModel = getDataModel();
			if (dataModel != null) {
				SystemObject systemObject = dataModel.getObject(_archiveObjectPid);
				if (systemObject instanceof ConfigurationObject) {
					object = (ConfigurationObject) systemObject;
				} else {
					throw new IllegalArgumentException("Das Archiv-Objekt vom typ.archiv konnte nicht bestimmt werden. Aufrufargument -archiv=" + _archiveObjectPid + " prüfen.");
				}
			}
		}
		if (object == null) {
			throw new IllegalArgumentException("Das Archiv-Objekt vom typ.archiv konnte nicht bestimmt werden. Eventuell besteht noch keine Datenverteiler-Verbindung.");
		}
		return object;
	}

	@Nullable
	@Override
	public DataModel getDataModel() {
		ClientDavInterface connection = getDavCon();
		if (connection == null) return null;
		return connection.getDataModel();
	}

	/**
	 * Gibt {@code true} zurück, wenn das Archivsystem sich gerade beendet.
	 * @return {@code true}, wenn das Archivsystem sich gerade beendet, sonst {@code false}
	 */
	@Override
	public boolean wasTerminated() {
		synchronized (_terminatedLock) {
			return terminated;
		}
	}

	private void parseArguments(ArgumentList argLst) {

		// Ganz am Anfang prüfen, damit bei Tests bei weiteren Parse-Fehlern die
		// JVM nicht beendet wird
		_skipSystemExit = argLst.fetchArgument("-noExit=false").booleanValue();
		
		// Kommandozeilenparameter:
		CacheManager.getInstance().init(argList);
		printParamsAndExit = argLst.hasArgument(P_PRINT_PARAMS);
		printParamsFile = argLst.hasArgument(P_PRINT_PARAMS_FILE) ? argLst.fetchArgument(P_PRINT_PARAMS_FILE).asFile() : null;
		checkPersDirMode = argLst.hasArgument(P_CHECK_PERS_DIR) ? argLst.fetchArgument(P_CHECK_PERS_DIR).asEnum(ScanModes.class) : null;
		archivePath = argLst.hasArgument(P_PERS_PATH) ? Paths.get(argLst.fetchArgument(P_PERS_PATH).asString()) : null;
		String cmdPort = argLst.hasArgument(P_REMOTE_CONTROL) ? argLst.fetchArgument(P_REMOTE_CONTROL).asString() : null;
		maximumQueriesPerApplication = argLst.hasArgument(P_MAX_QUERIES_PER_APPLICATION) ? argLst.fetchArgument(P_MAX_QUERIES_PER_APPLICATION).intValueBetween(1, Integer.MAX_VALUE) : DEFAULT_MAX_QUERIES_PER_APPLICATION;

		// Archiv-Objekt setzen
		_archiveObjectPid = argLst.fetchArgument("-archiv=").asString();

		rebuildMode = argLst.fetchArgument("-wiederherstellungsModus=Quick").asEnum(RebuildMode.class);

		_numOfArchTasks = argLst.fetchArgument("-archivierungsThreads=11").intValueBetween(1, Integer.MAX_VALUE);

		simulatePermanentDelete.set(!argLst.fetchArgument("-permanentLoeschen=nein").booleanValue());

		// Parameter für Archivqueues
		_totalCapacityOfOnlineQueues = argLst.fetchArgument("-archivierungsPuffer=200000").intValueBetween(250, Integer.MAX_VALUE);

		// Parameter für Thread-Anzahl
		_numCloseIndexThreads = argLst.fetchArgument("-indexUpdateThreads=" + getNumOfArchTasks()).intValueBetween(1, Integer.MAX_VALUE);
		_numCheckPersistenceThreads = argLst.fetchArgument("-pruefThreads=" + getNumOfArchTasks()).intValueBetween(1, Integer.MAX_VALUE);
		_numDeleteThreads = argLst.fetchArgument("-lzvThreads=1").intValueBetween(1, Integer.MAX_VALUE);

		// Parameter für Anmeldeverzögerung
		double minPerSecond = argLst.fetchArgument("-minAnmeldungenProSekunde=1").doubleValueBetween(1, Double.POSITIVE_INFINITY);
		double maxPerSecond = argLst.fetchArgument("-maxAnmeldungenProSekunde=Infinity").doubleValueBetween(1, Double.POSITIVE_INFINITY);
		_slidingWindowSize = argLst.fetchArgument("-maxAnmeldungenGleichzeitig=1000").intValueBetween(1, 1_000_000);

		// Begrenzung der offenen Containerdateien
		ContainerDataIterator.setOpenFileLimit(argLst.fetchArgument("-maxOffeneDateien=500").intValueBetween(1, Integer.MAX_VALUE));

		PersistenceManager.setDeleteBrokenContainers(argLst.fetchArgument("-defekteDateienVerschieben=false").booleanValue());

		_indexCacheMaxSize = argLst.fetchArgument("-maxIndexBytes=16384").intValueBetween(1, Integer.MAX_VALUE);

		// Max/Min-Vertauschung absichtlich wegen Kehrwert.
		_minWaitNanosPerSubscription = (long) (1_000_000_000.0 / maxPerSecond);
		_maxWaitNanosPerSubscription = (long) (1_000_000_000.0 / minPerSecond);

		// Konsistenzprüfung Kommandozeilenparameter:
		if (checkPersDirMode != null && archivePath == null)
			quitError("Ungültige Parameterkombination: " + P_CHECK_PERS_DIR + " erfordert " + P_PERS_PATH);

		argLst.ensureAllArgumentsUsed();

		if (checkPersDirMode != null && printParamsAndExit) {
			quitError("Ungültige Parameterkombination: " + P_CHECK_PERS_DIR + " und " + P_PRINT_PARAMS + " nicht gleichzeitig möglich");
		}

		if (!printParamsAndExit && printParamsFile != null) {
			quitError("Ungültige Parameterkombination: " + P_PRINT_PARAMS_FILE + " nur in Verbindung mit " + P_PRINT_PARAMS + " möglich");
		}

		if ((!printParamsAndExit || checkPersDirMode != null) && archivePath == null) {
			quitError("Ungültige Parameterkombination: " + P_PERS_PATH + " nicht vorhanden");
		}

		if (cmdPort != null && !cmdPort.equalsIgnoreCase(P_REMOTE_CONTROL_NORC)) {
			if (cmdPort.equals(P_REMOTE_CONTROL_DEFAULT)) {
				cmdPortNum = DEFAULT_CMD_PORT;
			} else {
				try {
					cmdPortNum = Integer.parseInt(cmdPort);
				} catch (NumberFormatException e) {
					quitError("Ungueltiger Wert fuer " + P_REMOTE_CONTROL + ": '" + cmdPort + "'");
				}
			}
		}
	}

	/**
	 * Startet das Archivsystem
	 *
	 * @param runtimeControl Laufzeit-Umgebung, zur Abfrage von Systemzeit, Spezialeinstellungen für Tests usw.,
	 *                       im produktiven Betrieb ein {@link SystemRuntimeControl}.
	 */
	public void run(RuntimeControl runtimeControl) {
		if (wasTerminated()) {
			logger.error("Das Archivsystem wurde bereits beendet.");
			throw new IllegalStateException("Das Archivsystem wurde bereits beendet.");
		}
		if (executeCheckMode()) {
			quit();
		} else {
			// startet den DAV
			if (executeSpecialModeWithDAV()) {
				quit();
			} else {
				// ##########################################################################################
				// hier gehts richtig los:
				logger.info("Archivsystem im Normalmodus gestartet");

				// Persistenzverzeichnis überprüfen (evtl anderes ArS angemeldet?)
				setDisconnectHandler(() -> quitError("Verbindung zum Datenverteiler wurde terminiert, Archivsystem wird kontrolliert beendet"));
				_shutDownHookThread = new Thread(new ShutdownHook(this));
				java.lang.Runtime.getRuntime().addShutdownHook(_shutDownHookThread);
				// Keine Sperrdatei vorhanden, eine angelegt, wir können starten
				runningProperly = true;

				QueueParameters parameters = new QueueParameters(
						_totalCapacityOfOnlineQueues,
						_minWaitNanosPerSubscription,
						_maxWaitNanosPerSubscription,
						_slidingWindowSize
				);

				inQueuesMgr = new InQueuesMgr(this, runtimeControl, persistenceMgr.getDataIdentTree(), parameters);
				taskScheduler = new TaskScheduler(this);
				simulationMgr = new SimulationManager(this);

				// Als Erstes das Kommando-Interface starten
				if (cmdPortNum != -1) new ArSCmdInterface(this, cmdPortNum).start();

				// Loesch-/Nachfordern-/Sichern-Uhren starten:
				if (!wasTerminated()) taskScheduler.start();

				// PersistenceManager ist der einzige der einen Startup-Check braucht,
				// alle anderen Module haben keine persistenten Daten.
				if (!persistenceMgr.startupProcedure(getRebuildMode())) {
					quitError("Fehler im Persistenzverzeichnis. Archivsystem kann nicht gestartet werden.");
				} else {
					enableExplicitApplicationReadyMessage();
					connectToDavQuit();    // Bis hierher sollte alles ohne DAV gehen

					if (!wasTerminated())
						inQueuesMgr.getObjectsFromDav();    // jeder Task darf benoetigte Objekte holen
					if (!wasTerminated())
						inQueuesMgr.startAllTasks();        // startet alle Tasks (mit Ausnahme von ArchivConfig)
					if (!wasTerminated()) inQueuesMgr.subscribeSettings();    // auf alle Parameter anmelden

					// mit der Anmeldung der zu archivierenden Daten gehts los...
					if (!wasTerminated())
						archivConfig.startSubscribeArchiveParams(inQueuesMgr, persistenceMgr.getDataIdentTree());

					if (!wasTerminated()) simulationMgr.start();                //Simulationsverwaltung starten

					if (!wasTerminated())
						new FreeDiskSpacePublisher(this).start();                //Simulationsverwaltung starten
				}
			}
		}
	}

	@Override
	public RuntimeControl getRuntimeControl() {
		if (inQueuesMgr != null) {
			return inQueuesMgr.getRuntimeControl();
		}
		return null;
	}

	@Override
	public long countDataInQueues() {   
		if (inQueuesMgr != null) {
			return inQueuesMgr.countDataInQueues();
		}
		return 0;
	}

	@Override
	public long estimateQueueMemoryUsage() {
		if (inQueuesMgr != null) {
			return inQueuesMgr.estimateQueueMemoryUsage();
		}
		return 0;
	}

	@Override
	public void suspendTaskIfNecessary(Task task) throws InterruptedException {
		if (inQueuesMgr != null) {
			inQueuesMgr.suspendTaskIfNecessary(task);
		}
	}

	/**
	 * Verbindet sich zum datenverteiler, wenn dabei ein Fehler auftritt, wird das System beendet.
	 */
	public void connectToDavQuit() {
		try {
			connectToDav();
		} catch (Exception e) {
			quitError("Es konnte keine Verbindung zum Datenverteiler aufgebaut werden.\n" + e.getMessage());
		}
	}

	/**
	 * Führt ggf. die Prüfung des persistenzverzeichnisses durch
	 *
	 * @return true: Prüfung wurde angefordert und durchgeführt, false: Keine Prüfung angefordert
	 */
	private boolean executeCheckMode() {
		if (checkPersDirMode != null) {
			logger.info("Archivsystem zur Prüfung des Persistenzverzeichnisses gestartet");
			checkPersistenceDir(checkPersDirMode.get());
			return true;
		}
		return false;
	}

	/**
	 * Prüft das Persistenzverzeichnis auf fremde Dateien und Verzeichnisse und falsche Reihenfolgen von ID-Verzeichnissen. Die Ergebnisse werden über den Logger
	 * protokolliert. Kann durch einen Kommandozeilenparameter beim Systemstart aufgerufen werden.
	 *
	 * @param mode Scan-Modus
	 */
	public void checkPersistenceDir(ScanMode mode) {
		ActivePersistenceDirectory activePersistenceDirectory = persistenceMgr.getActivePersistenceDirectory(0);

		if (activePersistenceDirectory == null) {
			return;
		}

		try {
			RescueResult result = PersistenceCheckRunner.scan(activePersistenceDirectory, mode);

			printDetails("Gelöschte Containerdateien, denen das letzte Byte fehlt", result.getOneByteShortDefects());
			printDetails("Gelöschte Containerdateien, bei denen der maximale Datenindex falsch ist", result.getBadRangeEndDefects());
			printDetails("Containerdateien mit abgeschnittenen Daten", result.getBadContainerFileEnd());
			printDetails("Containerdateien ohne Daten", result.getShortContainerFiles());
			printDetails("Defekte Indexdateien", result.getBadIndexFiles());
			printDetails("Sonstige Probleme", result.getCantFix());

			logger.info(
					"# geprüfte Container:        " + result.getCheckedContainerFileCount() + Debug.NEWLINE +
							"# Container ohne Defekte:    " + result.getValidContainerFileCount() + Debug.NEWLINE +
							"# Container mit Defekten:    " + result.getDefectContainerFileCount() + Debug.NEWLINE +
							"# geprüfte Indexdateien:     " + result.getCheckedIndexFileCount() + Debug.NEWLINE +
							"# Indexdateien ohne Defekte: " + result.getValidIndexFileCount() + Debug.NEWLINE +
							"# Indexdateien mit Defekten: " + result.getDefectIndexFileCount() + Debug.NEWLINE +
							"# korrigierbare Defekte:     " + result.getFixableDefectsCount() + Debug.NEWLINE +
							"# " + result.getDataIdentificationsVisited() + " Datenidentifikationen untersucht." + Debug.NEWLINE +
							"# Ergebnis: " + (result.isConsistent() ? "Das Archiv ist in einem konsistenten Zustand." : "Es sind Fehler aufgetreten.")
			);

			if (mode.fixErrors()) {
				PersistenceCheckRunner.fixDefects(result, activePersistenceDirectory.getLayoutInstance());
			}

		} catch (PersistenceException e) {
			logger.error("Konsistenzprüfung abgebrochen", e);
		}
	}

	private void printDetails(String title, Set<Defect> defects) {
		if (defects.isEmpty()) {
			return;
		}
		logger.info(title + ":"
				+ Debug.NEWLINE + "# Anzahl: " + defects.size()
				+ Debug.NEWLINE + "# Details nur in Debug-Level FINE"
		);
		for (Defect defect : defects) {
			logger.fine(defect.getFile() + Debug.NEWLINE + defect.getMessage(), defect.getException());
		}
	}

	private boolean executeSpecialModeWithDAV() {
		if (printParamsAndExit) {
			connectToDavQuit();
			logger.info(
					"Archivsystem zur Parameterausgabe " + (printParamsFile == null ? "auf der Konsole" : ("in '" + printParamsFile.getAbsolutePath() + "'"))
							+ " gestartet"
			);
			try {
				archivConfig.printArchiveParams(printParamsFile);
			} catch (Exception e) {
				logger.error("Parameterausgabe fehlgeschlagen: " + e.getMessage());
			}
			return true;
		}
		return false;
	}

	/**
	 * Beendet alle Tasks, meldet App vom DAV ab.
	 */
	private void cleanUp() {
		/*
		Der folgende synchronized-Block sorgt dafür, dass alle Threads die (u.U. gleichzeitig/kurz nacheinander) cleanUp() aufrufen,
		immer so lange warten, bis das cleanUp() (also das saubere Herunterfahren des Archivsystems) auch wirklich beendet wurde.

		Das heißt der erste Thread führt den CleanUp-Code aus, alle weiteren Threads warten dann so lange bis der erste Thread fertig ist.

		Das ist wichtig für den folgenden Fall:
		  - Jemand beendet das Archivsystem über die Telnet-Schnittstelle oder die Datenverteiler-Verbindung bricht ab
			- cleanUp() wird ausgeführt
		  - Zusätzlich terminiert jemand den Archivsystem-Prozess, indem ein SIGTERM, Ctrl+C o. ä. gesendet wird
			- jetzt führt der ShutdownHook ebenfalls cleanUp() aus
			  - wenn der Shutdown-Hook-Thread vorzeitig beendet wird, bevor das cleanUp() des ursprünglichen Threads abgeschlossen ist,
			   sorgt das dafür, dass sich die Applikation sofort beendet, da die JVM den Prozess nach Bearbeitung aller
			   ShutdownHooks sofort tötet.
			   --> Es ist wichtig, dass der Shutdown-Hook-Thread im cleanUp() solange wartet bis das cleanUp() im ursprünglichen
			   Thread fertig ist.
		*/
		synchronized (_terminatedLock) {
			final boolean wasTerminated;
			wasTerminated = terminated;
			terminated = true;
			if (wasTerminated) return;
			try {
				// Alle periodischen Tasks stoppen:
				logger.info("Beende Task Scheduler...");
				if (taskScheduler != null) taskScheduler.terminate();
				logger.info("Task Scheduler beendet");

				// Archivdaten abmelden
				logger.info("Melde Archivdaten ab...");
				if (archivConfig != null) archivConfig.unsubscribeArchiveData();
				logger.info("Archivdaten abgemeldet");

				// Archivparametrierung abmelden
				logger.info("Melde Archivparametrierung ab...");
				if (archivConfig != null) archivConfig.unsubscribeArchiveParams();
				logger.info("Archivparametrierung abgemeldet");

				//Simulationsverwaltung herunterfahren
				logger.info("Melde Simulation ab...");
				if (simulationMgr != null) simulationMgr.terminateTask();
				logger.info("Simulation abgemeldet");

				// Beenden aller (u.a. der ArchiveTasks) Tasks
				logger.info("Beende Tasks...");
				if (inQueuesMgr != null) inQueuesMgr.stopAllTasks();
				logger.info("Tasks beendet");

				if (persistenceMgr != null) {
					persistenceMgr.saveUnsubscriptionTime();
				}

				// Alle anderen abonnierten Objekte abmelden (nach dem Beenden der ArchivQueryTasks, weil nach dem
				// abgebrochenem take() das Totenglöckchen über den DAV gesendet werden muss):
				logger.info("Melde sonstige Parameter ab...");
				if (getDavCon() != null) ConnectionManager.unsubscribeAll(getDavCon());
				logger.info("Sonstige Parameter abgemeldet");

				// Quittungen erst nach Leeren der Archivwarteschlange abmelden
				logger.info("Melde Quittungen ab...");
				if (archivConfig != null) archivConfig.unsubscribeArchiveDataAck();
				logger.info("Quittungen abgemeldet");

				// Warten bis SettingsManager fertig
				logger.info("Warte auf SettingsManager...");
				if (archivConfig != null) {
					archivConfig.terminateTask();
					archivConfig.join();
				}
				logger.info("SettingsManager beendet");

				logger.info("Starte Finalisierung...");
				if (persistenceMgr != null && runningProperly) {
					persistenceMgr.prepareShutdown();
				}
				logger.info("Finalisierung beendet (Indexe geschrieben, Lock-Datei gelöscht)");
			} catch (Exception e) {
				logger.warning("Fehler beim Herunterfahren", e);
			}
			disconnectFromDav();

			persistenceMgr = null;
			inQueuesMgr = null;
			archivConfig = null;
			taskScheduler = null;
			simulationMgr = null;
		}
	}

	/**
	 * Fährt das ArS herunter und beendet die VM.
	 */
	public void quit() {
		logger.info("Archivsystem wird beendet.");
		cleanUp();    // setzt terminated=true
		logger.info("Bye.");
		if (!_skipSystemExit) {
			System.exit(0);
		}
	}

	/**
	 * Fährt das ArS herunter, ohne die VM zu beenden (dies ist bei Fernsteuerung des ArS aus JUnit-Tests notwendig).
	 */
	public void quitNoExit() {
		// Für den Fall, dass ein Test das ArS mehrfach beendet...
		if (!wasTerminated()) {
			logger.info("Archivsystem wird beendet - Java-VM laeuft weiter");
			cleanUp();    // setzt terminated=true
			logger.info("Bye.");
		}
	}

	/**
	 * Fährt das ArS mit einer Fehlermeldung herunter und beendet die VM.
	 * @param msg Fehlermeldung
	 */
	public void quitError(String msg) {
		logger.error(msg);
		logger.info("Archivsystem wird wegen eines Fehlers beendet.");
//		dumpThreads();
		if (inQueuesMgr != null) inQueuesMgr.setFastExit(true);
		cleanUp();
		logger.info("Bye (with error). " + msg);
		if (!_skipSystemExit) {
			System.exit(1);
		} else {
			throw new RuntimeException(msg);
		}
	}

	private void dumpThreads() {
		StringBuilder out = new StringBuilder();
		out.append("Full thread dump\n");
		Map<Thread, StackTraceElement[]> allStackTraces = Thread.getAllStackTraces();
		for (Map.Entry<Thread, StackTraceElement[]> entry : allStackTraces.entrySet()) {
			out.append("\n");
			Thread thread = entry.getKey();
			out.append(String.format("\"%s@%d\"%s prio=%d tid=0x%x%n", thread.getName(), thread.getId(), thread.isDaemon() ? " daemon" : "", thread.getPriority(), thread.getId()));
			StackTraceElement[] trace = entry.getValue();
			for (StackTraceElement traceElement : trace) {
				out.append("\t  at ").append(traceElement).append("\n");
			}
		}
		logger.info(out.toString());
	}

	/**
	 * Fährt das ArS mit einer Fehlermeldung herunter.
	 * @param msg Fehlermeldung
	 */
	public void quitErrorNoExit(String msg) {
		if (inQueuesMgr != null) inQueuesMgr.setFastExit(true);
		if (!wasTerminated()) {
			logger.error(msg);
			logger.info("Archivsystem wird wegen eines Fehlers beendet.");
			dumpThreads();
			cleanUp();
			logger.info("Bye (with error).");
		}
	}

	/**
	 * Startpunkt des Archivsystems.
	 *
	 * @param args Kommandozeilenparameter
	 */
	public static void main(String[] args) {
		try {
			ArchiveManager am = new ArchiveManager(args);
			am.run(new SystemRuntimeControl());
		} catch (Exception ex) {
			ex.printStackTrace(System.err);
			System.exit(4);
		}
	}

	public ArchivConfig getArchivConfig() {
		return archivConfig;
	}

	public InQueuesMgr getInQueuesMgr() {
		return inQueuesMgr;
	}

	public TaskScheduler getTaskScheduler() {
		return taskScheduler;
	}
	
	/**
	 * @return Liefert den SimulationsManager.
	 */
	public SimulationManager getSimulationMgr() {
		return simulationMgr;
	}

	public int getMaximumQueriesPerApplication() {
		return maximumQueriesPerApplication;
	}

	/**
	 * @return Liefert die original Command Line Argumente als unmodifiableList von String Objekten.
	 */
	public List<String> getOriginalCommandLineArguments() {
		List<String> result = new ArrayList<>();
		Collections.addAll(result, argList.getInitialArgumentStrings());
		return Collections.unmodifiableList(result);
	}

	/**
	 * Erhöht für eine Applikation die Anzahl der gleichzeitigen Anfragen um 1.
	 * @param application Applikation
	 * @return Neue Anzahl der gleichzeitigen Anfragen
	 */
	public int increaseArchiveQueryCountForApplication(SystemObject application) {
		synchronized (_applicationToArchiveQueryCount) {
			final Integer queryCount = _applicationToArchiveQueryCount.get(application);
			final int newCount;
			if (queryCount == null) {
				newCount = 1;
			} else {
				newCount = queryCount + 1;
			}
			_applicationToArchiveQueryCount.put(application, newCount);
			//System.out.println("increaseArchiveQueryCountForApplication: newCount = " + newCount);
			return newCount;
		}
	}

	/**
	 * Erniedrigt für eine Applikation die Anzahl der gleichzeitigen Anfragen um 1.
	 *
	 * @param application Applikation
	 */
	public void decreaseArchiveQueryCountForApplication(SystemObject application) {
		synchronized (_applicationToArchiveQueryCount) {
			final Integer queryCount = _applicationToArchiveQueryCount.get(application);
			final int newCount;
			if (queryCount == null) {
				throw new IllegalStateException("Anzahl Archivanfragen für Applikation " + application + " darf nicht kleiner als 0 werden");
			} else {
				newCount = queryCount - 1;
			}
			if (newCount == 0) {
				_applicationToArchiveQueryCount.remove(application);
			} else {
				_applicationToArchiveQueryCount.put(application, newCount);
			}
			//System.out.println("decreaseArchiveQueryCountForApplication: newCount = " + newCount);
		}
	}

	/**
	 * Gibt die aktuelle Anzahl der gleichzeitigen Anfragen für eine Applikation zurück
	 * @param application Applikation
	 * @return Anzahl
	 */
	public int getArchiveQueryCountForApplication(SystemObject application) {
		synchronized (_applicationToArchiveQueryCount) {
			final Integer queryCount = _applicationToArchiveQueryCount.get(application);
			return queryCount == null ? 0 : queryCount;
		}
	}

	/**
	 * Anzahl Threads fürs Index-Schließen beim Beenden
	 *
	 * @return Vom Benutzer vorgegebener Wert
	 */
	public int getNumCloseIndexThreads() {
		return _numCloseIndexThreads;
	}

	/**
	 * Anzahl Threads für Persistenz-Prüfung und Wiederherstellung
	 *
	 * @return Vom Benutzer vorgegebener Wert
	 */
	public int getNumCheckPersistenceThreads() {
		return _numCheckPersistenceThreads;
	}

	@Override
	public PersistenceManager getPersistenceManager() {
		return persistenceMgr;
	}

	/**
	 * Anzahl Archivierungstasks
	 * @return Vom Benutzer vorgegebener Wert
	 */
	public int getNumOfArchTasks() {
		return _numOfArchTasks;
	}

	/**
	 * Anzahl Threads fürs Löschen
	 * @return Vom Benutzer vorgegebener Wert
	 */
	public int getNumDeleteThreads() {
		return _numDeleteThreads;
	}

	/**
	 * Wird von ArchiveConfig ausgeführt, wenn alle Daten initial angemeldet wurden.
	 */
	public void archiveIsReady() {
		logger.info("Das Archivsystem ist bereit für Anfragen.");

		// Archivanfragen erlauben
		getInQueuesMgr().subscribeQueries();

		// Fertigmeldung versenden
		getDavCon().sendApplicationReadyMessage();
	}

	public int getIndexCacheMaxSize() {
		return _indexCacheMaxSize;
	}

	public RebuildMode getRebuildMode() {
		return rebuildMode;
	}

	/**
	 * Gibt {@code true} zurück, wenn das permanente Löschen nru simuliert wird.
	 *
	 * @return {@code true}, wenn das permanente Löschen nru simuliert wird, sonst {@code false}
	 */
	public boolean simulatePermanentDelete() {
		return simulatePermanentDelete.get();
	}

	/**
	 * Ändert die eigentschaft, ob das permanente Löschen aktiv ist oder nur simuliert wird.
	 */
	public void togglePermanentDelete() {
		simulatePermanentDelete.set(!simulatePermanentDelete.get());
	}

	private static class ShutdownHook implements Runnable {
		private final WeakReference<ArchiveManager> _archiveManager;

		public ShutdownHook(final ArchiveManager archiveManager) {
			this._archiveManager = new WeakReference<>(archiveManager);
		}

		@Override
		public void run() {
			ArchiveManager archiveManager = _archiveManager.get();
			if (archiveManager != null) {
				archiveManager.quitErrorNoExit("Terminierungssignal empfangen, Archivsystem wird kontrolliert beendet");
			}
		}
	}

	/**
	 * Implementierung eines UncaughtExceptionHandlers, der bei nicht abgefangenen Exceptions und Errors entsprechende Ausgaben macht und im Falle eines Errors den
	 * Prozess terminiert.
	 */
	@SuppressWarnings("UseOfSystemOutOrSystemErr")
	private class ExceptionHandler implements Thread.UncaughtExceptionHandler {

		/**
		 * Speicherreserve, die freigegeben wird, wenn ein Error auftritt, damit die Ausgaben nach einem OutOfMemoryError funktionieren
		 */
		private volatile byte[] _reserve = new byte[20000];

		@Override
		public void uncaughtException(Thread t, Throwable e) {
			if (e instanceof Error && _reserve != null) { // 2. Bedingung, damit nicht doppelt exit(1) aufgerufen wird
				// Speicherreserve freigeben, damit die Ausgaben nach einem OutOfMemoryError funktionieren
				_reserve = null;
				try {
					System.err.println("Schwerwiegender Laufzeitfehler: Ein Thread hat sich wegen eines Errors beendet, Prozess wird terminiert");
					System.err.println(t);
					e.printStackTrace(System.err);
					logger.error("Schwerwiegender Laufzeitfehler: " + t + " hat sich wegen eines Errors beendet, Prozess wird terminiert", e);
				} catch (Throwable ignored) {
					// Weitere Fehler während der Ausgaben werden ignoriert, damit folgendes exit() auf jeden Fall ausgeführt wird.
				}
				try {
					if (_shutDownHookThread != null) {
						// Shutdownhook entfernen damit Archiv sich sofort beendet, siehe https://www.nerz-ev.de/nerz-fehlerverwaltung/view.php?id=101#c293
						java.lang.Runtime.getRuntime().removeShutdownHook(_shutDownHookThread);
					}
				} catch (Throwable ignored) {
					// Fehler beim Shutdownhook entfernen ignorieren, damit folgendes exit() auf jeden Fall ausgeführt wird.
				}
				if (!_skipSystemExit) {
					System.exit(5);
				}
			} else {
				System.err.println("Laufzeitfehler: Ein Thread hat sich wegen einer Exception beendet:");
				System.err.println(t);
				e.printStackTrace(System.err);
				logger.error("Laufzeitfehler: " + t + " hat sich wegen einer Exception beendet", e);
			}
		}
	}
}
