/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.mgmt;

import de.bsvrz.ars.ars.mgmt.tasks.*;

import java.util.Collection;
import java.util.List;

/**
 * Verwaltung von Hintergrund-Tasks, aus {@link InQueuesMgr} ausgelagert.
 */
public class BackgroundTaskManager {
	final LongTermTaskStatePublisher _longTermTaskStatePublisher;
	private final ArchiveManager archMgr;

	final DeletePermanentlyTask deletePermanentlyTask;
	final ArchiveSettingsTask archiveSettingsTask;
	final DeleteSimVarTask deleteSimVarTask;
	final RequestGapTask requestGapTask;
	final NumQueriesInfoTask numQueriesInfoTask;
	final MirrorTask mirrorTask;

	/**
	 * Erstellt einen neuen BackgroundTaskManager.
	 *
	 * @param archMgr Referenz auf den {@link ArchiveManager}
	 */
	public BackgroundTaskManager(ArchiveManager archMgr) {
		_longTermTaskStatePublisher = new ArchiveLongTermTaskStatePublisher(archMgr, "Zustandsänderung eines Hintergrundprozesses:", "Zustände der Hintergrundprozesse:");
		this.archMgr = archMgr;
		deletePermanentlyTask = new DeletePermanentlyTask(archMgr, _longTermTaskStatePublisher, archMgr.getNumDeleteThreads());
		this.archiveSettingsTask = new ArchiveSettingsTask(archMgr);
		this.deleteSimVarTask = new DeleteSimVarTask(archMgr);
		this.requestGapTask = new RequestGapTask(archMgr);
		this.numQueriesInfoTask = new NumQueriesInfoTask(archMgr);
		this.mirrorTask = new MirrorTask(archMgr, _longTermTaskStatePublisher);
	}

	/**
	 * Alle Hintergrund-Tasks starten
	 */
	void startBackgroundTasks() {
		_longTermTaskStatePublisher.setDavConnection(archMgr.getDavCon());
		for (Task task : getBackgroundTasks()) {
			task.start();
		}
	}

	/**
	 * Alle Hintergrund-Tasks stoppen
	 */
	void stopBackgroundTasks() {
		for (final Task task : getBackgroundTasks()) {
			task.terminateTask();
		}
	}

	/**
	 * Gibt alle Hintergrund-Tasks zurück.
	 *
	 * @return alle Hintergrund-Tasks
	 */
	Collection<? extends Task> getBackgroundTasks() {
		return List.of(
				deletePermanentlyTask,
				archiveSettingsTask,
				deleteSimVarTask,
				requestGapTask,
				numQueriesInfoTask,
				mirrorTask
		);
	}

	/**
	 * Liefert den Task, der für die Bearbeitung der Archiveinstellungen verantwortlich ist. Dies ist notwendig, damit der {@link ArchiveManager} beim Start einen
	 * {@link ArchiveSettingsTask.ArSSettingListener} einhängen und darauf warten kann, dass die Bearbeitung der Archiveinstellungen
	 * abgeschlossen ist. Das Archivsystem wartet beim Start auf die Archiv-Einstellungen, da es nicht sinnvoll ist, die ersten Sekunden mit Default-Werten
	 * loszulaufen.
	 *
	 * @return Den Task, der für die Bearbeitung der Archiveinstellungen verantwortlich ist.
	 */
	public ArchiveSettingsTask getArchiveSettingsTask() {
		return archiveSettingsTask;
	}

	/**
	 * Gibt den Task für das endgültige Löschen zurück.
	 *
	 * @return den Task für das endgültige Löschen
	 */
	public DeletePermanentlyTask getDeletePermanentlyTask() {
		return deletePermanentlyTask;
	}

	/**
	 * Gibt den Nachforderungs-Tasks zurück, der Datenlücken von anderen Archivsystemen nachfordert.
	 *
	 * @return den Nachforderungs-Tasks
	 */
	public RequestGapTask getRequestGapTask() {
		return requestGapTask;
	}

	/**
	 * Gibt den Task zurück, der für das Löschen von Simulationsdaten verantwortlich ist.
	 *
	 * @return DeleteSimVarTask
	 */
	public DeleteSimVarTask getDeleteSimVarTask() {
		return deleteSimVarTask;
	}

	/**
	 * Gibt den Task zurück, mit dem die Limits für gleichzeitige Archivanfragen abgefragt werden können
	 *
	 * @return NumQueriesInfoTask
	 */
	public NumQueriesInfoTask getNumQueriesInfoTask() {
		return numQueriesInfoTask;
	}

	/**
	 * Gibt den Task für das datenkonsistente Backup zurück.
	 *
	 * @return den Task für das datenkonsistente Backup
	 */
	public MirrorTask getMirrorTask() {
		return mirrorTask;
	}
}
