/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt;

import de.bsvrz.ars.ars.persistence.directories.ActivePersistenceDirectory;
import de.bsvrz.dav.daf.main.*;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.util.cron.CronDefinition;
import de.bsvrz.dav.daf.util.cron.CronScheduler;
import de.bsvrz.sys.funclib.debug.Debug;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;

/**
 * Klasse die den freien Speicherplatz im Archivsystem veröffentlicht. Wird beim spontanen Löschen geupdated.
 *
 * @author Kappich Systemberatung
 */
public class FreeDiskSpacePublisher implements ClientSenderInterface{

	private static final Debug _debug = Debug.getLogger();

	private final ArchiveManager _archMgr;
	private DataDescription _dataDescription;
	private final CronScheduler cronScheduler = new CronScheduler();

	public FreeDiskSpacePublisher(ArchiveManager archMgr) {
		_archMgr = archMgr;
		ClientDavInterface clientDavInterface = _archMgr.getDavCon();
		DataModel dataModel = clientDavInterface.getDataModel();
		final AttributeGroup atg = dataModel.getAttributeGroup("atg.archivSpeicherplatz");
		final Aspect asp = dataModel.getAspect("asp.archivSystem");
		if(atg == null || asp == null)
		{
			_debug.info("Kann freien Speicherplatz nicht veröffentlichen, benötigte Attributgruppe und/oder Aspekt sind nicht im Datenmodell vorhanden. Es wird mindestens Version 37 von kb.systemModellGlobal.config benötigt.");
			return;
		}
		_dataDescription = new DataDescription(atg, asp);
		try {
			long freeMemory = -1;
			try {
				freeMemory = Files.getFileStore(_archMgr.getPersistenceManager().getRootPath()).getUsableSpace();
			}
			catch(Exception e) {
				_debug.warning("Fehler beim Ermitteln des freien Speicherplatzes", e);
			}
			clientDavInterface.subscribeSource(this, createResult(freeMemory));
		}
		catch(Exception e) {
			_debug.info("Kann freien Speicherplatz nicht veröffentlichen", e);
		}
	}

	private ResultData createResult(final long freeMemory) {
		Data data;
		if(freeMemory < 0) {
			data = null;
		}
		else {
			_debug.info(String.format("Freier Speicherplatz: %6.3f MB", freeMemory / 1048576.0));
			data = _archMgr.getDavCon().createData(_dataDescription.getAttributeGroup());
			data.getUnscaledValue("FreierSpeicherplatz").set(freeMemory);
		}
		return new ResultData(_archMgr.getArchiveObject(), _dataDescription, System.currentTimeMillis(), data);
	}

	/**
	 * Aktualisiert den freien Speicherplatz
	 * @param freeDiskSpace freier Speicherplatz in Bytes
	 */
	void sendFreeSpace(final long freeDiskSpace) {
		if(_dataDescription == null) return;
		try {
			_archMgr.getDavCon().sendData(createResult(freeDiskSpace));
		}
		catch(Exception e) {
			_debug.info("Kann freien Speicherplatz nicht veröffentlichen", e);
		}
	}

	@Override
	public void dataRequest(final SystemObject object, final DataDescription dataDescription, final byte state) {
	}

	@Override
	public boolean isRequestSupported(final SystemObject object, final DataDescription dataDescription) {
		return false;
	}

	public void start() {
		cronScheduler.schedule(() -> {
			ActivePersistenceDirectory activePersistenceDirectory = _archMgr.getPersistenceManager().getActivePersistenceDirectory(0);
			Path basePath;
			if (activePersistenceDirectory != null) {
				basePath = activePersistenceDirectory.getBasePath();
			} else {
				basePath = _archMgr.getPersistenceManager().getRootPath();
			}
			try {
				long freeSpace = Files.getFileStore(basePath).getUsableSpace();
				sendFreeSpace(freeSpace);
			} catch (IOException e) {
				_debug.info("Kann freien Speicherplatz nicht ermitteln", e);
				sendFreeSpace(-1);
			}
		}, CronDefinition.EVERY_MINUTE);
	}
}
