/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt;

import de.bsvrz.ars.ars.mgmt.datatree.DataIdentNode;
import de.bsvrz.ars.ars.mgmt.datatree.DataIdentTree;
import de.bsvrz.ars.ars.mgmt.simulation.SimulationResultData;
import de.bsvrz.ars.ars.mgmt.tasks.*;
import de.bsvrz.ars.ars.mgmt.tasks.base.TaskManager;
import de.bsvrz.ars.ars.mgmt.tasks.query.QueryHandler;
import de.bsvrz.ars.ars.persistence.CacheManager;
import de.bsvrz.ars.ars.persistence.IdDataIdentification;
import de.bsvrz.ars.ars.persistence.PersistenceManager;
import de.bsvrz.ars.ars.persistence.util.SignalingQueue;
import de.bsvrz.ars.ars.persistence.writer.*;
import de.bsvrz.dav.daf.main.*;
import de.bsvrz.dav.daf.main.archive.ArchiveData;
import de.bsvrz.dav.daf.main.archive.ArchiveQueryPriority;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.util.cron.CronDefinition;
import de.bsvrz.dav.daf.util.cron.CronScheduler;
import de.bsvrz.sys.funclib.dataSerializer.Deserializer;
import de.bsvrz.sys.funclib.dataSerializer.NoSuchVersionException;
import de.bsvrz.sys.funclib.dataSerializer.SerializingFactory;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.losb.datk.MessageType;
import de.bsvrz.sys.funclib.losb.datk.PidQuery;
import de.bsvrz.sys.funclib.losb.util.Util;
import de.bsvrz.sys.funclib.operatingMessage.MessageGrade;
import de.bsvrz.sys.funclib.operatingMessage.MessageSender;
import de.bsvrz.sys.funclib.operatingMessage.MessageState;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.time.Duration;
import java.util.*;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.LockSupport;

/**
 * Warteschlangen-Manager. Verteilt reinkommende Datentelegramm mit Datensätzen/Archivanfragen/Archiv-Informationsanfragen/
 * Backup-/Restore-/HeaderRestore-/LZV/Delete-/RequestGap-Aufträgen auf die jeweilige Warteschlange.
 *
 * @author beck et al. projects GmbH
 * @author Alexander Schmidt
 * @version $Revision$ / $Date$ / ($Author$)
 */
public final class InQueuesMgr {

	// Task Manager
	private static final int TM_QRYA_IDX = 0; /* Archivanfragen Prio A */

	private static final int TM_QRYB_IDX = 1; /* Archivanfragen Prio B */

	private static final int TM_QRYC_IDX = 2; /* Archivanfragen Prio C */

	private static final int TM_ARCINF_IDX = 3; /* Archiv-Informationsanfragen */

	// Anzahl der Query-Tasks pro Taskmanager:

	private static final int TM_QRY_A_NUM_DEFAULT = 3;  // Defaultwert gemäß Datenkatalog

	private static final int TM_QRY_B_NUM_DEFAULT = 6;  // Defaultwert gemäß Datenkatalog

	private static final int TM_QRY_C_NUM_DEFAULT = 9;  // Defaultwert gemäß Datenkatalog

	private static final int TM_ARC_INF_NUM = 4;

	private static final int NUM_OF_MULTITASK_MGR = 4;

	private final int queueCapacity;

	/**
	 * Ab dieser Auslastung werden Non-Write-Tasks angehalten
	 */
	private static final double NONWRITE_SUSPEND_LIMIT = 0.8;

	/**
	 * Ab dieser Auslastung werden Non-Write-Tasks wieder fortgesetzt (mit Hysterese)
	 */
	private static final double NONWRITE_RESUME_LIMIT = 0.7;

	private final SignalingQueue<ResultData> simParamQueue = new SignalingQueue<>();
	private final SignalingQueue<ResultData> queryQueueA = new SignalingQueue<>();
	private final SignalingQueue<ResultData> queryQueueB = new SignalingQueue<>();
	private final SignalingQueue<ResultData> queryQueueC = new SignalingQueue<>();
	private final SignalingQueue<ResultData> queryQueueInfo = new SignalingQueue<>();
	private final BackgroundTaskManager backgroundTaskManager;

	private SignalingQueue<ArchiveJob> archiveQueue;

	private final SerializeTask[] serializeTasks;

	private final SignalingQueue<AsyncSerializableDataset> serializeQueue = new SignalingQueue<>();

	private ArchiveTask archiveTask;

	// Alle direkt referenzierten Tasks (ohne die über MultiTaskManager referenzierten Tasks):


	private final MultiTaskManager[] tskMgrs = new MultiTaskManager[NUM_OF_MULTITASK_MGR];

	private final FlowControlTask flowCtrl;

	private final ArchiveManager archMgr;

	private final RuntimeControl runtimeControl;

	private final DataIdentTree didTree;

	private static final Debug logger = Debug.getLogger();

	private boolean terminated;

	// hier die Empfaenger-/Senderklassen
	private final ArchiveDataReceiver archiveDataReceiver = new ArchiveDataReceiver();

	private final QueryReceiver queryReceiver = new QueryReceiver();

	private final ArchiveSettingsReceiver arsSettingsReceiver = new ArchiveSettingsReceiver();

	private final DataAckSender dataAckSender = new DataAckSender();

	private final AtomicLong receivedCounterOnline = new AtomicLong(0);

	private final AtomicLong queueCounterRequested = new AtomicLong(0);

	private final AtomicLong queueCounterOnline = new AtomicLong(0);

	/**
	 * Flag, ob eine Überlastsituation vorliegt und deswegen alle nicht schreibenden Tasks unterbrochen werden müssen
	 */
	private volatile boolean suspendReadingTasks;

	private final Object suspendReadingTasksLock = new Object();

	private final Semaphore subscriptionSemaphore;

	private final Set<IdDataIdentification> _pendingSubscriptions = Collections.synchronizedSet(new HashSet<>());

	private final long _maxWaitNanosPerSubscription;

	private final long _minWaitNanosPerSubscription;

	/**
	 * Limit für die Queue-Größe. Wenn mehr als die angegebene Zahl an Datensätzen in der Warteschlange sind,
	 * dann werden weitere Anmeldungen ausgebremst, bis die Queue abgearbeitet ist.
	 */
	private final int _queueLimitForSubscriptions;

	/**
	 * Wenn "true" wartet das Archivsystem nicht beim Beenden auf das Leeren aller Queues.
	 */
	private volatile boolean _fastExit;

	/**
	 * Anmelden mit Verzögerung (Sliding-Window-Mechanismus)
	 *
	 * @param davCon          Datenverteilerverbindung
	 * @param so              Objekt
	 * @param dataDescription DataDescription
	 * @return true wenn die Anmeldung durchgeführt werden konnte, sonst false (z.B. bei Anmeldung eines Empfängers auf Senke-Aspekt)
	 * @throws InterruptedException Beim Warten unterbrochen
	 */
	public boolean subscribe(final ClientDavInterface davCon, final SystemObject so, final DataDescription dataDescription) throws InterruptedException {
		IdDataIdentification identification = new IdDataIdentification(so, dataDescription);
		long waitTimeNanos = _minWaitNanosPerSubscription;
		if (countDataInQueues() > _queueLimitForSubscriptions) {
			waitTimeNanos += 10_000_000; // 10 Millisekunden extra warten, wenn die Queue mehr als halbvoll ist um das System nicht zu überlasten
		}
		if (waitTimeNanos > 0) {
			LockSupport.parkNanos(waitTimeNanos);
		}
		boolean semaphoreAcquired = false;
		if (_pendingSubscriptions.add(identification)) {
			long additionalWait = _maxWaitNanosPerSubscription - waitTimeNanos;
			if (subscriptionSemaphore.tryAcquire(additionalWait, TimeUnit.NANOSECONDS)) {
				semaphoreAcquired = true;
			} else {
				_pendingSubscriptions.remove(identification);
			}
		}
		try {
			davCon.subscribeReceiver(archiveDataReceiver, so, dataDescription, ReceiveOptions.delayed(), ReceiverRole.receiver());
			return true;
		} catch (Exception e) {
			// Anmeldung fehlgeschlagen (z.B. Empfänger auf Senke)
			if (semaphoreAcquired) {
				_pendingSubscriptions.remove(identification);
				subscriptionSemaphore.release();
			}
			logger.fine("Empfangsanmeldung fuer " + so + ", " + dataDescription + " fehlgeschlagen: " + e.getMessage());
			return false;
		}
	}

	/**
	 * Erzeugt den Warteschlangen-Manager.
	 *
	 * @param aMgr           Archiv-Manager
	 * @param runtimeControl Laufzeitumgebung um z.B. die Systemzeit abzufragen oder Spezialcode für Tests zu aktivieren
	 * @param dTree          DataIdentTree
	 * @param parameters     Queue-Einstellungen
	 */
	public InQueuesMgr(ArchiveManager aMgr, RuntimeControl runtimeControl, DataIdentTree dTree, final QueueParameters parameters) {
		archMgr = aMgr;
		this.runtimeControl = runtimeControl;
		didTree = dTree;
		_minWaitNanosPerSubscription = parameters.getMinWaitNanosPerSubscription();
		_maxWaitNanosPerSubscription = parameters.getMaxWaitNanosPerSubscription();


		// Queue für Simulations-Parametrierung mit ArchivConfig verbinden:
		archMgr.getArchivConfig().setSimConfigQueue(simParamQueue);

		backgroundTaskManager = new BackgroundTaskManager(archMgr);

		assignMultitasks();

		flowCtrl = new FlowControlTask();

		// parametrierbare Anzahl Query-Tasks starten:
		setQueryTaskNumbers(TM_QRY_A_NUM_DEFAULT, TM_QRY_B_NUM_DEFAULT, TM_QRY_C_NUM_DEFAULT);

		// die Zahl der ArchivInfo-Tasks ist fest:
		for (int i = 0; i < TM_ARC_INF_NUM; i++) {
			tskMgrs[TM_ARCINF_IDX].addTask(new ArchiveInfoQueryTask(tskMgrs[TM_ARCINF_IDX]));
		}

		// Alle Serialisierungstasks arbeiten die gleiche Queue ab
		serializeTasks = new SerializeTask[archMgr.getNumOfArchTasks()];
		for (int i = 0; i < serializeTasks.length; i++) {
			serializeTasks[i] = new SerializeTask(archMgr, serializeQueue);
		}

		queueCapacity = parameters.getTotalCapacityOfOnlineQueues();

		_queueLimitForSubscriptions = Math.min(parameters.getTotalCapacityOfOnlineQueues() / 2, 100000);

		if (queueCapacity < 1) {
			throw new IllegalArgumentException("Größe der Queues ist zu klein.");
		}

		assignArchiveTasks();
		subscriptionSemaphore = new Semaphore(parameters.getSubscriptionSlidingWindowSize());
	}

	/**
	 * Archiv-Tasks initialisieren
	 */
	private void assignArchiveTasks() {
		archiveQueue = new SignalingQueue<>();
		archiveTask = new ArchiveTask(archMgr, archiveQueue);
	}

	/**
	 * Task-Manager (Multitasks für Archivinfo und Queries, parametrierbare Anzahl)
	 */
	private void assignMultitasks() {
		// :
		tskMgrs[TM_QRYA_IDX] = new MultiTaskManager(archMgr, queryQueueA, "QueryPrioA");
		tskMgrs[TM_QRYB_IDX] = new MultiTaskManager(archMgr, queryQueueB, "QueryPrioB");
		tskMgrs[TM_QRYC_IDX] = new MultiTaskManager(archMgr, queryQueueC, "QueryPrioC");
		tskMgrs[TM_ARCINF_IDX] = new MultiTaskManager(archMgr, queryQueueInfo, "ArchivInfo");
	}

	/**
	 * Setzt die Anzahl der Threads für Archivanfragen auf die angegebenen Werte. Falls Threads entfernt werden, werden noch behandelte Anfragen zuende
	 * bearbeitet.
	 *
	 * @param numHi  Anzahl Tasks der Prioritaet "hoch"
	 * @param numMid Anzahl Tasks der Prioritaet "mittel"
	 * @param numLo  Anzahl Tasks der Prioritaet "niedrig"
	 */
	public void setQueryTaskNumbers(int numHi, int numMid, int numLo) {
		installQueryTask(tskMgrs[TM_QRYA_IDX], numHi);
		installQueryTask(tskMgrs[TM_QRYB_IDX], numMid);
		installQueryTask(tskMgrs[TM_QRYC_IDX], numLo);
	}

	public int getHiQueryTaskNum() {
		return tskMgrs[TM_QRYA_IDX].getTaskNum();
	}

	public int getMidQueryTaskNum() {
		return tskMgrs[TM_QRYB_IDX].getTaskNum();
	}

	public int getLoQueryTaskNum() {
		return tskMgrs[TM_QRYC_IDX].getTaskNum();
	}

	private void installQueryTask(MultiTaskManager mtm, int anz) {
		if (mtm.shouldTerminate()) return;

		int anzQTasks = mtm.getTaskNum();
		if (anz > anzQTasks) {
			for (int i = 0; i < (anz - anzQTasks); i++) {
				ArchiveQueryTask newTask = new ArchiveQueryTask(archMgr, mtm, flowCtrl, archMgr.getMaximumQueriesPerApplication());
				mtm.addTask(newTask);
			}
		} else {
			for (int i = 0; i < (anzQTasks - anz); i++) {
				mtm.removeTask();
			}
		}
	}

	/**
	 * Prüft ob Tasks temporaer blockiert werden sollen und blockiert ggf. den aufrufenden Task.
	 * <p>
	 * Diese Methode und {@link #suspendNonWriteTasks()} verwenden eine reduzierte Synchronisierung, um die Zahl der durchlaufenen synchronized-Bloecke
	 * gering zu halten.
	 * <p>
	 * Diese Methode darf nur ausgeführt werden, wenn der task nicht auf eine Datenidentifikation synchronisiert ist (Deadlock-Gefahr)
	 * </p>
	 *
	 * @param task Anzuhaltender Task
	 * @throws InterruptedException Thread unterbrochen
	 */
	public void suspendTaskIfNecessary(final Task task) throws InterruptedException {
		archMgr.getPersistenceManager().assertNoLocks();

		if (suspendReadingTasks)    // synchronized nur ausfuehren falls notwendig
		{
			synchronized (suspendReadingTasksLock) {
				while (suspendReadingTasks)    // hier ist der echte Test
				{
					if (terminated || task.isTerminated()) {
						// Während das Archiv oder der Task sich beendet, nicht blockieren.
						return;
					}

					suspendReadingTasksLock.wait(10000);

					// Nach 1 Sekunde erneut prüfen, ob die Tasks wieder fortgesetzt werden dürfen
					// Der Aufruf in de.bsvrz.ars.ars.mgmt.InQueuesMgr.ArchiveDataReceiver.processData() reicht nicht, da Tasks auch fortgesetzt werden sollen, wenn keine Daten eintreffen.
					suspendNonWriteTasks();
				}
			}
		}
	}

	/**
	 * Setzt das Suspend-Flag oder hebt es auf abhaengig von den beiden Grenzen.
	 * <p>
	 * Diese Methode und {@link #suspendTaskIfNecessary(Task)} verwenden eine reduzierte Synchronisierung um die Zahl der durchlaufenen synchronized-Bloecke
	 * gering zu halten. Voraußetzung: diese Methode wird nur von einem Thread aufgerufen.
	 */
	private void suspendNonWriteTasks() {
		// Zugriff auf nonWriteTasksSuspended findet nur in dieser Methode statt
		if (!suspendReadingTasks && getQueueLoad() > NONWRITE_SUSPEND_LIMIT) {
			suspendReadingTasks = true; // zu beliebigen Zeiten möglich
			logger.fine("Bearbeitung von Archivanfragen wegen Ueberlast ausgesetzt");
			//Absetzen der Betriebsmeldung
			MessageSender.getInstance().sendMessage(
					de.bsvrz.sys.funclib.operatingMessage.MessageType.APPLICATION_DOMAIN,
					MessageGrade.INFORMATION,
					"Die Warteschlangen sind zu " + (NONWRITE_SUSPEND_LIMIT * 100) + "% gefüllt - das Schreiben von Daten erhält eine höhere Priorität."
			);
		} else if (suspendReadingTasks && getQueueLoad() < NONWRITE_RESUME_LIMIT) {
			synchronized (suspendReadingTasksLock) {
				suspendReadingTasks = false;    // nur synchronisiert möglich
				suspendReadingTasksLock.notifyAll();
				logger.fine("Bearbeitung von Archivanfragen wieder aufgenommen");
				//Absetzen der Betriebsmeldung
				MessageSender.getInstance().sendMessage(
						de.bsvrz.sys.funclib.operatingMessage.MessageType.APPLICATION_DOMAIN,
						MessageGrade.INFORMATION,
						"Die Warteschlangen sind nur noch zu " + (NONWRITE_RESUME_LIMIT * 100) + "% gefüllt. Alle Tasks arbeiten wieder mit gleicher Priorität."
				);
			}
		}
	}

	/**
	 * Hält alle lesenden Tasks direkt an, ohne weitere Bedingungen zu prüfen.
	 */
	public void suspendNonWriteTasksDirect() {
		suspendReadingTasks = true;
	}


	/**
	 * In dieser Methode wird allen Tasks (falls notwendig) die Möglichkeit gegeben, benoetigte Objekte vom DAV zu laden.
	 */
	public void getObjectsFromDav() {
		for (MultiTaskManager tskMgr : tskMgrs) {
			tskMgr.getObjectsFromDav();
		}
	}

	/**
	 * In dieser Methode wird allen Tasks (falls notwendig) die Möglichkeit gegeben, sich auf Objekte anzumelden. Abmelden geschieht zentral über den {@link
	 * de.bsvrz.sys.funclib.losb.kernsoftware.ConnectionManager}.
	 */
	public void subscribeSettings() {
		ArchiveSettingsTask.subscribeObjects(archMgr, arsSettingsReceiver);
	}

	/**
	 * In dieser Methode wird allen Tasks (falls notwendig) die Möglichkeit gegeben, sich auf Objekte anzumelden. Abmelden geschieht zentral über den {@link
	 * de.bsvrz.sys.funclib.losb.kernsoftware.ConnectionManager}.
	 */
	public void subscribeQueries() {
		ArchiveQueryTask.subscribeObjects(archMgr, queryReceiver);
	}

	/**
	 * Startet alle Tasks (Multi- und Single-Tasks) sowie die Archiv-Tasks.
	 */
	public void startAllTasks() {
		resetDSCounter();
		for (int i = 0; i < NUM_OF_MULTITASK_MGR; i++) {
			tskMgrs[i].start();
		}
		backgroundTaskManager.startBackgroundTasks();

		for (SerializeTask serializeTask : serializeTasks) {
			serializeTask.start();
		}

		archiveTask.resetDSCounter();
		archiveTask.start();
	}

	/**
	 * Beendet alle Tasks (Multi- und Single-Tasks) sowie die Archiv-Tasks.
	 *
	 * @throws InterruptedException Thread unterbrochen
	 */
	public void stopAllTasks() throws InterruptedException {
		terminated = true;
		int totalTasks = 1 + NUM_OF_MULTITASK_MGR + backgroundTaskManager.getBackgroundTasks().size() + serializeTasks.length;

		backgroundTaskManager.stopBackgroundTasks();
		for (final MultiTaskManager tskMgr : tskMgrs) {
			tskMgr.terminateTask();
		}

		waitForEmptyArchiveQueues();

		for (SerializeTask serializeTask : serializeTasks) {
			serializeTask.terminateTask();
		}

		archiveTask.terminateTask();

		// Keine weiteren Ausgaben
		archiveDataReceiver._cronScheduler.shutdownNow();

		Thread.sleep(100);
		int debugCounter = 0;
		while (true) {
			int numTasksAlive = 0;

			for (final Task singleTsk : backgroundTaskManager.getBackgroundTasks()) {
				if (singleTsk.isAlive()) {
					numTasksAlive++;
				}
			}
			for (final MultiTaskManager tskMgr : tskMgrs) {
				if (tskMgr.isAlive()) {
					numTasksAlive++;
				}
			}

			for (SerializeTask serializeTask : serializeTasks) {
				if (serializeTask.isAlive()) {
					numTasksAlive++;
				}
			}
			if (archiveTask.isAlive()) {
				numTasksAlive++;
			}

			if (numTasksAlive == 0) break;
			if (debugCounter++ == 100) {
				debugCounter = 0;
				final StringBuilder builder = new StringBuilder();
				builder.append("Tasks werden beendet...\n");
				builder.append("Es müssen noch ");
				builder.append(numTasksAlive);
				builder.append(" von ");
				builder.append(totalTasks);
				builder.append(" Tasks beendet werden.\n");
				builder.append("Verbleibende Tasks:\n");
				for (final Task singleTsk : backgroundTaskManager.getBackgroundTasks()) {
					if (singleTsk.isAlive()) {
						builder.append(" - ").append(singleTsk.getName()).append("\n");
					}
				}
				for (final MultiTaskManager tskMgr : tskMgrs) {
					if (tskMgr.isAlive()) {
						builder.append(" - ").append(tskMgr.getName()).append("\n");
					}
				}
				for (final SerializeTask serializeTask : serializeTasks) {
					if (serializeTask.isAlive()) {
						builder.append(" - ").append(serializeTask.getName()).append("\n");
					}
				}
				if (archiveTask.isAlive()) {
					builder.append(" - ").append(archiveTask.getName()).append("\n");
				}
				logger.warning(builder.toString());
			}
			//noinspection BusyWait
			Thread.sleep(100); // OK, periodische Ausgabe erwünscht
		}

		logger.info("Alle Tasks korrekt beendet");
	}

	private void waitForEmptyArchiveQueues() {

		if (_fastExit) {
			archiveTask.terminateTask();
			archiveQueue.clear();
			logger.error("Warten auf leere Archiv-Warteschlange wurde übersprungen.");
			return;
		}

		int numOfWaitingData = archiveQueue.size();

		if (numOfWaitingData > 0) {
			try {
				logger.info(numOfWaitingData + " Datensaetze in der Warteschlange noch zu archivieren...");

				while (!_fastExit && !archiveQueue.waitUntilEmpty(Duration.ofMinutes(1))) {
					logger.info(archiveQueue.size() + " Datensaetze in der Warteschlange noch zu archivieren...");
				}
			} catch (InterruptedException e) {
				logger.error("Warten auf leere Archiv-Warteschlange unterbrochen.");
				return;
			}
			if (_fastExit) {
				archiveTask.terminateTask();
				archiveQueue.clear();
				logger.error("Warten auf leere Archiv-Warteschlange wurde übersprungen.");
				return;
			}

			logger.info("Alle Datensaetze archiviert.");
		}
	}

	/**
	 * Fuegt einen nachgeforderten Datensatz in die entsprechende Queue ein.
	 *
	 * @param archiveTime Archivzeit
	 * @param ad          Datensatz
	 */
	public void archiveRequestedData(long archiveTime, ArchiveData ad) {
		archiveQueue.add(new ArchiveRequestedData(SerializableDataset.createAsync(ad, archiveTime, serializeQueue), new IdDataIdentification(ad), ad.getDataKind().isDelayed()));
		queueCounterRequested.incrementAndGet();
	}

	/**
	 * Fügt dem Simulationsvarianten-Lösch-Task einen Auftrag hinzu.
	 *
	 * @param resultData Auftrag.
	 */
	public void insertInSimVarDeleteQueue(ResultData resultData) {
		backgroundTaskManager.getDeleteSimVarTask().submit(resultData);
	}

	/**
	 * Fügt dem Parametrierungs-Task einen weiteren Auftrag hinzu.
	 *
	 * @param resultData Auftrag. Enthält Informationen, auf welche Datenidentifikationen sich das Archivsystem anmelden soll.
	 */
	public void insertSimVarParam(SimulationResultData resultData) {
		simParamQueue.add(resultData);
	}

	/**
	 * Fügt dem Lösch-Task (endgültiges Löschen) einen Auftrag hinzu.
	 */
	public void startDeletePermanently() {
		backgroundTaskManager.getDeletePermanentlyTask().submit(new DeletePermanentlyTask.DeletePermanentlyToken());
	}

	/**
	 * Fügt dem Nachfordern-Task einen Auftrag hinzu.
	 *
	 * @param resultData Auftrag.
	 */
	public void insertInRequestQueue(ResultData resultData) {
		backgroundTaskManager.getRequestGapTask().submit(resultData);
	}


	/**
	 * Fügt in die Archivqueue ein {@link CloseContainerObject} ein, das signalisiert, dass der offene Container
	 * aufgrund einer Parameterierungsänderung abgeschlossen werden soll.
	 *
	 * @param cco {@link CloseContainerObject}
	 */
	public void insertCloseContainer(CloseContainerObject cco) {
		archiveQueue.add(cco);
		queueCounterOnline.incrementAndGet();
	}

	/**
	 * @return Gesamtkapazität der Archiv-Warteschlange
	 */
	public int getArchiveQueueCapacity() {
		return queueCapacity;
	}

	/**
	 * @return Auslastungsgrad der Warteschlange. Anhand dieses Wertes wird ermittelt, ob die Bearbeitung nicht schreibender Tasks ausgesetzt wird.
	 */
	public double getQueueLoad() {
		return (double) countDataInQueues() / (double) getArchiveQueueCapacity();
	}

	/**
	 * Zählt die Datensätze in der Eingangsqueue
	 *
	 * @return Anzahl Datensätze
	 */
	public int countDataInQueues() {
		return archiveQueue.size();
	}

	/**
	 * Setzt alle Zähler auf 0.
	 */
	public void resetDSCounter() {
		receivedCounterOnline.set(0);
		queueCounterRequested.set(0);
		queueCounterOnline.set(0);
		archiveTask.resetDSCounter();
	}

	/**
	 * Liefert den Task, der für die Bearbeitung der Archiveinstellungen verantwortlich ist. Dies ist notwendig, damit der {@link ArchiveManager} beim Start einen
	 * {@link de.bsvrz.ars.ars.mgmt.tasks.ArchiveSettingsTask.ArSSettingListener} einhängen und darauf warten kann, dass die Bearbeitung der Archiveinstellungen
	 * abgeschlossen ist. Das Archivsystem wartet beim Start auf die Archiv-Einstellungen, da es nicht sinnvoll ist, die ersten Sekunden mit Default-Werten
	 * loszulaufen.
	 *
	 * @return Den Task, der für die Bearbeitung der Archiveinstellungen verantwortlich ist.
	 */
	public ArchiveSettingsTask getArchiveSettingsTask() {
		return backgroundTaskManager.getArchiveSettingsTask();
	}


	/**
	 * Die Klasse {@link ArchivConfig} meldet die zu archivierenden Daten mit dem Objekt als Empfaenger an, das von dieser Methode geliefert wird.
	 *
	 * @return Empfaengerobjekt für zu archivierende Daten.
	 */
	public ArchiveDataReceiver getArchiveDataReceiver() {
		return archiveDataReceiver;
	}

	/**
	 * Die Klasse {@link ArchivConfig} meldet die Quittungen für die zu archivierenden Daten mit dem Objekt als Sender an, das von dieser Methode geliefert wird.
	 *
	 * @return Senderobjekt für die Quittungen der zu archivierenden Daten.
	 */
	public DataAckSender getDataAckSender() {
		return dataAckSender;
	}

	/**
	 * Setzt, ob sich der InQueuesMgr schnell terminieren soll. Bei einer schnellen Terminierung wird nicht
	 * auf die Abarbeitugn aller Queues gewartet.
	 *
	 * @param quickExit true wenn schnell terminiert werden soll, sonst false
	 */
	public void setFastExit(final boolean quickExit) {
		_fastExit = quickExit;
	}

	public DeletePermanentlyTask getDeletePermanentlyTask() {
		return backgroundTaskManager.getDeletePermanentlyTask();
	}

	public ArchiveTask getArchiveTask() {
		return archiveTask;
	}

	/**
	 * Fügt in die Archivqueue einen Benachrichtigungstoken ein, der bei seiner "Archivierung" signalisiert,
	 * dass alle (zuvor in die Queue eingefügten) nachgeforderten Datensätze archiviert wurden.
	 */
	public void insertRequestedDataFinishedNotification() {
		archiveQueue.add(new RequestedDataFinished());
	}

	/**
	 * Schätzt den Speicherverbrauch der Archivqueue ab
	 *
	 * @return Archivqueue
	 */
	public long estimateQueueMemoryUsage() {
		return 12L + archiveQueue.capacityEstimate() * 4L + archiveQueue.sumElements(ArchiveJob::estimateMemoryUsage);
	}

	public BackgroundTaskManager getBackgroundTaskManager() {
		return backgroundTaskManager;
	}

	/**
	 * Empfaengerklasse für Archiveinstellungen
	 */
	public final class ArchiveSettingsReceiver extends DataReceiver {

		@Override
		public void processData(ResultData rd) {
			logger.finer("Archiveinstellung erhalten: ", rd);
			if (rd.hasData()) {
				backgroundTaskManager.getArchiveSettingsTask().submit(rd);
			}
		}
	}

	/**
	 * Empfaengerklasse für Archivanfragen
	 */
	public final class QueryReceiver extends DataReceiver {

		private Deserializer deserializer;

		@Override
		public void processData(ResultData rd) {
			if (rd.hasData()) {
				int msgType = rd.getData().getUnscaledValue(PidQuery.ATT_MESSAGE_TYP_NAME).intValue();
				switch (msgType) {
					case MessageType.QUERY -> {  // Anfrage
						ArchiveQueryPriority prio;
						try {
							prio = getQueryPrio(rd);
						} catch (Exception e) {
							logger.warning("Die Priorität einer Archivanfrage konnte nicht ermittelt werden: " + rd, e.getMessage());
							prio = ArchiveQueryPriority.MEDIUM;

						}
						SignalingQueue<ResultData> archQueue = getArchiveQueue(prio);
						archQueue.add(rd);
					}
					case MessageType.QUERY_INFO -> // Informationsanfrage:
							queryQueueInfo.add(rd);
					case MessageType.DELETE -> // Loeschauftrag für eine Simulationsvariante:
							backgroundTaskManager.getDeleteSimVarTask().submit(rd);
					case MessageType.STREAM_CONTROL ->    // Flusskontrolle:
							flowCtrl.processFlowCtrl(rd);
					case MessageType.REQUEST_DID, MessageType.REQUEST_AUTOM ->
							backgroundTaskManager.getRequestGapTask().submit(rd);
					case MessageType.NUM_QUERIES_INFO -> backgroundTaskManager.getNumQueriesInfoTask().submit(rd);
					default -> {
						logger.warning("Nachrichtentyp erhalten, der nicht bearbeitet werden kann: " + msgType + " aus: " + Util.rd2Str(rd));
						// Wir kennen den Anfragetyp nicht, können aber davon ausgehen, dass der zugehörige Antworttyp eins größer ist.
						QueryHandler handler = new QueryHandler("InQueuesMgr", archMgr, msgType + 1, rd.getData());
						handler.sendErrorResponse("Nachrichtentyp erhalten, der nicht bearbeitet werden kann: \"" + rd.getData().getTextValue(PidQuery.ATT_MESSAGE_TYP_NAME).getText() + "\". Die Funktion wird von dieser Version des Archivsystems nicht unterstützt.");
					}
				}
			}
		}

		private ArchiveQueryPriority getQueryPrio(ResultData rd) throws IOException, NoSuchVersionException {
			byte[] data = rd.getData().getUnscaledArray(PidQuery.ATT_DATA_NAME).getByteArray();
			InputStream stream = new ByteArrayInputStream(data, 4, data.length - 4);
			int deserVersion = Util.getSerVersion(data);
			if (deserializer == null || deserializer.getVersion() != deserVersion) {
				deserializer = SerializingFactory.createDeserializer(deserVersion, stream);
			} else {
				deserializer.setInputStream(stream);
			}
			return ArchiveQueryPriority.getInstance(deserializer.readInt()); // Prioritaet
		}
	}

	private SignalingQueue<ResultData> getArchiveQueue(ArchiveQueryPriority prio) {
		return switch (prio.getCode()) {
			case 1 -> queryQueueA;
			case 2 -> queryQueueB;
			case 3 -> queryQueueC;
			default -> throw new AssertionError();
		};
	}

	/**
	 * Empfaengerklasse für Archivdaten
	 */
	public final class ArchiveDataReceiver extends DataReceiver {

		/** Betriebsmeldung-Nachricht fürs Anhalten bei Archivzeit-Rücksprung */
		public static final String MSG_PID_ATBACKSTEP_SUSPEND = "Archivzeit-Ruecksprung (suspend)";

		/** Betriebsmeldung-Nachricht fürs Fortsetzen bei Archivzeit-Rücksprung */
		public static final String MSG_PID_ATBACKSTEP_RESUME = "Archivzeit-Ruecksprung (resume)";

		/** Betriebsmeldung-Nachricht fürs Anhalten bei Archivzeit-Rücksprung */
		public static final String MSG_PID_ATFWSTEP_SUSPEND = "Archivzeit-Vorwaertssprung (suspend)";

		/** Betriebsmeldung-Nachricht fürs Fortsetzen bei Archivzeit-Vorwärtssprung */
		public static final String MSG_PID_ATFWSTEP_RESUME = "Archivzeit-Vorwaertssprung (resume)";

		private boolean suspendArchMessageSentBS, resumeArchMessageSentBS;    // Backstep

		private boolean suspendArchMessageSentFS, resumeArchMessageSentFS;    // Forward Step

		/**
		 * Wird auf false gesetzt, nachdem der erste Datensatz empfangen wurde. Wird verwendet, damit Vorwärtssprungs auch nach einem Neustart ses Archivsystems
		 * korrekt erkannt wird
		 */
		private boolean firstData = true;

		/**
		 * max. Archivzeitsprung vorwärts (siehe TAnfArS 5.1.2.4.3.3)
		 */
		private long tMaxATimeForwardStep = 3600 * 1000;    // Defaultwert 1 Stunde gemäß DatK

		private volatile long _statisticLastTotalReceivedDataSetCount;

		private volatile long _statisticLastTotalQueuedDataSetCount;

		private volatile long _statisticLastTotalSuccessDataSetCount;

		private volatile long _statisticLastTotalFailedDataSetCount;

		private volatile long _statisticLastTotalQueuedDataSetCountRequested;

		private volatile long _statisticLastTotalSuccessDataSetCountRequested;

		private volatile long _statisticLastTotalFailedDataSetCountRequested;

		private long[] _statisticLastCacheCounts = {0, 0, 0, 0};

		private final CronScheduler _cronScheduler;

		/** 
		 * Erstellt einen neuen ArchiveDataReceiver.
		 */
		public ArchiveDataReceiver() {
			_cronScheduler = new CronScheduler(1, r -> {
				Thread statistics = new Thread(r, "Statistics");
				statistics.setDaemon(true);
				return statistics;
			});
			_cronScheduler.schedule(this::printStatistics, CronDefinition.EVERY_MINUTE);
		}

		@Override
		public void update(final ResultData[] rds) {
			for (ResultData rd : rds) {
				if (!_pendingSubscriptions.isEmpty() && _pendingSubscriptions.remove(new IdDataIdentification(rd.getObject(), rd.getDataDescription()))) {
					subscriptionSemaphore.release();
				}
			}
			super.update(rds);
		}

		@Override
		public void processData(ResultData rd) {
			long aTime = runtimeControl.getSystemTime();

			receivedCounterOnline.incrementAndGet();

			// Zu Testzwecken
			if (runtimeControl.archiveOnlyData() && !rd.hasData()) return;

			DataIdentNode din = didTree.get(rd);
			if (din.arSParamIsArchivieren() && din.isArSParameterized()) {
				final long lastArchiveTime = ArchiveTask.getLastArchiveTime();
				// Folgende Abfrage sorgt dafür, dass kurze Rücksprünge der Zeit (bis 10 Sekunden) toleriert werden, in dem einfach die entsprechende
				// Zeit abgewartet wird und die aktuelle Zeit erneut abgefragt wird.
				if (aTime < lastArchiveTime) {
					long delta = lastArchiveTime - aTime;
					if (delta < 10000) {
						try {
							System.out.println(delta);
							Thread.sleep(delta);
						} catch (InterruptedException ignored) {
						}
						aTime = runtimeControl.getSystemTime();
					}
				}
				if (!testForArchiveTimeBackstep(aTime, lastArchiveTime) && !testForArchiveTimeForwardStep(aTime, lastArchiveTime)) {
					putDataInQueue(rd, aTime);
					ArchiveTask.setLastArchiveTime(aTime);

					// Auf Überlast prüfen und ggf. alle nicht schreibenden Tasks anhalten
					suspendNonWriteTasks();
				}
			}
		}

		/**
		 * Laufende Statistik ausgeben
		 */
		public void printStatistics() {
			StringBuilder message = new StringBuilder();
			message.append("Aktivität der Warteschlange im letzten Ausgabeintervall:");

			final long totalReceivedDataSetCount = getReceivedCountOnline();
			message.append(String.format("%n%20d Datensätze empfangen", totalReceivedDataSetCount - _statisticLastTotalReceivedDataSetCount));
			_statisticLastTotalReceivedDataSetCount = totalReceivedDataSetCount;

			final long totalQueuedDataSetCount = getQueuedCountOnline();
			message.append(String.format("%n%20d Datensätze weitergeleitet", totalQueuedDataSetCount - _statisticLastTotalQueuedDataSetCount));
			_statisticLastTotalQueuedDataSetCount = totalQueuedDataSetCount;

			final long totalSuccessDataSetCount = getSuccessCountOnline();
			message.append(String.format("%n%20d Datensätze erfolgreich archiviert", totalSuccessDataSetCount - _statisticLastTotalSuccessDataSetCount));
			_statisticLastTotalSuccessDataSetCount = totalSuccessDataSetCount;

			final long totalFailedDataSetCount = getFailedCountOnline();
			message.append(String.format("%n%20d Datensätze nicht erfolgreich archiviert", totalFailedDataSetCount - _statisticLastTotalFailedDataSetCount));
			_statisticLastTotalFailedDataSetCount = totalFailedDataSetCount;

			final long totalQueuedDataSetCountRequested = getQueuesCountRequested();
			message.append(String.format("%n%20d nachgeforderte Datensätze empfangen", totalQueuedDataSetCountRequested - _statisticLastTotalQueuedDataSetCountRequested));
			_statisticLastTotalQueuedDataSetCountRequested = totalQueuedDataSetCountRequested;

			final long totalSuccessDataSetCountRequested = getSuccessCountRequested();
			message.append(String.format("%n%20d nachgeforderte Datensätze erfolgreich archiviert", totalSuccessDataSetCountRequested - _statisticLastTotalSuccessDataSetCountRequested));
			_statisticLastTotalSuccessDataSetCountRequested = totalSuccessDataSetCountRequested;

			final long totalFailedDataSetCountRequested = getFailedCountRequested();
			message.append(String.format("%n%20d nachgeforderte Datensätze nicht erfolgreich archiviert", totalFailedDataSetCountRequested - _statisticLastTotalFailedDataSetCountRequested));
			_statisticLastTotalFailedDataSetCountRequested = totalFailedDataSetCountRequested;

			PersistenceManager.Statistics statistics = archMgr.getPersistenceManager().getStatistics();

			long queueSize = statistics.getQueueSize();
			int queueLoad = (int) ((double) queueSize / (double) getArchiveQueueCapacity() * 100.0);
			message.append(String.format("%n%20d Datensätze noch in der Warteschlange befindlich", queueSize));
			message.append(String.format("%n%19d%% Auslastung der Warteschlange", queueLoad));

			message.append("\nAktivität der Zwischenspeicherung in geschlossenen Cache-Objekten: ");
			final long[] cacheCounts = CacheManager.getInstance().getCounts();
			message.append(String.format("%n%20d Datei-Schreibvorgänge", cacheCounts[0] - _statisticLastCacheCounts[0]));
			message.append(String.format("%n%20d verarbeitete Datenblöcke", cacheCounts[1] - _statisticLastCacheCounts[1]));
			message.append(String.format("%n%20d zwischengespeicherte Datenblöcke", cacheCounts[2] - _statisticLastCacheCounts[2]));
			message.append(String.format("%n%20d nicht zwischengespeicherte Datenblöcke", cacheCounts[3] - _statisticLastCacheCounts[3]));
			_statisticLastCacheCounts = cacheCounts;

			message.append("\nAktueller Arbeitsspeicherverbrauch: ");
			Runtime runtime = Runtime.getRuntime();
			long usedMemory = runtime.totalMemory() - runtime.freeMemory();
			message.append(String.format("%n%20.3f MB insgesamt, davon", usedMemory / 1048576.0));
			message.append(String.format("%n%20.3f MB zur Synchronisierung", statistics.getLockMemory() / 1048576.0));
			message.append(String.format("%n%20.3f MB für %d Datenknoten", statistics.getNodeMemory() / 1048576.0, statistics.getNumNodes()));
			message.append(String.format("%n%20.3f MB für %d offene Container", statistics.getActiveMemory() / 1048576.0, statistics.getNumOpenContainerData()));
			message.append(String.format("%n%20.3f MB für %d Datensätze in der Warteschlange", statistics.getQueueMemory() / 1048576.0, queueSize));
			message.append(String.format("%n%20.3f MB zwischengespeichert für Archivierungs-Cache (davon %.0f%% ungenutzt)",
					statistics.getCachedMemory().totalBytesUsed() / 1048576.0,
					100.0 - 100.0 * (double) statistics.getCachedMemory().cachedBytes() / statistics.getCachedMemory().totalBytesUsed())
			);
			var tasks = TaskManager.getActiveTasks();
			message.append("\nLaufende Tasks: ");
			int i = 1;
			for (var task : tasks) {
				message.append(String.format("%n%19d) %s", i++, task.getName()));
			}
			if (tasks.isEmpty()) {
				message.append(String.format("%n%20s Keine aktiven Tasks", "---"));
			}

			logger.info(message.toString());
		}

		private boolean testForArchiveTimeForwardStep(long aTime, final long lastArchiveTime) {
			if (firstData) {    //nach einem Neustart des Archivsystems soll die Überprüfung erst beim zweiten Archivdatensatz vorgenommen werden
				firstData = false;
				return false;
			} else if (aTime - lastArchiveTime > tMaxATimeForwardStep) {
				if (!suspendArchMessageSentFS) {
					sendArchiveSuspendMsgFS(aTime);
					suspendArchMessageSentFS = true;
					resumeArchMessageSentFS = false;
				}
				return true;
			} else {
				if (suspendArchMessageSentFS && !resumeArchMessageSentFS) {
					sendArchiveResumeMsgFS(aTime);
					resumeArchMessageSentFS = true;
					suspendArchMessageSentFS = false;
				}
				return false;
			}
		}

		/**
		 * Prüft ob ein Rücksprung der Archivzeit aufgetreten ist und sendet entsprechende Betriebsmeldungen.
		 *
		 * @param aTime           aktuelle Archivzeit
		 * @param lastArchiveTime letzte archivierte Archivzeit
		 * @return Wahr, wenn ein Rücksprung aufgetreten ist, falsch sonst
		 */
		private boolean testForArchiveTimeBackstep(long aTime, final long lastArchiveTime) {
			if (aTime < lastArchiveTime) {
				if (!suspendArchMessageSentBS) {
					sendArchiveSuspendMsgBS(aTime);
					suspendArchMessageSentBS = true;
					resumeArchMessageSentBS = false;
				}
				return true;
			} else {
				if (suspendArchMessageSentBS && !resumeArchMessageSentBS) {
					sendArchiveResumeMsgBS(aTime);
					resumeArchMessageSentBS = true;
					suspendArchMessageSentBS = false;
				}
				return false;
			}
		}

		/**
		 * Legt das Datum in die passende Warteschlange. Falls die voll ist, wird einmal eine Log-Message ausgegeben. Wenn sie nicht mehr voll ist, wird ebenfalls
		 * einmal eine Log-Message ausgegeben.
		 *
		 * @param resultData  zu archivierendes Datum
		 * @param archiveTime Archivzeit des zu archivierenden Datums
		 */
		private void putDataInQueue(ResultData resultData, long archiveTime) {
			archiveQueue.add(new ArchiveOnlineData(SerializableDataset.createAsync(resultData, archiveTime, InQueuesMgr.this.serializeQueue), new IdDataIdentification(resultData), resultData.getDataKind().isDelayed()));
			queueCounterOnline.incrementAndGet();
		}

		private void sendArchiveSuspendMsgFS(long aTime) {
			String msg = "Es ist ein Vorwaertssprung der Archivzeit aufgetreten: die Archivzeit des soeben erhaltenen Datensatzes ("
					+ Util.timestrMillisFormatted(aTime) + ") liegt mehr als " + tMaxATimeForwardStep / 1000 + "sec ("
					+ Util.relTimestrMillis(tMaxATimeForwardStep) + ") nach der Archivzeit des zuletzt archivierten Datensatzes ("
					+ Util.timestrMillisFormatted(ArchiveTask.getLastArchiveTime())
					+ "). Das Archivsystem stellt die Archivierung ein, bis die Archivzeit wieder innerhalb der zulaessigen Spanne liegt.";
			MessageSender.getInstance().sendMessage(
					MSG_PID_ATFWSTEP_SUSPEND,
					de.bsvrz.sys.funclib.operatingMessage.MessageType.APPLICATION_DOMAIN,
					"",
					MessageGrade.FATAL,
					MessageState.MESSAGE,
					msg
			);
			logger.info("Betriebsmeldung:" + Debug.NEWLINE + msg);
		}

		private void sendArchiveResumeMsgFS(long aTime) {
			String msg = "Der Vorwaertssprung der Archivzeit ist behoben: die Archivzeit des soeben erhaltenen Datensatzes ("
					+ Util.timestrMillisFormatted(aTime) + ") liegt wieder innerhalb der zulaessigen Spanne von "
					+ tMaxATimeForwardStep / 1000 + "sec (" + Util.relTimestrMillis(tMaxATimeForwardStep)
					+ ") seit der Archivzeit des zuletzt archivierten Datensatzes (" + Util.timestrMillisFormatted(ArchiveTask.getLastArchiveTime())
					+ ". Das Archivsystem hat die Archivierung wieder aufgenommen.";
			MessageSender.getInstance().sendMessage(
					MSG_PID_ATFWSTEP_RESUME,
					de.bsvrz.sys.funclib.operatingMessage.MessageType.APPLICATION_DOMAIN,
					"",
					MessageGrade.INFORMATION,
					MessageState.MESSAGE,
					msg
			);
			logger.info("Betriebsmeldung:" + Debug.NEWLINE + msg);
		}

		private void sendArchiveSuspendMsgBS(long aTime) {
			String msg = "Es ist ein Ruecksprung der Archivzeit aufgetreten: die Archivzeit des zuletzt archivierten " + "Datensatzes ("
					+ Util.timestrMillisFormatted(ArchiveTask.getLastArchiveTime()) + ") liegt nach der Archivzeit des soeben erhaltenen Datensatzes ("
					+ Util.timestrMillisFormatted(aTime) + "). Das Archivsystem stellt die Archivierung ein, bis die Archivzeit wieder nach der Archivzeit"
					+ " des zuletzt archivierten Datensatzes liegt.";
			MessageSender.getInstance().sendMessage(
					MSG_PID_ATBACKSTEP_SUSPEND,
					de.bsvrz.sys.funclib.operatingMessage.MessageType.APPLICATION_DOMAIN,
					"",
					MessageGrade.FATAL,
					MessageState.MESSAGE,
					msg
			);
			logger.info("Betriebsmeldung:" + Debug.NEWLINE + msg);
		}

		private void sendArchiveResumeMsgBS(long aTime) {
			String msg = "Der Ruecksprung der Archivzeit ist behoben: die Archivzeit des zuletzt archivierten " + "Datensatzes ("
					+ Util.timestrMillisFormatted(ArchiveTask.getLastArchiveTime())
					+ ") liegt wieder vor der Archivzeit des soeben erhaltenen Datensatzes (" + Util.timestrMillisFormatted(aTime)
					+ "). Das Archivsystem hat die Archivierung wieder aufgenommen.";
			MessageSender.getInstance().sendMessage(
					MSG_PID_ATBACKSTEP_RESUME,
					de.bsvrz.sys.funclib.operatingMessage.MessageType.APPLICATION_DOMAIN,
					"",
					MessageGrade.INFORMATION,
					MessageState.MESSAGE,
					msg
			);
			logger.info("Betriebsmeldung:" + Debug.NEWLINE + msg);
		}

		/**
		 * Setzt den maximal erlaubten Vorwärtssprung der Archivzeit.
		 *
		 * @param tMaxSeconds max. erlaubter Sprung (in Sekunden)
		 */
		public void setTMaxATimeForwardStep(long tMaxSeconds) {
			tMaxATimeForwardStep = tMaxSeconds * 1000;
		}

		public long getTMaxATimeForwardStep() {
			return tMaxATimeForwardStep;
		}
	}

	/**
	 * @return Zahl aller seit dem Start oder letzten {@link #resetDSCounter()} von {@link #getArchiveDataReceiver()} empfangenen Datensätze, egal ob diese
	 * archiviert werden oder nicht. Kann zur Test-Synchronisation verwendet werden.
	 */
	public long getReceivedCountOnline() {
		return receivedCounterOnline.longValue();
	}

	/**
	 * @return Zahl aller seit dem Start oder letzten {@link #resetDSCounter()} Datensätze, die in eine Queue eingefügt wurden.
	 */
	public long getQueuedCountTotal() {
		return getQueuedCountOnline() + getQueuesCountRequested();
	}

	/**
	 * @return Zahl aller seit dem Start oder letzten {@link #resetDSCounter()} Online-Datensätze, die in eine Queue eingefügt wurden.
	 */
	public long getQueuedCountOnline() {
		return queueCounterOnline.get();
	}

	/**
	 * @return Zahl aller seit dem Start oder letzten {@link #resetDSCounter()} nachgeforderten Datensätze, die in eine Queue eingefügt wurden.
	 */
	public long getQueuesCountRequested() {
		return queueCounterRequested.get();
	}

	/**
	 * @return Zahl aller seit dem Start oder letzten {@link #resetDSCounter()} Datensätze, die nicht erfolgreich archiviert wurden
	 */
	public long getFailedCountTotal() {
		return archiveTask.getFailedCount();
	}

	/**
	 * @return Zahl aller seit dem Start oder letzten {@link #resetDSCounter()} Datensätze, die erfolgreich archiviert wurden
	 */
	public long getSuccessCountTotal() {
		return archiveTask.getSuccessCount();
	}

	/**
	 * @return Zahl aller seit dem Start oder letzten {@link #resetDSCounter()} Online-Datensätze, die nicht erfolgreich archiviert wurden
	 */
	public long getFailedCountOnline() {
		return archiveTask.getFailedCountOnline();
	}

	/**
	 * @return Zahl aller seit dem Start oder letzten {@link #resetDSCounter()} Online-Datensätze, die erfolgreich archiviert wurden
	 */
	public long getSuccessCountOnline() {
		return archiveTask.getSuccessCountOnline();
	}

	/**
	 * @return Zahl aller seit dem Start oder letzten {@link #resetDSCounter()} nachgeforderten Datensätze, die nicht erfolgreich archiviert wurden
	 */
	public long getFailedCountRequested() {
		return archiveTask.getFailedCountRequested();
	}

	/**
	 * @return Zahl aller seit dem Start oder letzten {@link #resetDSCounter()} nachgeforderten Datensätze, die erfolgreich archiviert wurden
	 */
	public long getSuccessCountRequested() {
		return archiveTask.getSuccessCountRequested();
	}

	/**
	 * @return Zahl aller seit dem Start oder letzten {@link #resetDSCounter()} erfolgreich verarbeiteten Close-Container-Datensätze. Kann zur
	 * Test-Synchronisation verwendet werden.
	 */
	public long getCloseContainerSuccess() {
		return archiveTask.getCloseContainerSuccess();
	}

	/**
	 * Gibt die Laufzeitsteuerung zurück, die u.a. die Archivzeiten ermittelt und Test-Verhalten ermöglicht.
	 * @return RuntimeControl
	 */
	public RuntimeControl getRuntimeControl() {
		return runtimeControl;
	}

	/**
	 * Abstrakte Klasse, die zum Empfang bestimmter Daten (z.B. von Archiv-Queries) verwendet werden kann. Auf diese Weise muss das eingehende Datum nicht mehr
	 * anhand von IDs unterschieden und der passenden Warteschlange zugewiesen werden.
	 */
	public sealed abstract class DataReceiver implements ClientReceiverInterface {

		@Override
		public void update(ResultData[] rds) {
			logger.finer(() -> getClass().getSimpleName() + " update(): ResultData[] " + ("mit " + rds.length + " Datensaetzen") + " erhalten");

			for (ResultData rd : rds) {

				// *** NUR FÜR TESTZWECKE START
				long dataIndexOffset = runtimeControl.getDataIndexOffset();
				if (dataIndexOffset != 0 || runtimeControl.fixNoSourceDataTime()) {
					byte errorFlag = (byte) (rd.getDataType().getCode() - 1);
					long dataTime = rd.getDataTime();
					if (runtimeControl.fixNoSourceDataTime() && rd.getDataType().equals(DataState.NO_SOURCE)) {
						dataTime = runtimeControl.getSystemTime();
					}
					rd = new ResultData(rd.getObject(), rd.getDataDescription(), rd.isDelayedData(), rd.getDataIndex() + dataIndexOffset, dataTime, errorFlag, rd.getData());
				}
				// *** NUR FÜR TESTZWECKE ENDE

				if (terminated) {
//					logger.info("Archivsystem wird gerade beendet, Daten werden nicht mehr bearbeitet.");
					return;
				}
				// TAnfArS 5.1.2.4.3.7: initiale "keine Quelle"-DS ignorieren
				if (!isInitialNoSourceDS(rd)) {
					processData(rd);
				}
			}
		}

		/**
		 * Prüft, ob es sich bei dem Datensatz um einen initialen "Keine Quelle"-Datensatz handelt, der vom DaV erzeugt wird, sobald sich ein Empfaenger angemeldet
		 * hat, ohne dass eine Quelle existiert.
		 *
		 * @param rd Datensatz
		 * @return Wahr, falls initialer "Keine Quelle"-Datensatz, falsch sonst
		 */
		private boolean isInitialNoSourceDS(ResultData rd) {
			return Util.dIdxSrcSubscrTime(rd) == 0 && Util.dIdxLfdnr(rd) == 0;
		}

		/**
		 * Verarbeitet ein ResultData-Objekt. Diese Methode wird von abgeleiteten Klassen implementiert.
		 * @param rd Datensatz vom Datenverteiler
		 */
		public abstract void processData(ResultData rd);
	}

	/**
	 * Klasse, die Quittungen sendet
	 */
	public final class DataAckSender implements ClientSenderInterface {

		/**
		 * Sendet eine Quittung
		 * @param rd     Datensatz
		 * @param qAspID Quittungsaspekt
		 * @throws DataNotSubscribedException Daten noch nicht angemeldet?
		 * @throws SendSubscriptionNotConfirmed Sendeanmeldung noch nicht bestätigt
		 */
		public void sendAck(ResultData rd, long qAspID) throws DataNotSubscribedException, SendSubscriptionNotConfirmed {
			ResultData qRd = new ResultData(
					rd.getObject(),
					new DataDescription(rd.getDataDescription().getAttributeGroup(), archMgr.getAsp(qAspID)),
					runtimeControl.getSystemTime(),
					rd.getData()
			);
			archMgr.getDavCon().sendData(new ResultData[]{qRd});
		}

		@Override
		public boolean isRequestSupported(SystemObject object, DataDescription dataDescription) {
			return true;
		}

		@Override
		public void dataRequest(SystemObject object, DataDescription dataDescription, byte state) {
			if (state == ClientSenderInterface.STOP_SENDING_NOT_A_VALID_SUBSCRIPTION) {
				logger.warning(
						"Sendesteuerung fuer Quittung '" + object + ", " + dataDescription + "': STOP_SENDING_NOT_A_VALID_SUBSCRIPTION" + Debug.NEWLINE
								+ "Moeglicherweise ist unter dieser Datenidentifikation bereits eine Quelle angemeldet."
				);
			} else if (state == ClientSenderInterface.STOP_SENDING_NO_RIGHTS) {
				logger.warning("Sendesteuerung fuer Quittung " + object + ", " + dataDescription + " : STOP_SENDING_NO_RIGHTS");
			}
		}
	}
}
