/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.commands;

import de.bsvrz.ars.ars.mgmt.ArchiveManager;
import de.bsvrz.ars.ars.mgmt.tasks.ArchiveQueryTask;
import de.bsvrz.ars.ars.mgmt.tasks.scheduler.TaskScheduler;
import de.bsvrz.ars.ars.persistence.directories.mgmt.PersistenceDirectoryManager;
import de.bsvrz.ars.ars.persistence.directories.mgmt.TimeBasedPersistenceDirectoryManager;
import de.bsvrz.ars.ars.persistence.layout.ShortPersistenceDirectoryLayout;
import de.bsvrz.ars.ars.persistence.writer.ArchiveTask;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.losb.datk.ContainerSettings;
import de.bsvrz.sys.funclib.losb.datk.ContainerSettings.ContSettingsExc;
import de.bsvrz.sys.funclib.losb.util.Util;
import de.bsvrz.sys.funclib.losb.util.cmdinterface.*;

import java.io.*;
import java.net.Socket;
import java.net.UnknownHostException;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import static java.nio.charset.StandardCharsets.ISO_8859_1;


/**
 * @author beck et al. projects GmbH
 * @author Alexander Schmidt
 * @version $Revision$ / $Date$ / ($Author$)
 */
public class ArSCmdInterface {

	/** Max. Zeit der Inaktivität, nach der der Server die Verbindung zum Client beendet. */
	public static final int SEND_TIMEOUT = 60 * 5;

	/** Das Beenden-Kommando ist auf der ersten Menueebene unter dieser Nummer zu finden: */
	private static final int QUIT_CMD_NR = 5;

	private final ArchiveManager archMgr;

	private final CmdInterpreter cmdInt;

	/**
	 * Erzeugt den Telnet Server. Startet ihn jedoch nicht - dazu muss {@link #start()} aufgerufen werden.
	 *
	 * @param am   ArchivManager
	 * @param port Port des Telnet Servers
	 */
	public ArSCmdInterface(ArchiveManager am, int port) {
		final int timeout = 10;
		archMgr = am;
		cmdInt = new CmdInterpreter(port, timeout);
		CmdMenu root = new CmdMenu("Archivsystem", "");

		//Submenü Einstellungen
		CmdMenu subMenu = new CmdMenu("Einstellungen", "");
		root.addNode(subMenu);
		subMenu.addCmd(new EnableDeleteCommand(am));
		CmdMenu debugLevelMenu = InfoCommands.getDebugLevelMenu();
		subMenu.addNode(debugLevelMenu);
		subMenu.addCmd(
				new Command("Debug-Level ausgeben", "") {
					@Override
					public void execute() throws Exception {
						println("Angemeldete Logger:\n" + Debug.debugInfo());
					}
				}
		);

		// Archivinformationen
		subMenu = new CmdMenu("Archivinformationen", "");
		root.addNode(subMenu);

		// Details einer Datenidentifikation abrufen (manuelle Navigation)
		subMenu.addCmd(new DataIdentInfo(archMgr));

		subMenu.addCmd(
				new Command("Auslastung der Archivierungs-Warteschlange", "Beenden mit 'q'") {
					@Override
					public void execute() throws Exception {
						println("Ausgabe kann mit 'q' und Enter beendet werden.");
						println("Die Ausgabe wird unterdrückt, wenn sie sich zur vorherigen nicht geändert hat.");
						println("Aktualisierungsrate in Millisekunden:");
						ArchiveMonitor monitor = null;
						try {
							int n = Integer.parseInt(readln());
							monitor = new ArchiveMonitor(archMgr, n, this);
							monitor.start();
						}
						catch(Exception e) {
							println("Ungültige Eingabe!");
						}
						while(true) {
							if(readln().equalsIgnoreCase("q")) {
								if(monitor != null) monitor.terminate();
								break;
							}
						}
					}
				}
		);

		subMenu.addCmd(InfoCommands.memoryInfo());
		subMenu.addCmd(InfoCommands.runtimeInfo());
		subMenu.addCmd(InfoCommands.systemInfo());

		subMenu.addCmd(
				new Command("Archiveinstellungen", "") {
					@Override
					public void execute() throws Exception {
						TaskScheduler ts = archMgr.getTaskScheduler();
						printlnPlain(
								"TMax:                    " + Util.relTimestrMillis(archMgr.getInQueuesMgr().getArchiveDataReceiver().getTMaxATimeForwardStep())
						);
						printlnPlain("Endueltiger Loeschtask");
						printlnPlain("  Letzter Lauf:          " + ts.getDeletePermanentlyLastRun());
						printlnPlain("  Aktuelle Einstellung:  " + ts.getDeletePermanentlyIntervals());
						printlnPlain("  Naechster Lauf:        " + ts.getDeletePermanentlyNextRun());
						printlnPlain("");
						printlnPlain("Nachforderungstask");
						printlnPlain("  Letzter Lauf:          " + ts.getRequestLastRun());
						printlnPlain("  Aktuelle Einstellung:  " + ts.getRequestIntervals());
						printlnPlain("  Naechster Lauf:        " + ts.getRequestNextRun());
						printlnPlain("");
						printlnPlain("Anzahl Bytes:            " + ArchiveQueryTask.getBufferSizeMux());
						printlnPlain("Anzahl Blocks:           " + ArchiveQueryTask.getBlockingFactorMux());
						printlnPlain("");
						printlnPlain("par. Anfr. hohe Prio.:   " + archMgr.getInQueuesMgr().getHiQueryTaskNum());
						printlnPlain("par. Anfr. mitt. Prio.:  " + archMgr.getInQueuesMgr().getMidQueryTaskNum());
						printlnPlain("par. Anfr. niedr. Prio.: " + archMgr.getInQueuesMgr().getLoQueryTaskNum());
					}
				}
		);

		subMenu.addCmd(
				new Command("Container Abschlussparameter", "") {
					@Override
					public void execute() throws Exception {
						ContainerSettings settings = ArchiveTask.getContainerSettings();
						printlnPlain("Anz. Datensaetze = " + Util.kiloBlocks(settings.stdCloseConditions.maxContAnzDS) + " Stk.");
						printlnPlain("Cont. Groesse    = " + Util.kiloBlocks(settings.stdCloseConditions.maxContSize) + " Byte");
						printlnPlain("Zeitspanne Cont. = " + Util.relTimestr(settings.stdCloseConditions.maxContTime));
						if (!settings.exceptions.isEmpty()) {
							printlnPlain(" Ausnahmen:");
							for(ContSettingsExc cSettingsEx : settings.exceptions) {
								for(Iterator<Long> it = cSettingsEx.atgSet.iterator(); it.hasNext();) {
									Long appId = it.next();
									if(it.hasNext()) {
										printlnPlain("  " + archMgr.getAtg(appId).getPidOrNameOrId() + ",");
									}
									else {
										printlnPlain("  " + archMgr.getAtg(appId).getPidOrNameOrId() + " = ");
										printlnPlain(
												"      " + Util.kiloBlocks(cSettingsEx.excCloseConditions.maxContAnzDS) + " Stk., "
														+ Util.kiloBlocks(cSettingsEx.excCloseConditions.maxContSize) + " Byte, "
														+ Util.relTimestr(cSettingsEx.excCloseConditions.maxContTime)
										);
									}
								}
							}
						}
					}
				}
		);

		subMenu.addCmd(new MemoryDumpCmd("Arbeitsspeicher-Analyse", "", archMgr));
		
		subMenu.addCmd(new LargeContainerDirDump("Übergroße Containerverzeichnisse", "", archMgr));
		
		subMenu = new CmdMenu("System Info", "");
		root.addNode(subMenu);


		subMenu.addCmd(
				new Command("Command Line Arguments", "") {
					@Override
					public void execute() throws Exception {
						List<String> ocla = archMgr.getOriginalCommandLineArguments();
						for(String arg : ocla) {
							printlnPlain("  " + arg);
						}
					}
				}
		);

		subMenu.addCmd(
				new Command("Environment", "") {
					@Override
					public void execute() throws Exception {
						Map<String, String> map = System.getenv();
						for(Map.Entry<String, String> entry : map.entrySet()) {
							printlnPlain("  " + entry.getKey() + "=" + entry.getValue());
						}
					}
				}
		);

		subMenu.addCmd(
				new Command("System Properties", "") {
					@Override
					public void execute() throws Exception {
						Properties props = System.getProperties();
						for(String s : props.stringPropertyNames()) {
							printlnPlain("  " + s + "=" + props.getProperty(s));
						}
					}
				}
		);

		PersistenceDirectoryManager persistenceDirectoryManager = am.getPersistenceManager().getPersistenceDirectoryManager();
		if (persistenceDirectoryManager instanceof TimeBasedPersistenceDirectoryManager<?> manager) {
			CmdMenu childMenu = new CmdMenu("Persistenzverzeichnisse", "Persistenzverzeichnisse einhängen und auswerfen");
			childMenu.addCmd(new ListPersistenceDirectoriesCommand<>(manager));
			childMenu.addCmd(new EjectPersistenceDirectoriesCommand<>(manager));
			childMenu.addCmd(new LoadPersistenceDirectoriesCommand<>(manager));

			root.addNode(childMenu);
		}

		root.addCmd(new CreateIndexesCommand(am));
		
		root.addCmd(
				new Command("Converter ID<->PID", "") {
					@Override
					public void execute() throws Exception {
						printlnPlain("Bitte geben Sie eine ID oder eine PID oder einen komprimierten Ordnernamen ein.");
						String eingabe = readln();
						if (eingabe == null || eingabe.isBlank()) {
							printlnPlain("Bitte geben Sie einen Wert ein.");
						}
						else {
							eingabe = eingabe.trim();
							try { // Erst gucken ob's eine Zahl ist -> ID
								long id = ShortPersistenceDirectoryLayout.decodeFast(eingabe);
								if (id == -1) {
									id = Long.parseLong(eingabe);
								}
								SystemObject obj = archMgr.getObj(id);
								if(obj != null) {
									printlnPlain("  ID: " + id + "   PID: " + obj.getPid() + "   Name: " + obj.getName());
								}
								else {
									printlnPlain("  Die ID " + eingabe + " ist nicht bekannt.");
								}
								printlnPlain("  Kodierte ID: " + ShortPersistenceDirectoryLayout.encode(id));
							}
							catch(Exception e) {  // keine Zahl
								try { // jetzt als text versuchen -> PID
									SystemObject obj = archMgr.getObj(eingabe);
									if(obj != null) {
										printlnPlain("  PID: " + eingabe + "   ID: " + obj.getId() + "   Name: " + obj.getName());
										printlnPlain("  Kodierte ID: " + ShortPersistenceDirectoryLayout.encode(obj.getId()));
									}
									else {
										printlnPlain("  Die PID " + eingabe + " ist nicht bekannt.");
									}
								}
								catch(Exception ex) {
									printlnPlain("Unbekannter Fehler: " + ex.getMessage());
								}
							}
						}
					}
				}
		);
		
		root.addCmd(
				new Command("Archivsystem stoppen", "") {
					@Override
					public void execute() throws Exception {
						println("Soll das Archivsystem beendet werden? [j|n]");
						if(readln().equals("j")) {
							Debug.getLogger().info("Archivsystem per Kommandoverbindung beendet");
							archMgr.quit();
						}
					}
				}
		);

		//"Geheime" Kommandos
		root.addCmd(            //Kommando hinzufügen, das nicht angezeigt wird
				new HiddenCommand("Gesamtanzahl der online-aktuellen Datensätze", 131008) {
					@Override
					public void execute() throws Exception {
						println(String.valueOf(archMgr.getInQueuesMgr().countDataInQueues()));
					}
				}
		);

		root.addCmd(new HiddenCommand("Archivsystem sofort terminieren", 131009) {
			@Override
			public void execute() {
				archMgr.quitError("Archivsystem wurde über Telnet sofort terminiert");
			}
		});
		cmdInt.setMenu(root);
	}

	/** Startet den Telnet Server. */
	public void start() {
		cmdInt.start();
	}

	/**
	 * Beendet das Archivsystem über den Telnet Server.
	 *
	 * @param host Host des Servers
	 * @param port Port des Servers
	 * @throws UnknownHostException Unbekannter Host.
	 * @throws IOException          Fehler beim erzeugen des Sockets.
	 */
	public static void sendQuitCmd(String host, int port) throws IOException {
		try(Socket connection = new Socket(host, port)) {
			connection.setSoTimeout(SEND_TIMEOUT * 1000);
			BufferedWriter out = new BufferedWriter(new OutputStreamWriter(connection.getOutputStream(), ISO_8859_1));
			BufferedReader in = new BufferedReader(new InputStreamReader(connection.getInputStream(), ISO_8859_1));

			readPrompt(in);
			write(out, String.valueOf(QUIT_CMD_NR));
			readPrompt(in);
			write(out, "j");
			in.readLine();
		}
	}

	/**
	 * Die Methode readPrompt list eine Eingabe vom BufferedWriter
	 *
	 * @param in von Typ BufferedReader  
	 * @throws IOException wenn ein IO-Fehler auftritt  
	 */
	private static void readPrompt(BufferedReader in) throws IOException {
		while(true) {
			if(in.readLine().endsWith(CmdInterpreter.PROMPT)) break;
		}
	}

	/**
	 * Die Methode write schreibt eine Nachricht an den Benutzer (auf den übergebenen {@link BufferedWriter}).
	 *
	 * @param out von Typ BufferedWriter  
	 * @param s von Typ String  
	 * @throws IOException IOFehler  
	 */
	private static void write(BufferedWriter out, String s) throws IOException {
		out.write(s);
		out.newLine();
		out.flush();
	}
}
