/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.mgmt.commands;

import com.google.common.collect.Range;
import com.google.common.collect.TreeRangeSet;
import de.bsvrz.ars.ars.persistence.directories.mgmt.TimeBasedPersistenceDirectoryManager;
import de.bsvrz.ars.ars.persistence.directories.mgmt.range.TimeDomain;
import de.bsvrz.ars.ars.persistence.directories.mgmt.range.TimeRange;
import de.bsvrz.sys.funclib.losb.util.cmdinterface.Command;

import java.nio.file.Path;
import java.time.LocalDate;
import java.util.Set;

/**
 * Telnet-Kommando um vorher ausgeworfene Verzeichnisse wieder zu integrieren
 *
 * @param <T>
 */
public class LoadPersistenceDirectoriesCommand<T extends TimeRange<T>> extends Command {
	private final TimeBasedPersistenceDirectoryManager<T> directoryManager;

	/**
	 * Erstellt eine neue LoadPersistenceDirectoriesCommand-Instanz.
	 * @param directoryManager Verzeichnisverwaltung
	 */
	public LoadPersistenceDirectoriesCommand(TimeBasedPersistenceDirectoryManager<T> directoryManager) {
		super("Verzeichnisse hinzufügen", "");
		this.directoryManager = directoryManager;
	}

	@Override
	public void execute() throws Exception {

		TreeRangeSet<LocalDate> availableRanges = TreeRangeSet.create();

		printlnPlain("  Folgende Zeitbereiche können integriert werden:");
		directoryManager.executeOnSubDirectories((p) -> {
			Range<LocalDate> timeRange = detectUnloadedPath(p);
			if (timeRange != null) {
				availableRanges.add(timeRange);
			}
		});

		if (availableRanges.isEmpty()) {
			printlnPlain("  Es gibt keine Verzeichnisse, die hinzugefügt werden können.");
			return;
		}
		TimeDomain<T> domain = directoryManager.getDomain();
		printlnPlain("  " + DomainUtil.format(availableRanges));

		printlnPlain("  Zeitbereich, der integriert werden soll:");
		final Set<T> timeRanges = DomainUtil.getTimeRangesFromUser(this, domain);
		if (timeRanges == null) return;
		for (T timeRange : timeRanges) {
			boolean b = directoryManager.addPersistenceDirectory(timeRange);
			if (b) {
				printlnPlain("  " + domain.getPath(timeRange) + ": Erfolgreich hinzugefügt.");
			} else {
				printlnPlain("  " + domain.getPath(timeRange) + ": Hinzufügen aktuell nicht möglich.");
			}
		}
	}

	/**
	 * Gibt den Pfad aus, wenn er aktuell nicht ins Archivsystem eingebunden ist
	 *
	 * @param path Pfad
	 * @return Bereich von Tagen
	 */
	private Range<LocalDate> detectUnloadedPath(Path path) {
		T timeRange = directoryManager.getId(path);
		if (timeRange == null) {
			return null;
		}
		if (directoryManager.getPersistenceDirectory(timeRange) == null) {
			return timeRange.getDayRange();
		}
		return null;
	}

}
