/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.datatree;

import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SynchronizationManager;
import de.bsvrz.ars.ars.persistence.IdDataIdentification;
import de.bsvrz.dav.daf.main.Dataset;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.dataIdentificationSettings.DataIdentification;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Klasse, die zu Datenidentitifikationen {@link DataIdentNode}s speichert (enthält z. B. die Parametrierung der Datenidentifikationen).
 * <p> 
 * Enthält außerdem ein {@link SynchronizationManager}, mit dem auf die Datenidentifikationen synchronisiert werden kann.
 * 
 * @author Kappich Systemberatung
 */
public final class DataIdentTree implements Iterable<DataIdentNode> {

	private final Map<IdDataIdentification, DataIdentNode> _nodeMap = new ConcurrentHashMap<>();

	/**
	 * Liefert den DataIdentNode zur angegebenen Daten-Identifikation.
	 *
	 * @param did Datenidentifikation
	 *
	 * @return DataIdentNode
	 */
	public DataIdentNode get(DataIdentification did) {
		return get(new IdDataIdentification(did));
	}

	/**
	 * Liefert den DataIdentNode zur angegebenen Daten-Identifikation.
	 *
	 * @param rd Datensatz vom Datenverteiler
	 *
	 * @return DataIdentNode zur angegebenen Daten-Identifikation.
	 */
	public DataIdentNode get(Dataset rd) {
		return get(new IdDataIdentification(rd));
	}


	/**
	 * Liefert den DataIdentNode zur angegebenen Daten-Identifikation.
	 *
	 * @param idDataIdentification Datenidentifikation (enthält objID, atgID, aspID, sv)
	 *
	 * @return DataIdentNode zur angegebenen Daten-Identifikation.
	 */
	public DataIdentNode get(IdDataIdentification idDataIdentification) {
		return _nodeMap.computeIfAbsent(idDataIdentification, DataIdentNode::new);
	}


	/**
	 * Liefert den DataIdentNode zur angegebenen Daten-Identifikation.
	 *
	 * @param idDataIdentification Datenidentifikation (enthält objID, atgID, aspID, sv)
	 *
	 * @return DataIdentNode zur angegebenen Daten-Identifikation.
	 */
	@Nullable
	public DataIdentNode getIfPresent(IdDataIdentification idDataIdentification) {
		return _nodeMap.get(idDataIdentification);
	}

	/**
	 * Loescht den DataIdentNode zur angegebenen Daten-Identifikation. Nach dem Loeschen des DataIdentNodes leer gewordene ObjNodes, AtgNodes oder AspNodes
	 * werden ebenfalls aus dem Baum entfernt.
	 *
	 * @param did Datenidentifikation
	 */
	public synchronized void deleteDataIdentNode(IdDataIdentification did) {
		_nodeMap.remove(did);
	}

	/**
	 * Liefert alle Objekte, die im rootNode gespeichert sind. ObjekteIds werden mit Hilfe des {@link DataModel} aufgelöst.
	 *
	 * @param model das DataModel
	 *
	 * @return eine Liste von Systemobjekten
	 *
 	 */
	public List<SystemObject> getObjects(DataModel model) {
		return model.getObjects(_nodeMap.keySet().stream().mapToLong(IdDataIdentification::getObjectId).distinct().toArray());
	}

	/**
	 * Liefert alle Attributgruppen zu einem Objekt, die im Baum gespeichert sind.
	 *
	 * @param objId Objekt-ID
	 * @param model Datenmodell
	 *
	 * @return ein Array von Attributgruppen
	 *
 	 */
	public List<SystemObject> getAtgs(long objId, DataModel model) {
		return model.getObjects(_nodeMap.keySet().stream().filter(it -> it.getObjectId() == objId).mapToLong(IdDataIdentification::getAtgId).distinct().toArray());
	}

	/**
	 * Liefert alle möglichen Aspekte zu einem Objekt und einer zugehörigen Attributgruppe.
	 *
	 * @param objId die Id des Objekts
	 * @param atgId die Id der Attributgruppe
	 * @param model das DataModel
	 *
	 * @return ein Array von Aspekten
	 */
	public List<SystemObject> getAsps(long objId, long atgId, DataModel model) {
		return model.getObjects(
			_nodeMap.keySet().stream().filter(it -> it.getObjectId() == objId && it.getAtgId() == atgId).mapToLong(IdDataIdentification::getAspectId)
				.distinct().toArray());
	}

	/**
	 * Liefert alle möglichen Simulationsvarianten zu einem Objekt mit Attributgruppe und Aspekt.
	 *
	 * @param objId die Id des Objekts
	 * @param atgId die Id der Attributgruppe
	 * @param aspId die Id des Aspekts
	 *
	 * @return ein Array von möglichen Simulationsvarianten
	 */
	public int[] getSimVars(long objId, long atgId, long aspId) {
		return _nodeMap.keySet().stream().filter(it -> it.getObjectId() == objId && it.getAtgId() == atgId && it.getAspectId() == aspId).mapToInt(IdDataIdentification::getSimVariant).sorted().toArray();
	}

	/**
	 * Liefert die Anzahl der DataIdentNodes im Baum zurück.
	 *
	 * @return Anzahl DataIdentNodes
	 */
	public int size() {
		return _nodeMap.size();
	}

	@NotNull
	@Override
	public Iterator<DataIdentNode> iterator() {
		return _nodeMap.values().iterator();
	}
}
