/*
 * Copyright 2019 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.datatree;

import de.bsvrz.ars.ars.persistence.ContainerDirectory;
import de.bsvrz.ars.ars.persistence.IdContainerFileDir;
import de.bsvrz.ars.ars.persistence.PersistenceManager;
import de.bsvrz.ars.ars.persistence.directories.PersistenceDirectory;

import java.nio.file.Path;
import java.util.Objects;

/**
 * Hashmap-Schlüssel um einen Index zu Referenzieren. Siehe {@link IndexTree}.
 *
 * @author Kappich Systemberatung
 */
public final class IndexId {
	
	private final IdContainerFileDir _containerFileDir;
	
	private final IndexImpl _indexClass;

	/**
	 * Erstellt eine neue IndexId
	 *
	 * @param containerFileDir Ort des Indexes (Containerverzeichnis)
	 * @param indexClass       Art des Indexes
	 */
	public IndexId(final IdContainerFileDir containerFileDir, final IndexImpl indexClass) {
		_containerFileDir = Objects.requireNonNull(containerFileDir);
		_indexClass = Objects.requireNonNull(indexClass);
	}

	/**
	 * Erstellt eine neue IndexId
	 *
	 * @param containerFileDir Ort des Indexes (Containerverzeichnis)
	 * @param indexClass       Art des Indexes
	 */
	public IndexId(final ContainerDirectory containerFileDir, final IndexImpl indexClass) {
		_containerFileDir = IdContainerFileDir.of(containerFileDir);
		_indexClass = Objects.requireNonNull(indexClass);
	}

	/** 
	 * Gibt den Ort des Indexes im Dateisystem zurück
	 * @param persistenceDirectory {@link PersistenceManager}
	 * @return Indexdatei als {@link Path}
	 */
	public Path toFile(PersistenceDirectory persistenceDirectory) {
		return persistenceDirectory.getPath(_containerFileDir).resolve(_indexClass.getFileName());
	}

	/** 
	 * Gibt den Ort zurück
	 * @return den Ort
	 */
	public IdContainerFileDir getContainerFileDir() {
		return _containerFileDir;
	}

	/** 
	 * Gibt die Art des Index zurück
	 * @return die Art des Index
	 */
	public IndexImpl getIndexClass() {
		return _indexClass;
	}

	@Override
	public boolean equals(final Object o) {
		if(this == o) return true;
		if(o == null || getClass() != o.getClass()) return false;

		final IndexId indexId = (IndexId) o;

		if(!_containerFileDir.equals(indexId._containerFileDir)) return false;
		return _indexClass == indexId._indexClass;
	}

	@Override
	public int hashCode() {
		int result = _containerFileDir.hashCode();
		result = 31 * result + _indexClass.hashCode();
		return result;
	}

	@Override
	public String toString() {
		return "IndexId{" +
				"_containerFileDir=" + _containerFileDir +
				", _indexClass=" + _indexClass +
				'}';
	}
}
