/*
 * Copyright 2019-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.datatree.synchronization;

import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.BooleanSupplier;

/**
 * Schlüssel zur Synchronisierung auf ein Objekt (im Archivsystem in der Regel eine Datenidentifikation).
 *
 * @author Kappich Systemberatung
 * @param <T> Typ, der gelockt wird (in der Regel {@link de.bsvrz.ars.ars.persistence.IdDataIdentification}). 
 */
public interface SyncKey<T> extends AutoCloseable {
	
	/** 
	 * Gibt das gelockte Element zurück.
	 * @return das gelockte Element
	 * @see SynchronizationManager
	 */
	T getElement();

	/** 
	 * Gibt den anfordernden Thread zurück. Ein SyncKey ist nur für diesen Thread gültig.
	 * @return den anfordernden Thread
	 */
	Thread getThread();

	/**
	 * Gibt die Synchronisierung wieder frei. Diese Methode kann gefahrlos mehrmals aufgerufen werden.
	 */
	@Override
	void close();

	/** 
	 * Gibt zurück, ob der Schlüssel für den aktuellen Thread gültig ist.
	 * False, falls er mit {@link #close()} geschlossen wurde oder der Methodenaufruf von einem falschen Thread erfolgt.
	 * @return true falls gültig, sonst false 
	 */
	boolean isValid();

	/**
	 * Gibt einen {@link BooleanSupplier} zurück, mit dem gefragt werden kann, ob ein SyncKey noch nicht geschlossen wurde, und das ohne eine harte Referenz auf den SyncKey zu halten.
	 * <p>Das zurückgegebene Objekt hält nur eine Referenz auf das interne {@link AtomicBoolean} des SyncKeys</p>
	 * @return Supplier
	 */
	BooleanSupplier notClosed();
}
