/*
 * Copyright 2019-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.datatree.synchronization;

import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.time.Duration;
import java.util.Collection;

/**
 * Exception, die geworfen wird, wenn die Synchronisation auf eine Datenidentifikation fehlgeschlagen ist
 */
public class SynchronizationFailedException extends Exception {

	/**
	 * Exception-Konstruktor
	 *
	 * @param timeout    Abgelaufener Timeout (kann null sein, falls kein Timeout abgelaufen ist)
	 * @param element    Element, auf das Synchronisiert wurde (z. B. Datenidentifikation)
	 * @param otherLocks Andere gehaltene Locks, die vermutlich die Synchronisierung verhinderten
	 * @param parent     Verursachende Exception (i. d. R. eine {@link InterruptedException})
	 */
	public SynchronizationFailedException(@Nullable Duration timeout, Object element, Collection<? extends SyncKey<?>> otherLocks, @Nullable Throwable parent) {
		super(constructMessage(timeout, element, otherLocks), parent);
		for (SyncKey<?> otherLock : otherLocks) {
			ThreadDump tmp = new ThreadDump(otherLock.getThread());
			tmp.setStackTrace(otherLock.getThread().getStackTrace());
			addSuppressed(tmp);
		}
	}

	private static String constructMessage(@Nullable Duration timeout, Object element, Collection<? extends SyncKey<?>> otherLocks) {
		Thread myThread = Thread.currentThread();
		Thread otherThread = otherLocks.stream().limit(1).map(SyncKey::getThread).findAny().orElse(null);
		StringBuilder result = new StringBuilder();
		result.append("Thread '")
			.append(myThread)
			.append("' konnte nicht auf '")
			.append(element)
			.append("' synchronisieren. ");
		if(timeout != null) {
			result.append("Timeout: ").append(formatDuration(timeout)).append(". ");
		}
		if(otherThread != null) {
			result.append("Ressource wird durch Thread '")
				.append(otherThread)
				.append("' belegt.");
		}
		return result.toString();
	}

	/**
	 * Formatiert eine Zeitdauer.
	 * @param runtime Dauer
	 * @return Formatierte Ausgabe
	 */
	private static String formatDuration(final Duration runtime) {
		long days = runtime.toDays();
		long hours = runtime.toHours() % 24;
		long minutes = runtime.toMinutes() % 60;
		long seconds = runtime.getSeconds() % 60;
		if(hours == 0) return String.format("%02d:%02d", minutes, seconds);
		if(days == 0) return String.format("%02d:%02d:%02d", hours, minutes, seconds);
		if(days == 1) return String.format("1 Tag, %02d:%02d:%02d", hours, minutes, seconds);

		return String.format("%d Tage, %02d:%02d:%02d", days, hours, minutes, seconds);
	}

	private static class ThreadDump extends Throwable {
		public ThreadDump(Thread thread) {
			super(thread.toString());
		}
	}
}
