/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.simulation;

import de.bsvrz.ars.ars.mgmt.ArchiveManager;
import de.bsvrz.dav.daf.main.*;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.losb.datk.PidSimulation;
import de.bsvrz.sys.funclib.losb.exceptions.FailureException;
import de.bsvrz.sys.funclib.losb.kernsoftware.ConnectionManager;

import java.util.concurrent.LinkedBlockingQueue;

/**
 * Deterministischer endlicher Automat für die Simulation. Nicht erlaubte Transitionen werden ignoriert. Treten Fehler auf, so werden diese geloggt.
 *
 * @author beck et al. projects GmbH
 * @author Martin Hilgers
 * @version $Revision$ / $Date$ / ($Author$)
 */
public sealed class SimulationAutomaton implements ClientReceiverInterface, SimulationManager.Transition permits OfflineSimulation, OnlineSimulation {
	/////////////////////////////
	/// Zustand des Automaten ///
	/////////////////////////////

	/**
	 * Undefinierter Zustand / Undefinierte Simulationsvariante
	 */
	public static final int UNDEFINED = -1;

	/**
	 * Automat wurde erzeugt, ist aber noch nicht aktiv.
	 */
	private static final int CREATED = -2;

	/**
	 * Automat wurde erzeugt und wurde aktiviert - d.h. Anmeldung auf Simulationssteuerung ist erfolgt.
	 */
	private static final int SUBSCRIBED = -3;

	/*
	 * !!!WICHTIG!!!
	 * Die folgenden Werte müssen mit denen aus dem Datenkatalog übereinstimmen!
	 */
	/**
	 * Neu - Zustand.
	 */
	protected static final int NEW = 0;

	/**
	 * Vorstart - Zustand.
	 */
	protected static final int INIT = 1;

	/**
	 * Start - Zustand.
	 */
	protected static final int START = 2;

	/**
	 * Pause - Zustand.
	 */
	protected static final int PAUSE = 5;

	/**
	 * Stop - Zustand.
	 */
	protected static final int STOP = 3;

	/**
	 * Gelöscht - Zustand.
	 */
	protected static final int ERASED = 4;

	/**
	 * Zustand des Automaten.
	 */
	private int state = CREATED;

	/////////////////////////////
	/// Eigenschaften des DFA ///
	/////////////////////////////
	/**
	 * Debug Ausgabe.
	 */
	private final Debug debug = Debug.getLogger();

	/**
	 * Simulationsobjekt
	 */
	private final SystemObject simulation;

	/**
	 * Simulationsstreckenobjekt.
	 */
	private final SystemObject simulationRange;

	/**
	 * Queue mit den Transitionen, die ausgeführt werden sollen.
	 */
	private final LinkedBlockingQueue<Integer> transitions;

	/**
	 * Simulationsvariante der Simulation.
	 */
	private final short simVar;

	/**
	 * Zeigt an, dass der letzte erhaltene Datensatz für das SImulationsobjekt ein 'keine Quelle' Datensatz war.
	 */
	private boolean noSource;

	/**
	 * Steuerungsinformationen, auf die sich die Simulation beim Dav anmelden soll.
	 */
	private final DataDescription simulationControl;

	/**
	 * Simulationsverwaltung
	 */
	private final SimulationManager simManager;

	private StateSender _stateSender;


	/**
	 * Erzeugt einen neuen Automaten für eine Simulation. Meldet den Automaten als Empfänger für das Simuationsobjekt an.
	 *
	 * @param simManager        Simulationsverwaltung
	 * @param simulation        Simulationsobjekt.
	 * @param simulationControl Steuerungsinformationen, auf die sich die Simulation beim Dav anmelden soll.
	 */
	public SimulationAutomaton(SimulationManager simManager, SystemObject simulation, DataDescription simulationControl) {
		if (simulationControl.getAspect() == null || simulationControl.getAttributeGroup() == null) {
			throw new IllegalArgumentException("Ungültige DataDescription: " + simulationControl);
		} else if (simulation == null) throw new IllegalArgumentException("Simulationsobjekt ist null.");
		this.simManager = simManager;
		this.simulation = simulation;
		this.simulationControl = simulationControl;
		transitions = new LinkedBlockingQueue<>();

		//Daten der Simulation abrufen
		AttributeGroup atgProperties = simManager.getDav().getDataModel().getAttributeGroup(PidSimulation.atgProperties);
		Data properties = simulation.getConfigurationData(atgProperties);
		this.simulationRange = properties.getReferenceValue(PidSimulation.attSimulationRange).getSystemObject();
		this.simVar = properties.getUnscaledValue(PidSimulation.attSimVar).shortValue();
		if (simVar < 0 || simVar > 999)
			throw new IllegalStateException("Simulationsvariante außerhalb des Wertebereiches: " + properties);
	}


	/**
	 * Meldet das Objekt als Empfänger für die Simulationssteuerung an.
	 */
	private void subscribeSimulation() {
		try {
			ConnectionManager.subscribeReceiver(simManager.getDav(), this, simulation, simulationControl, ReceiveOptions.normal(), ReceiverRole.receiver());
		} catch (FailureException e) {
			debug.error("", e);
		}
	}


	/**
	 * Meldet das Objekt vom Empfang für Simulationssteuerungsdaten ab.
	 */
	private void unsubscribeSimulation() {
		ConnectionManager.unsubscribeReceiver(simManager.getDav(), this, simulation, simulationControl);
	}


	/**
	 * Meldet das Objekt als Empfänger für die Simulationsstrecke an.
	 */
	private void subscribeSimulationRange() {
		try {
			DataDescription dd = new DataDescription(
					simManager.getDav().getDataModel().getAttributeGroup(PidSimulation.atgArchiveData),
					simManager.getDav().getDataModel().getAspect(PidSimulation.aspTarget)
			);
			ConnectionManager.subscribeReceiver(simManager.getDav(), this, simulationRange, dd, ReceiveOptions.normal(), ReceiverRole.receiver());
		} catch (FailureException e) {
			debug.error("", e);
		}
	}


	/**
	 * Meldet das Objekt vom Empfang für Simulationsstreckendaten ab.
	 */
	private void unsubscribeSimulationRange() {
		DataDescription dd = new DataDescription(
				simManager.getDav().getDataModel().getAttributeGroup(PidSimulation.atgArchiveData),
				simManager.getDav().getDataModel().getAspect(PidSimulation.aspTarget)
		);
		ConnectionManager.unsubscribeReceiver(simManager.getDav(), this, simulationRange, dd);
	}


	/**
	 * Transition in den neuen Zustand. Es wird überprüft, ob die Transition möglich ist.
	 *
	 * @return {@code true} falls der Automat in den neuen Zustand überführt werden konnte. {@code false} sonst.
	 */
	public synchronized boolean transition() {
		if (state == CREATED)            //der erste Aufruf von Transition führt zur Initialisierung
		{
			state = SUBSCRIBED;
			subscribeSimulation();
			subscribeSimulationState();
			return true;
		}

		Integer newState = transitions.poll();
		if (newState == null) {
			return false;
		}

		/*
		 * Erlaubte Transitionen
		 * UNDEFINED -> NEW
		 * NEW -> INIT
		 * INIT -> START
		 * START -> PAUSE | STOP
		 * PAUSE -> PAUSE | START
		 * STOP -> INIT | ERASED
		 */
		if (
				(state == SUBSCRIBED && newState == NEW) ||        //Nur gültige Transitionen ausführen
						(state == NEW && newState == INIT) ||
						(state == INIT && newState == START) ||
						(state == START && (newState == STOP || newState == PAUSE)) ||
						(state == PAUSE && (newState == START || newState == PAUSE || newState == STOP)) ||
						(state == STOP && (newState == INIT || newState == ERASED))) {
			debug.fine("Zustandsübergang der Simulation " + simVar + ": " + state + " -> " + newState);
			state = newState;
			switch (state) {
				case NEW -> enterNewState();
				case INIT -> enterInitState();
				case START -> enterStartState();
				case STOP -> enterStopState();
				case ERASED -> enterErasedState();
			}
			return true;
		} else if (state == SUBSCRIBED) {
			if (newState == INIT) {
				debug.fine("Zustandsübergang der Simulation " + simVar + ": " + state + " -> " + newState);
				//Archivsystem wurde erst hochgefahren, nachdem die Simulation initialisiert wurde → Simulation kann trotzdem durchgeführt werden
				enterNewState();
				state = INIT;
				enterInitState();
				return true;
			} else {
				debug.warning("Nicht erlaubter Zustandsübergang der Simulation " + simVar + ": " + state + " -> " + newState);
				//Simulation ist bereits zu weit fortgeschritten
				sendSimulationState(false);
				return false;
			}
		} else {
			//alle übrigen Zustandsübergänge werden ignoriert
			debug.warning("Nicht erlaubter Zustandsübergang der Simulation " + simVar + ": " + state + " -> " + newState);
			return false;
		}
	}


	/**
	 * Zustand 'Neu' wird erreicht. Meldet den Empfang von Steuerungsdaten an. Sendet, dass das Archivsystem nicht für die Durchführung der Simulation zur
	 * Verfügung steht.
	 */
	protected void enterNewState() {
		sendSimulationState(false);
	}

	/*
	 * Ablauf für die Methoden enterInitState, continueInitStateGetParams, continueInitStateSubscribe & finishInitState:
	 * 1. Aufruf von enterInitState 											(Wirf Löschtask an)
	 * 2. Löschtask Callback -> dataDeleted ruft continueInitStateGetParams auf
	 * 3. continueInitStateGetParams											(meldet sich auf Dav Daten an um zu erfahren, welche Daten zu archivieren sind)
	 *  -> Dav Callback -> update()
	 *  4. update ruft continueInitStateSubscribe() auf. 						(Dort werden die Simulationsdaten, auf die sich das Archivsystem anmelden soll, ausgewertet)
	 *  5. ArchivConfig Callback -> finishInitState 							(Das Archivsystem hat sich auf die Simulationsdaten angemeldet.)
	 */

	/**
	 * Zustand 'Vorstart' wird erreicht. - Löschen der archivierten Simulationsdaten. Ist diese Aufgabe abgeschlossen, wird die Callback Methode{@link
	 * #dataDeleted(boolean)} vom Löschtask aufgerufen.
	 */
	protected void enterInitState() {
		//Löschen der Simulationsdaten
		SimulationResultData ard = new SimulationResultData(this);
		// Altes Persistenzverzeichnis ggf. löschen
		simManager.insertDeleteQuery(ard);
	}


	/**
	 * - Anmelden auf Information, welche Daten archiviert werden sollen. Geht diese Information ein, wird {@link #continueInitStateSubscribe(ResultData)} durch
	 * {@link #update(ResultData[])} aufgerufen. Diese Methode wird von {@link #dataDeleted(boolean)} aufgerufen, wenn der Zustand der Simulation {@link #INIT}
	 * ist.
	 */
	protected void continueInitStateGetParams() {
		//Daten herausfinden, die Archiviert werden sollen.
		subscribeSimulationRange();
		// Neues leeres Persistenzverzeichnis anlegen
		ArchiveManager archiveManager = simManager.getArchMgr();
		archiveManager.getPersistenceManager().getPersistenceDirectoryManager().createSimulationDirectory(simVar);
	}


	/**
	 * Fortsetzung Zustand 'Vorstart'. - Meldet das Archivsystem auf die Simulationsdaten an. Wird durch {@link #update(ResultData[])} aufgerufen, sobald die zu
	 * archivierenden Daten bekannt sind.
	 *
	 * @param resultData Information, auf welche Daten sich das Archivsystem anmelden soll.
	 */
	protected void continueInitStateSubscribe(ResultData resultData) {
		SimulationResultData ard = new SimulationResultData(this, resultData);
		simManager.getArchMgr().getInQueuesMgr().insertSimVarParam(ard);
	}


	/**
	 * Wird vom ArchivConfig Task aufgerufen, nachdem das Archivsystem sich auf die Simulationsdaten angemeldet hat - Sendet die 'Bereit' Meldung, wenn Daten
	 * gelöscht, falls der Automat im Zustand Vorstart und die Anmeldung auf die zu archivierenden Simulationsdaten erfolgreich verlaufen ist.
	 *
	 * @param successful Zeigt an, ob der Auftrag erfolgreich durchgeführt wurde.
	 */
	protected void finishInitState(boolean successful) {
		if (!successful) state = UNDEFINED;
		sendSimulationState(successful);
	}


	/**
	 * Zustand 'Start' wird erreicht. Meldet die Simulation vom Empfang von Anmeldeinformationen
	 */
	protected void enterStartState() {
		unsubscribeSimulationRange();
	}


	/**
	 * Zustand 'Stop' wird erreicht. - 'Ende'-Meldung senden - Abmeldung vom Datenempfang
	 */
	protected void enterStopState() {
		sendSimulationState(false);
		simManager.getArchMgr().getArchivConfig().unsubscribeSimVarData(this.getSimVar());
	}


	/**
	 * Zustand 'Gelöscht' wird erreicht. - Löschen der Daten
	 */
	protected void enterErasedState() {
		SimulationResultData srd = new SimulationResultData(this);
		simManager.insertDeleteQuery(srd);
	}

	/**
	 * Wird aufgerufen, nachdem alle Daten gelöscht wurden. Wird durch {@link #dataDeleted(boolean)} aufgerufen, wenn der Zustand {@link #ERASED} ist. -
	 * Benachrichtigung senden, dass Simulationsobjekt gelsöcht werden kann.
	 */
	protected void finishErasedState() {
		sendSimulationState(false);
		unsubscribeSimulationState();
	}


	private void subscribeSimulationState() {
		final ClientDavInterface daf = simManager.getDav();
		DataModel model = daf.getDataModel();
		AttributeGroup atg = model.getAttributeGroup(PidSimulation.atgArchiveState);
		DataDescription dd = new DataDescription(atg, model.getAspect(PidSimulation.aspState));
		try {
			_stateSender = new StateSender();
			daf.subscribeSender(_stateSender, simulation, dd, SenderRole.source());
		} catch (OneSubscriptionPerSendData e) {
			debug.warning("Mehrfachanmeldung auf Simulationszustand, Simulation: " + simulation, e);
		}
	}

	private void unsubscribeSimulationState() {
		final StateSender stateSender = _stateSender;
		if (stateSender != null) {
			_stateSender = null;
			final ClientDavInterface daf = simManager.getDav();
			DataModel model = daf.getDataModel();
			AttributeGroup atg = model.getAttributeGroup(PidSimulation.atgArchiveState);
			DataDescription dd = new DataDescription(atg, model.getAspect(PidSimulation.aspState));
			try {
				daf.unsubscribeSender(stateSender, simulation, dd);
			} catch (Exception e) {
				debug.warning("Abmeldung auf Simulationszustand nicht möglich, Simulation: " + simulation, e);
			}
		}
	}

	private static class StateSender implements ClientSenderInterface {

		@Override
		public void dataRequest(SystemObject object, DataDescription dataDescription, byte state) {
		}

		@Override
		public boolean isRequestSupported(SystemObject object, DataDescription dataDescription) {
			return false;
		}
	}

	/**
	 * Legt den Simulationsstatus auf den Datenverteiler. Blockiert, bis das Senden abgeschlossen wurde, maximal jedoch 10 Sekunden.
	 *
	 * @param archiveReady Ist das Archiv für die Simulation bereit?
	 * @return {@code true} falls das Senden erfolgreich durchgeführt wurde.
	 */
	protected boolean sendSimulationState(boolean archiveReady) {
		if (_stateSender == null) {
			debug.warning("Versand des Simulationszustands nicht möglich, weil nicht angemeldet, Simulation: " + simulation);
			return false;
		}
		final ClientDavInterface daf = simManager.getDav();
		DataModel model = daf.getDataModel();
		AttributeGroup atg = model.getAttributeGroup(PidSimulation.atgArchiveState);
		DataDescription dd = new DataDescription(atg, model.getAspect(PidSimulation.aspState));
		Data data = daf.createData(atg);

		data.getUnscaledValue(PidSimulation.attArchiveReady).set((archiveReady) ? 1 : 0);
		try {
			daf.sendData(new ResultData(simulation, dd, System.currentTimeMillis(), data));
			return true;
		} catch (SendSubscriptionNotConfirmed e) {
			debug.warning("Versand des Simulationszustands nicht möglich, Simulation: " + simulation, e);
			return false;
		}
	}

	/**
	 * Stoppt die Simulationsanmeldung. Wartet nicht.
	 */
	protected void unsubscribe() {
		if (_stateSender != null) {
			sendSimulationState(false);
			unsubscribeSimulationState();
		}
	}


	/**
	 * Wird aufgerufen, nachdem die Daten der Simulation gelöscht wurden.
	 *
	 * @param successful Zeigt an, ob die Daten gelöscht werden konnten.
	 */
	public void dataDeleted(boolean successful) {
		if (!successful) debug.warning("Es konnten nicht alle Simulationsdaten gelöscht werden.");
		if (state == INIT) {
			continueInitStateGetParams();
		} else if (state == ERASED) finishErasedState();
	}


	/**
	 * Beendet den Automaten. Meldet den Empfang von Daten ab.
	 */
	public void terminate() {
		unsubscribeSimulation();
		//ggf Meldung absetzten
		if (state != ERASED) unsubscribe();
		state = UNDEFINED;
	}


	/**
	 * Callback Methode die vom Dav aufgerufen wird, wenn neue Daten zum Simulationsobjekt eingehen. Benachrichtigt Simulationsverwaltung, wenn eine
	 * Zustandsänderung der Simulation eingegangen ist.
	 *
	 * @see SimulationManager#addTransition(SimulationManager.Transition)
	 */
	@Override
	public void update(ResultData[] results) {
		for (ResultData result : results) {
			try {
				//Überprüfen, zu welchem Objekt die Daten gehören
				if (result.getObject().equals(simulation)) {
					if (result.getDataState().equals(DataState.NO_SOURCE)) {
						if (state != SUBSCRIBED) {
							//'keine Quelle' Datensatz -> auf weiteren Steuerbefehl warten
							noSource = true;
							simManager.addWakeUp(this);
						} else {
							//Archivsystem ist hochgefahren und hat nur den keine Quelle Datensatz zur Verfügung:
							//Meldung Senden, dass das Archivsystem nicht für die Simulation nicht zur Verfügung steht!
							sendSimulationState(false);
						}
					} else if (result.hasData()) {
						//Es handelt sich um eine Zustandsänderung
						if (noSource) {
							noSource = false;
							simManager.cancelWakeUp(this);
						}
						Data data = result.getData();
						int i = (int) data.getUnscaledValue(PidSimulation.attSimulationState).getState().getValue();
						transitions.put(i);
						simManager.addTransition(this);
					}
				} else if (result.getObject().equals(simulationRange) && result.hasData()) {
					//Es handelt sich um Informationen zur Parametrierung
					continueInitStateSubscribe(result);
				}
			} catch (Exception npe) {
				Debug.getLogger().warning("Fehler beim Auswerten der eingehenden Daten. Daten werden ignoriert! " + result, npe);
			}
		}
	}


	/**
	 * Benachrichtigt die Simulation, dass die Wartezeit abgelaufen ist.
	 *
	 * @see SimulationManager#addWakeUp(SimulationAutomaton)
	 */
	public void wakeUp() {
		if (!sendSimulationState(false))
			debug.fine("Kontakt zur Smulationssteuerung verloren. Senden einer Statusmeldung nicht möglich.");
		simManager.getArchMgr().getArchivConfig().unsubscribeSimVarData(this.getSimVar());
	}


	/**
	 * Liefert die Simulationsvariante der Simulation.
	 *
	 * @return Simulationsvariante. {@link #UNDEFINED} falls noch nicht initialisiert.
	 */
	public short getSimVar() {
		return simVar;
	}

	@Override
	public String toString() {
		return "SimulationAutomaton{" +
				"state=" + state +
				", simulation=" + simulation +
				", simVar=" + simVar +
				'}';
	}
}
