/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2019 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.simulation;

import de.bsvrz.ars.ars.mgmt.ArchiveManager;
import de.bsvrz.ars.ars.mgmt.tasks.RepeatingTask;
import de.bsvrz.ars.ars.persistence.util.SignalingQueue;
import de.bsvrz.ars.ars.persistence.util.TerminatedException;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.config.MutableSet;
import de.bsvrz.dav.daf.main.config.MutableSetChangeListener;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.losb.datk.PidSimulation;

import java.time.Duration;
import java.util.*;

/**
 * Diese Klasse reagiert auf Änderungen der Menge der Simulationsobjekte. Pro Simulationsobjekt wird eine Instanz von {@link
 * de.bsvrz.ars.ars.mgmt.simulation.SimulationAutomaton} erzeugt.<br>
 * <p>
 * Wird das Archivsystem während einer Simulation heruntergefahren wird, so werden die bisher abgelegten Simulationsdaten nicht gelöscht. Dies geschieht dann
 * entweder auf Anfrage oder wenn die Simulation erneut gestartet und in den Zustand 'Vorstart' gebracht wird.<br>
 *
 * @author beck et al. projects GmbH
 * @author Martin Hilgers
 * @version $Revision$ / $Date$ / ($Author$)
 */
public class SimulationManager extends RepeatingTask implements MutableSetChangeListener {

	/**
	 * Interface für Elemente in der {@link #transitions}-Warteschlange. Es handelt sich entweder um
	 * {@link SimulationAutomaton}-Objekte auf denen dann die {@link SimulationAutomaton#transition()}-methode
	 * aufgerufen wird, oder um {@link WakeUpNow}-Objekte, nur die ein Aufwecken der Simulationen bewirken
	 * ({@link WakeUpManager#wakeUp()}).
	 */
	sealed interface Transition permits SimulationAutomaton, WakeUpNow {
	}

	record WakeUpNow() implements Transition {
	}

	/**
	 * Objektmenge Simulationsobjekte.
	 */
	private MutableSet mutableSet;

	/** Abbildung SystemObjekt Simulation -> Java Objekt Simulation */
	private final HashMap<SystemObject, SimulationAutomaton> simulations;

	/** FIFO Puffer mit den Simulationen die Transitionen durchführen müssen. */
	private final SignalingQueue<Transition> transitions;

	/** Verwalten der Weckzeiten */
	private final WakeUpManager wakeUp;

	/** Zeitraum in ms in dem eine erneute Steuerungsinformation für eine Simulation eingehen muss. */
	private long wakeUpTime = 120 * 1000;

	/** Debug Ausgabe. */
	private static final Debug logger = Debug.getLogger();

	/**
	 * Meldet Objekt als Empfänger für Änderungen an {@link de.bsvrz.sys.funclib.losb.datk.PidSimulation#objectSet} an.
	 *
	 * @param archMgr Archivmanager
	 */
	public SimulationManager(ArchiveManager archMgr) {
		super(archMgr);
		simulations = new HashMap<>();
		transitions = new SignalingQueue<>();
		wakeUp = new WakeUpManager();
	}
	
	/** @see java.lang.Thread#start() */
	@Override
	public void start() {
		mutableSet = getArchMgr().getArchiveObject().getMutableSet(
				PidSimulation.objectSet
		);
		if(mutableSet == null) {
			logger.warning("Fehler bei der Abfrage von Konfigurationsdaten. Mutable Set \"Simulationen\" steht nicht zur Verfügung. Simulation deaktiviert.");
		}
		else {
			mutableSet.addChangeListener(this);
			// -> Seit KSv3.0 wird durch addChangeListener die Methode update(MutableSet,...) nicht mehr automatisch bei der Anmeldung aufgerufen
			// -> Manueller Aufruf notwenig!
			//Bis KSv2.0 wurde die update Methode mit mit allen im Set enthaltenen Objekten aufgerufen.
			update(mutableSet, mutableSet.getElements().toArray(new SystemObject[0]), new SystemObject[0]);
			super.start();
		}
	}

	@Override
	public void update(MutableSet set, SystemObject[] addedObjects, SystemObject[] removedObjects) {
		try {
			if(set != null && set.equals(mutableSet)) {
				if(simulations.size() + addedObjects.length - removedObjects.length > 1000) {
					logger.warning(
							"Zu viele Simulationen gestartet! Es werden keine weiteren Simulationen gestartet, bis die Anzahl der Simulationen unter 1000 sinkt."
					);
				}
				else {
					//für jede Simulation ein neues Java-Simulationsobjekt anlegen.
					for(SystemObject simulation : addedObjects) {
						//Falls bereits ein Eintrag in der Hashtabelle existiert ist das ein Systemfehler!
						if(simulations.get(simulation) != null) {
							logger.error("Es existiert bereits eine eingetragene Simulation für " + simulation + ": " + simulations.get(simulation));
						}
						else {
							SimulationAutomaton dfa = null;
							//Typ der Simulation feststellen
							switch (simulation.getType().getPid()) {
								case PidSimulation.typeOnlineSimulation -> dfa = new OnlineSimulation(this, simulation);
								case PidSimulation.typeOfflineSimulation ->
										dfa = new OfflineSimulation(this, simulation);
								default -> logger.warning("Unbekanntes Simulationsobjekt: " + simulation);
							}

							if(dfa != null) {
								simulations.put(simulation, dfa);
								transitions.add(dfa);                    //dfa soll schnellstmöglich in den Zustand NEW wechseln.
							}
						}
					}
				}
				//Simulationsobjekte entfernen
				Arrays.asList(removedObjects).forEach(simulations::remove);
			}
		}
		catch(Exception ex) {
			StringBuilder sb = new StringBuilder();
			sb.append("Set: ").append(mutableSet);
			sb.append("Hinzugefügte Objekte: ");
			if(addedObjects != null) {
				for(SystemObject so : addedObjects) {
					sb.append(so.getPid()).append(", ");
				}
			}
			sb.append("Entfernte Objekte: ");
			if(removedObjects != null) {
				for(SystemObject so : removedObjects) {
					sb.append(so.getPid()).append(", ");
				}
			}
			Debug.getLogger().warning("Fehler beim Auswerten der eingehenden Daten. Daten werden ignoriert! " + sb, ex);
		}
	}

	/** @return Verbindung zum Datenverteiler. */
	public ClientDavInterface getDav() {
		return getArchMgr().getDavCon();
	}

	/**
	 * Eine Transition soll durchgeführt werden.
	 *
	 * @param dfa Simulationsobjekt, das eine Transition durchführen will.
	 */
	void addTransition(@NotNull Transition dfa) {
		transitions.add(dfa);
	}

	/**
	 * Führt zum Aufrufen von {@link SimulationAutomaton#wakeUp()} nach dem Ablauf von der parametrierbaren Wartezeit.
	 *
	 * @param dfa Automat der benachrichtigt werden soll.
	 */
	public void addWakeUp(SimulationAutomaton dfa) {
		wakeUp.addWakeUp(this, dfa, System.currentTimeMillis() + wakeUpTime);
	}

	/**
	 * Bricht das Aufwecken der übergebenen Simulation ab.
	 *
	 * @param dfa Simulation
	 */
	public void cancelWakeUp(SimulationAutomaton dfa) {
		wakeUp.cancelWakeUp(dfa);
	}

	@Override
	public void step() throws InterruptedException {
		//Ggf. Simulationen aufwecken
		wakeUp.wakeUp();
		//Simulation holen, die eine Transition ausführen möchte. Solange Warten, bis eine Transition auszuführen ist, oder bis geweckt werden soll.
		Transition transition;
		try {
			long timeToWakeUp = wakeUp.timeToWakeUp();
			if (timeToWakeUp == 0) {
				// Keine Wartezeit
				transition = transitions.take();
			} else {
				transition = transitions.take(Duration.ofMillis(timeToWakeUp));
			}
		} catch (TerminatedException e) {
			return;
		}
		//Transition durchführen
		if (transition instanceof SimulationAutomaton simulationAutomaton) {
			boolean transitioned = simulationAutomaton.transition();
			if (!transitioned) {
				_debug.fine("Übergang fehlgeschlagen");
			}
		}
		taskStepDone();
	}

	@Override
	public void terminateTask() {
		super.terminateTask();
		transitions.terminateNow();
	}

	@Override
	protected void cleanUp() {
		//von Mengenüberwachung abmelden
		if(mutableSet != null) mutableSet.removeChangeListener(this);

		//Meldung für alle Simulationen senden
		for(SimulationAutomaton simulationAutomaton : simulations.values()) {
			simulationAutomaton.terminate();
		}
	}

	/**
	 * Leitet einen Löschauftrag an den ArchiveManager weiter.
	 *
	 * @param ard Auftragsparameter
	 */
	public void insertDeleteQuery(SimulationResultData ard) {
		getArchMgr().getInQueuesMgr().insertInSimVarDeleteQueue(ard);
	}

	/**
	 * Liefert den Zeitraum in ms in dem eine erneute Steuerungsinformation für eine Simulation eingehen muss.
	 *
	 * @return Zeitraum in ms in dem eine erneute Steuerungsinformation für eine Simulation eingehen muss.
	 */
	public long getWakeUpTime() {
		return wakeUpTime;
	}

	/** @param timeout Zeit in ms in der ein Steuerbefehl von der Simulationssteuerung kommen muss, nachdem diese einen 'keine Quelle' Datensatz gesendet hat. */
	public void setTimeOutPeriod(long timeout) {
		this.wakeUpTime = timeout;
	}
}

/**
 * Diese Klasse verwaltet die Weckzeiten der Simulationen.
 *
 * @author beck et al. projects GmbH
 * @author Martin Hilgers
 */
class WakeUpManager {

	/** 'Weckzeit' -> Liste mit aufzuweckenden Simulation */
	private final Hashtable<Long, ArrayList<SimulationAutomaton>> simulations = new Hashtable<>();

	/** Simulation -> Liste in der die Simulation auftaucht. (Diese Liste wird in simulations verwendet!) */
	private final Hashtable<SimulationAutomaton, ArrayList<SimulationAutomaton>> simList = new Hashtable<>();

	/** Verwaltet die Weckzeit aufsteigend sortiert. */
	private final TreeSet<Long> minTime = new TreeSet<>();

	/** Weckt alle fälligen Simulationen auf. */
	public synchronized void wakeUp() {
		while (!minTime.isEmpty()) {
			Long wakeUpTime = minTime.first();
			if (wakeUpTime > System.currentTimeMillis()) {
				break;
			}

			// Weckzeit noch nicht erreicht.
			// Alle zu weckenden Simulationen aufzählen
			ArrayList<SimulationAutomaton> sims = simulations.remove(wakeUpTime);
			for (SimulationAutomaton sim : sims) {
				sim.wakeUp();
				simList.remove(sim);    //Die Simulation muss nicht mehr geweckt werden -> kann entfernt werden
			}
			sims.clear();
			minTime.remove(wakeUpTime);
		}
	}

	/**
	 * Fügt die Weckzeit für eine Simulation hinzu. Für jede Simulation darf nur eine Weckzeit hinterlegt werden. Ist bereits eine Weckzeit für die Simulation
	 * hinterlegt, so werden die weiteren Weckzeiten ignoriert. Benachrichtigt die Simulationsverwaltung {@link Thread#interrupt()}. falls die neue Weckzeit vor
	 * der bisher nächsten Weckzeit liegt.
	 *
	 * @param simManager Simulationsverwaltung
	 * @param dfa        Simulation die geweckt werden soll.
	 * @param wakeUpTime Weckzeit.
	 */
	public synchronized void addWakeUp(SimulationManager simManager, SimulationAutomaton dfa, long wakeUpTime) {
		if(wakeUpTime > 0 && simList.get(dfa) == null) {
			//Simulation in die passende Liste der zu weckenden Simulationen einfügen (d.h .Simulationen mit der gleichen Weckzeit zusammenfassen)
			ArrayList<SimulationAutomaton> sims = simulations.computeIfAbsent(wakeUpTime, k -> new ArrayList<>());
			sims.add(dfa);
			//Abbildung (Simulation -> Liste in der sie abgelegt wird) aktualisieren.
			simList.put(dfa, sims);

			//Weckzeit einsortieren
			if(minTime.isEmpty()) {
				//Erste Weckzeit, die eingefügt wird.
				minTime.add(wakeUpTime);
				// Simulationsverwaltung benachrichtigen -> da sich die minimale Weckzeit geändert hat
				simManager.addTransition(new SimulationManager.WakeUpNow());
			}
			else {
				//Es existieren bereits andere Weckzeiten
				Long first = minTime.first();
				if(first != wakeUpTime) {
					minTime.add(wakeUpTime);
					if(!first.equals(minTime.first()))
					// Simulationsverwaltung benachrichtigen -> da sich die minimale Weckzeit geändert hat
					{
						simManager.addTransition(new SimulationManager.WakeUpNow());
					}
				}
			}
		}
	}

	/**
	 * Bricht das Aufwecken der übergebenen Simulation ab.
	 *
	 * @param dfa Simulation
	 */
	public synchronized void cancelWakeUp(SimulationAutomaton dfa) {
		ArrayList<SimulationAutomaton> sims = simList.remove(dfa);
		if(sims != null) sims.remove(dfa);
	}

	/**
	 * Berechnet die Zeit in ms, in der Simulationen aufgeweckt werden müssen.
	 *
	 * @return Falls eine Weckzeit eingestellt wurde: Zeit bis zum Aufwecken oder {@code 1}, je nachdem welcher Wert größer ist. Wurde bisher noch keine
	 * Weckzeit eingestellt, so wird {@code 0} zurückgegeben.
	 */
	public synchronized long timeToWakeUp() {
		if(minTime.isEmpty()) {
			return 0;
		}
		else {
			long time = minTime.first() - System.currentTimeMillis();
			return (time < 1) ? 1 : time;
		}
	}
}
