/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.tasks;

import de.bsvrz.ars.ars.mgmt.ArchiveManager;
import de.bsvrz.ars.ars.mgmt.InQueuesMgr.DataReceiver;
import de.bsvrz.ars.ars.mgmt.tasks.base.TaskManager;
import de.bsvrz.dav.daf.main.ClientSenderInterface;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.archive.ArchiveDataSpecification;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.impl.archive.ArchiveDataCompression;
import de.bsvrz.sys.funclib.communicationStreams.StreamMultiplexer;
import de.bsvrz.sys.funclib.communicationStreams.StreamMultiplexerDirector;
import de.bsvrz.sys.funclib.dataSerializer.NoSuchVersionException;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.bsvrz.sys.funclib.losb.datk.MessageType;
import de.bsvrz.sys.funclib.losb.datk.PidQuery;
import de.bsvrz.sys.funclib.losb.kernsoftware.ConnectionManager;

import java.io.IOException;

/**
 * Task für Bearbeitung von Archivanfragen.
 * <p>
 * Die Archivanfrage wird analysiert und in ihre Unterabfragen {@link Query} zerlegt. Die Beantwortung erfolgt streambasiert mit einem Stream pro Unterabfrage.
 *
 * @author beck et al. projects GmbH
 * @author Thomas Schäfer
 * @author Alexander Schmidt
 * @version $Revision$ / $Date$ / ($Author$)
 */
public class ArchiveQueryTask extends QueryTask {

    /**
     * Default-Anzahl der Pakete, die pro Stream initial an den Empfaenger geschickt werden duerfen. Wert laut Datenkatalog.
     */
    private static final int BLOCKING_FACTOR_MUX_DEFAULT = 10;

    /**
     * Parametrierte Anzahl der Pakete, die pro Stream an den Empfaenger geschickt werden duerfen.
     */
    private static int blockingFactorMuxParam = BLOCKING_FACTOR_MUX_DEFAULT;

    /**
     * Tatsaechlich benutzte Anzahl der Pakete, die pro Stream an den Empfaenger geschickt werden, angepasst an die Größe des Empfaengerpuffers.
     */
    private int blockingFactorMuxUsed;

    /**
     * Default-Größe des Puffers im Stream-Multiplexer. Wert laut Datenkatalog.
     */
    private static final int BUFFER_SIZE_MUX_DEFAULT = 100000;

    /**
     * Parametrierte Größe des Puffers im Stream-Multiplexer.
     */
    private static int bufferSizeMuxParam = BUFFER_SIZE_MUX_DEFAULT;

    /**
     * Tatsaechlich benutzte Größe des Puffers im Stream-Multiplexer, angepasst an die Größe des Empfaengerpuffers.
     */
    private int bufferSizeMuxUsed;

    /**
     * Der StreamMultiplexer zum Verpacken der Ergebnis-Datensätze.
     */
    private volatile StreamMultiplexer mux;

    /**
     * Task zur Verarbeitung der Flusskontroll-Steuerungspakete.
     */
    private final FlowControlTask flowCtrl;

    /**
     * Größe des Empfangspuffers der anfragenden Applikation.
     */
    private int receiverBufferSize;

    /**
     * Wurde die initiale Antwort (kann auch Fehlermeldung sein) verschickt?
     */
    private boolean initialResponseSent;

    /**
     * Maximale Anzahl paralleler Archivanfragen je Applikation
     */
    private final int maximumRequestsPerApplication;


    private ArchiveQueryTask.Director _director;

    /**
     * Erzeugt einen neuen Query-Task.
     *
     * @param archiveMgr                    Archiv-Verwaltung
     * @param tMgr                          Zugeordneter Task-Manager
     * @param flowCtrl                      Verwaltung der Flusskontrolle
     * @param maximumRequestsPerApplication Maximale Anzahl von gleichzeitigen Anfragen pro Applikation
     */
    public ArchiveQueryTask(ArchiveManager archiveMgr, MultiTaskManager tMgr, FlowControlTask flowCtrl, int maximumRequestsPerApplication) {
        super(archiveMgr, tMgr);
        this.flowCtrl = flowCtrl;
        this.maximumRequestsPerApplication = maximumRequestsPerApplication;
    }

    /**
     * Fuehrt alle notwendigen Anmeldungen durch.
     *
     * @param archMgr  Archiv-Verwaltung
     * @param receiver Empfaengerobjekt
     */
    public static void subscribeObjects(ArchiveManager archMgr, DataReceiver receiver) {
        String atgPid = PidQuery.ATG_QUERY_PID;
        String aspPid = PidQuery.ASP_QUERY_PID;
        try {
            ConnectionManager.subscrDrainNormal(archMgr.getDavCon(), receiver, archMgr.getArchiveObject(), atgPid, aspPid);
        } catch (Exception e) {
            Debug.getLogger().error(
                    "Archivsystem konnte sich nicht als Senke für Archivanfragen anmelden: " + archMgr.getArchiveObject() + "/" + atgPid + "/" + aspPid + Debug
                            .NEWLINE + e.getMessage()
            );
        }
    }

    /**
     * Über diese Methode kann die Parametrierung den Blocking-Faktor des StreamMultiplexers auf den parametrierten Wert setzen (Attribut "AnzahlBlocks" in
     * Attributgruppe "ArchivEinstellung").
     *
     * @param bFMuxParam Parametrierter Blocking-Faktor des StreamMultiplexers
     */
    public static void setBlockingFactorMux(int bFMuxParam) {
        blockingFactorMuxParam = bFMuxParam;
    }

    /**
     * Über diese Methode kann die Parametrierung die PufferGröße des StreamMultiplexers auf den parametrierten Wert setzen (Attribut "AnzahlBytes" in
     * Attributgruppe "ArchivEinstellung").
     *
     * @param bSMuxParam Parametrierte PufferGröße des StreamMultiplexers
     */
    public static void setBufferSizeMux(int bSMuxParam) {
        bufferSizeMuxParam = bSMuxParam;
    }

    /**
     * @see #setBlockingFactorMux(int)
     */
    public static int getBlockingFactorMux() {
        return blockingFactorMuxParam;
	}

	/**
	 * @see #setBlockingFactorMux(int)
	 */
	public static int getBufferSizeMux() {
        return bufferSizeMuxParam;
    }

    //////////////////////////////////////////////////////////////////////////////////////////////////////////
    // Task Methoden
    //////////////////////////////////////////////////////////////////////////////////////////////////////////

    /**
     * @see de.bsvrz.ars.ars.mgmt.tasks.MultiTask#work()
     */
    @Override
    public void work() {
        //logger.finer(getName() + " Archivanfrage erhalten: " + dataset);
        init();
	    ArchiveManager archiveManager = getArchMgr();
	    archiveManager.getPersistenceManager().assertNoLocks();
        boolean counted = false;
        try {
            final ArchiveDataSpecification[] ads = analyze();
            if (queryAppObj == null) {
                return;
            }
            final int archivQueryCountForApplication = getArchMgr().increaseArchiveQueryCountForApplication(queryAppObj);
            counted = true;
            final SendingState state;
            sendingState = SendingState.STOP;
            subscribeSender();
            state = waitForSendControl();
            suspendTaskIfNecessary();
            TaskManager.run("Archivanfrage", (tpi -> {
                try (QueryData queries = QueryData.createQueryData(getArchMgr(), ads, archivQueryCountForApplication, maximumRequestsPerApplication, queryAppObj)) {
                    // Stream-Multiplexer nur vorbereiten, falls Unterabfragen zu beantworten sind (ein Stream pro Unterabfrage):
                    if (queries.size() > 0) {
                        configureMuxParameter();
                        _director = new Director(queries);
                        mux = new StreamMultiplexer(queries.size(), blockingFactorMuxUsed, bufferSizeMuxUsed, serializerVersion, _director);
                        flowCtrl.subscribe(queryAppObj, queryIdx, mux);
                    }

                    if (state == SendingState.SEND) {
                        queries.sendReply(this::sendInitialResponse, mux);
                    }
                }
            }));
        } catch (InterruptedException e) {
            _debug.finer(getName() + ": Archivanfrage durch Interrupt beendet.");
            if (mux != null) mux.killAllStreams();
        } catch (Exception e) {
            _debug.error(getName() + ": Fehler beim Bearbeiten der Archivanfrage: " + resultData, e);
            if (mux != null) mux.killAllStreams();
        } finally {
            if (counted && queryAppObj != null) {
                getArchMgr().decreaseArchiveQueryCountForApplication(queryAppObj);
            }
            unsubscribeSender(); // Als Sender der Archiv-Antwort abmelden
            _debug.finer(
                    getName() + " Archivanfrage beendet." + (initialResponseSent ? "" : " Wegen eines Fehlers konnte keine initiale Antwort gesendet werden")
            );
        }
    }


    //////////////////////////////////////////////////////////////////////////////////////////////////////////
    // ClientSenderInterface Methoden
    //////////////////////////////////////////////////////////////////////////////////////////////////////////

    /**
     * @see ClientSenderInterface#dataRequest(SystemObject, DataDescription, byte)
     */
    @Override
    public void dataRequest(SystemObject object, DataDescription dataDescription, byte state) {
        if (sendingState == SendingState.SEND && state != ClientSenderInterface.START_SENDING) {
            _director.killAllStreams();
        }
        super.dataRequest(object, dataDescription, state);
    }

    @Override
    public void terminateTask() {
        super.terminateTask();
        StreamMultiplexer mux = this.mux;
        if (mux != null) mux.killAllStreams();
    }

    @Override
    protected void cleanUp() {
        if (mux != null) mux.killAllStreams();
    }

    //////////////////////////////////////////////////////////////////////////////////////////////////////////
    // Private Methoden
    //////////////////////////////////////////////////////////////////////////////////////////////////////////

    /**
     * Initialisiert den Task für eine neue Archivanfrage.
     */
    @Override
    protected void init() {
        super.init();
        receiverBufferSize = 0;
        mux = null;
        blockingFactorMuxUsed = 0;
        bufferSizeMuxUsed = 0;
        initialResponseSent = false;
    }

    /**
     * Analysiert die aktuelle Archivanfrage und fuellt die Liste der {@link Query}.
     *
     * @throws NoSuchVersionException Anfrage kann nicht deserialisiert werden, weil die Serialisierer-Version unbekannt ist
     * @throws IOException            Allgemeiner IO-Fehler
     */
    private ArchiveDataSpecification[] analyze() throws NoSuchVersionException, IOException {
        createQueryData();
        deserializer.readInt();                         // Prioritaet; bereits vom InQueuesMgr erledigt
        receiverBufferSize = deserializer.readInt();    // Größe des Empfangspuffers

        return parseArchiveDataSpec(deserializer, getArchMgr().getDataModel());
    }

    /**
     * Meldet den Task als Sender für Archiv-Antworten an das angegebene Empfaenger-Applikations-Objekt ab. Deregistriert den StreamMultiplexer für den Empfang
     * von Flusskontroll-Steuerungspaketen.
     */
    @Override
    protected void unsubscribeSender() {
        if (mux != null) flowCtrl.unsubscribe(queryAppObj, queryIdx);
        super.unsubscribeSender();
    }

    /**
     * Sendet die initiale Ergebnismeldung zur Archiv-Anfrage an das Empfaenger-Applikations-Objekt.
     *
     * @param success  Anfrage erfolgreich
     * @param errorMsg Fehlermeldung bei Misserfolg
     * @throws IOException Allgemeiner IO-Fehler
     */
    private void sendInitialResponse(boolean success, String errorMsg) throws IOException {
        // ======		   Datenaufbereitung erfolgt hier gemäß		   ======
        // ======  Datenkatalog Kapitel 7.2.3.12 ArchivAnfrageSchnittstelle  ======
        // ======			 Unterpunkt ArchivNachrichtenTyp 2			  ======
        bosResult.reset();
        serializer.writeByte(success ? 1 : 0);      // Anfrage erfolgreich ja/nein
        serializer.writeString(errorMsg);           // Fehlermeldung bei Misserfolg
        serializer.writeInt(blockingFactorMuxUsed); // Blocking-Faktor des Stream-Multiplexers und -Demultiplexers
        sendResultData(bosResult.toByteArray(), MessageType.INITIAL_QUERY_RESULT);
        initialResponseSent = true;
    }

    public interface ResponseSender {
        void sendInitialResponse(boolean success, String errorMsg) throws IOException;
    }

    /**
     * Bestimmt die Parameter bufferSizeStreamMultiplexer und blockingFactor, mit denen der StreamMultiplexer per Konstruktor parametriert wird, anhand der Werte
     * aus den Archivsystem-Einstellungen und dem tatsaechlich vorhandenen Empfangspuffer des Empfaengers.
     */
    private void configureMuxParameter() {
        // Parametrierte sendeseitige PufferGröße:
        bufferSizeMuxUsed = bufferSizeMuxParam;
        blockingFactorMuxUsed = blockingFactorMuxParam;

        // Negative Werte und Null ausschliessen:
        if (bufferSizeMuxUsed < 1) bufferSizeMuxUsed = 1;
        if (blockingFactorMuxUsed < 1) blockingFactorMuxUsed = 1;

        if (receiverBufferSize > 1) {
            // Empfangspuffer hat sinnvollen Wert -> Sender daran anpassen
            if (receiverBufferSize >= bufferSizeMuxUsed) {
                // Fall 1: Empfangspuffer Größer Sendepuffer (Dafault-Fall)
                blockingFactorMuxUsed = receiverBufferSize / bufferSizeMuxUsed;
            } else {
                // Fall 2: Empfangspuffer kleiner Sendepuffer
                blockingFactorMuxUsed = 1;
                bufferSizeMuxUsed = receiverBufferSize;
            }
        }
    }

    //////////////////////////////////////////////////////////////////////////////////////////////////////////
    // Private Klasse Query
    //////////////////////////////////////////////////////////////////////////////////////////////////////////

    private class Director implements StreamMultiplexerDirector {
        private final QueryData queries;

        public Director(final QueryData queries) {
            this.queries = queries;
        }


        @Override
        public void sendData(byte[] streamDataPacket) {
            sendResultData(streamDataPacket, MessageType.STREAM_DATA);
        }

        @Nullable
        @Override
        public byte[] take(int indexOfStream) {
            Query query = null;
            try {
                query = queries.get(indexOfStream); // Unterabfrage zum Stream
                if (query.done) {
                    query.close();
                    return null;        // Abbruch, falls Query fertig
                }

                SimpleArchiveData arcData = query.take(); // Nächster zu sendender Datensatz
                if (arcData == null) {
                    query.close();
                    return null;                // Abbruch, falls Query fertig
                }

                // Datenaufbereitung gemäß DatK 7.2.3.12
                bosResult.reset();
                serializer.writeInt(arcData.dataKind().getCode());
                serializer.writeLong(arcData.dataTime());
                serializer.writeLong(arcData.arcTime());
                serializer.writeLong(arcData.dataIdx());
                serializer.writeInt(arcData.dataState().getCode());
                serializer.writeInt(serializerVersion);
                serializer.writeByte(arcData.compressed() ? ArchiveDataCompression.ZIP.getCode() : ArchiveDataCompression.NONE.getCode());

                if (arcData.rawData() != null) {
                    serializer.writeInt(arcData.rawData().length);
                    serializer.writeBytes(arcData.rawData());
                } else {
                    serializer.writeInt(0);
                }
                return bosResult.toByteArray();
            } catch (Exception e) {
                _debug.error(getName() + ": Unterabfrage " + indexOfStream + ": Erstellung Datensatz fuer Archivantwort fehlgeschlagen.", e);
            }
            if (query != null)
                query.done = true;     // Query beenden, Speicher freigeben
            return null;                        // Kennzeichen für "keine Daten mehr"
        }

        @Override
        public void streamAborted(int indexOfStream) {
            _debug.finest(getName() + " QueryTask verwirft Stream mit Index " + indexOfStream);
            if (indexOfStream >= 0 && indexOfStream < queries.size()) {
                queries.get(indexOfStream).done = true;
            }
        }

        public void killAllStreams() {
            // Synchronisiert sich mit der Methode Work und verhindert, dass bereits die Verarbeitung einer neuen Anfrage begonnen wird
			// während die Sendesteuerung noch verarbeitet wird.
			synchronized (queries) {
				if (mux != null) mux.killAllStreams();
			}
		}
	}
}
