/*
 * Copyright 2019 by Kappich Systemberatung Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.tasks;

import com.google.common.collect.DiscreteDomain;
import com.google.common.collect.Range;
import com.google.common.collect.RangeSet;
import com.google.common.collect.TreeRangeSet;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.losb.util.Util;

import java.util.Iterator;
import java.util.stream.Collectors;

/**
 * Klasse für bereiche von Datenindexlücken
 *
 * @author Kappich Systemberatung
 */
public final class DataIndexRangeSet implements Iterable<Range<Long>> {

	/**
	 * Dieses Set enthält z. B. die Datenindexbereiche, für die keine Daten vorliegen.
	 * <p> 
	 * Hier werden nur {@link com.google.common.collect.Range#canonical(DiscreteDomain) canonical}-Elemente eingefügt. 
	 */
	private final RangeSet<Long> _rangeSet = TreeRangeSet.create();

	/**
	 * Erstellt ein neues DataIndexRangeSet.
	 */
	public DataIndexRangeSet() {
	}

	/**
	 * Kopiert eine {@link DataIndexRangeSet}
	 * @param initialValue bestehendes {@link DataIndexRangeSet}
	 */
	public DataIndexRangeSet(final DataIndexRangeSet initialValue) {
		addAll(initialValue);
	}

	/**
	 * Fügt einen Datenindexbereich hinzu 
	 * @param minIdx          Von-Index (wo beginnt die Lücke)
	 * @param maxIdxExclusive Bis-Index (wo ist der nächste vorhandene Datensatz?)
	 */
	public void addRange(long minIdx, long maxIdxExclusive) {
		_rangeSet.add(Range.closedOpen(minIdx, maxIdxExclusive));
	}

	/**
	 * Entfernt alle angegebenen Lücken 
	 * @param other Lücken
	 */
	public void removeAll(DataIndexRangeSet other) {
		_rangeSet.removeAll(other._rangeSet);
	}

	/**
	 * Fügt alle angegebenen Lücken hinzu
	 * @param other Lücken
	 */
	public void addAll(DataIndexRangeSet other) {
		_rangeSet.addAll(other._rangeSet);
	}

	/**
	 * Gibt zurück, ob ein angegebener Bereich vollständig (als Lücke) enthalten ist, d. h. wenn
	 * {@link #addRange(long, long)} mit den angegebenen parametern dieses Objekt nicht ändern würde.
	 *
	 * @param minIdx          Von-Index (inklusiv)
	 * @param maxIdxExclusive Bis-Index (exklusiv)
	 * @return Ist die lücke enthalten?
	 */
	public boolean contains(long minIdx, long maxIdxExclusive) {
		return _rangeSet.asRanges().contains(Range.closedOpen(minIdx, maxIdxExclusive));
	}

	@Override
	public String toString() {
		return _rangeSet.asRanges()
				.stream()
				.map(it -> Util.dIdx2StrExt(it.lowerEndpoint()) + "-" + Util.dIdx2StrExt(it.upperEndpoint()))
				.collect(Collectors.joining(", ", "[", "]"));
	}

	/**
	 * Gibt {@code true} zurück, wenn das RangeSet leer ist.
	 * @return {@code true}, wenn das RangeSet leer ist, sonst {@code false}
	 */
	public boolean isEmpty() {
		return _rangeSet.isEmpty();
	}

	/**
	 * Gibt die Anzahl an Lücken zurück.
	 * @return die Anzahl an Lücken
	 */
	public int size() {
		return _rangeSet.asRanges().size();
	}

	@NotNull
	@Override
	public Iterator<Range<Long>> iterator() {
		return _rangeSet.asRanges().iterator();
	}
}
