/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.tasks;

import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.sys.funclib.dataSerializer.Serializer;
import de.bsvrz.sys.funclib.dataSerializer.SerializingFactory;

import java.io.ByteArrayOutputStream;
import java.io.IOException;

import static de.bsvrz.ars.ars.persistence.ContainerFile.*;

/**
 * Klasse, die Datensätze serialisiert, um diese in Containerdateien zu schreiben.
 * <p>
 * Diese Klasse ist nicht Thread-safe, muss also für jeden Abarbeitungs-Thread neu instantiiert werden.
 */
public class DavDataSerializer {

	/**
	 * Puffer für den Serialisierer und zum Lesen eines Datensatzes.
	 */
	private final ByteArrayOutputStream bos = new ByteArrayOutputStream(BUFFER_SIZE);

	/**
	 * Datensatz-Serialisierer.
	 */
	private final Serializer serializer = SerializingFactory.createSerializer(bos);

	/**
	 * Serialisiert den übergebenen Datensatz in ein byte-Array. Das Byte-Array enthält noch keinen Header
	 * mit Zeitstempeln u.ä., es geht nur um den rohen Datensatz.
	 *
	 * @param data Zu serialisierender Datensatz
	 * @return Byte-Array mit serialisiertem Datensatz
	 */
	public byte[] serializeData(Data data) {
		bos.reset();
		try {
			serializer.writeData(data);
		} catch (IOException e) {
			// Auf ByteArrayOutputStream treten keine IOExceptions auf
			throw new AssertionError("Datensatz konnte nicht serialisiert werden: " + e.getMessage() + "\n" + data, e);
		}
		return bos.toByteArray();
	}
}
