/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2019 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.tasks;

import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.communicationStreams.StreamMultiplexer;
import de.bsvrz.sys.funclib.dataSerializer.Deserializer;
import de.bsvrz.sys.funclib.dataSerializer.SerializingFactory;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.losb.datk.PidQuery;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;

/**
 * KLasse zur Verarbeitung der Steuerungspakete, die von anderen Applikationen zur Flusskontrolle an das Archivsystem geschickt werden. Der Empfaenger einer
 * Archivantwort kann den Sendefluss des im {@link de.bsvrz.ars.ars.mgmt.tasks.ArchiveQueryTask} eingesetzten {@link StreamMultiplexer} steuern. Die dazu
 * versandten Steuerungspakete werden vom {@link de.bsvrz.ars.ars.mgmt.InQueuesMgr} empfangen und dann von diesem Task an die entsprechenden Multiplexer
 * weitergeleitet.
 *
 * @author beck et al. projects GmbH
 * @author Thomas Schaefer
 * @version $Revision$ / $Date$ / ($Author$)
 */
public class FlowControlTask {

	/** Spezielles Objekt zur Synchronisierung. */
	private final Object syncObj = new Object();

	private final Debug logger;

	/** Datenhaltung: Anfragendes ApplikationsObjekt -> Anfrage-Index -> StreamMultiplexer. */
	private final Map<SystemObject, Map<Integer, StreamMultiplexer>> ctrlMap;

	/**
	 * Erzeugt einen neuen Flusskontroll-Task.
	 */
	public FlowControlTask() {
		logger = Debug.getLogger();
		ctrlMap = new HashMap<>();
	}

	/**
	 * @see de.bsvrz.ars.ars.mgmt.tasks.SingleTask#work (ResultData)
	 */
	public void processFlowCtrl(ResultData resultData) {
		Data data = resultData.getData();                                                              // Datensatz
		SystemObject queryAppObj = data.getReferenceValue(PidQuery.ATT_SENDER_NAME).getSystemObject(); // Anfrage-Applikation
		int queryIdx = data.getUnscaledValue(PidQuery.ATT_QUERY_IDX_NAME).intValue();                  // Anfrage-Index
		byte[] dataArray = data.getUnscaledArray(PidQuery.ATT_DATA_NAME).getByteArray();               // Datenbytes mit Steuerungspaket

		// Multiplexer suchen, der für diese Anfrage-Applikation/Anfrage-Index registriert ist,
		// und Steuerungspaket zur Flusskontrolle übergeben:
		synchronized (syncObj) {
			Map<Integer, StreamMultiplexer> map = ctrlMap.get(queryAppObj);
//			System.out.println("processFlowCtrl: map für App " + queryAppObj.getId() + " und queryIdx " + queryIdx + (map == null ? " nicht" : "") + " gefunden");
			if (map != null) {
				StreamMultiplexer mux = map.get(queryIdx);
//				System.out.println("processFlowCtrl: map für mux " + queryAppObj.getId() + " und queryIdx " + queryIdx + (mux == null ? " nicht" : "") + " gefunden");
				if (mux != null) {
					// Nur für Fehlersuche.
//					log(queryAppObj, queryIdx, dataArray);
					try {
						mux.setMaximumStreamTicketIndexForStream(dataArray);
					}
					catch(IOException e) {
						logger.warning("Flusskontroll-Steuerungspaket konnte nicht an StreamMultiplexer uebergeben werden.", e);
					}
				}
			}
		}
	}

	/**
	 * Registriert den angegebenen StreamMultiplexer für den Empfang von Steuerpaketen vom angegebenen Anfrage-Applikations-Objekt und Anfrage-Index, die
	 * dieser gerade mit einer Archivantwort versorgt.
	 *
	 * @param queryAppObj Anfrage-Applikations-Objekt
	 * @param queryIdx    Anfrage-Index
	 * @param mux         StreamMultiplexer
	 */
	public void subscribe(SystemObject queryAppObj, int queryIdx, StreamMultiplexer mux) {
		synchronized(syncObj) {
			// Map der anfragenden Applikation holen; anlegen falls noch nicht vorhanden:
			Map<Integer, StreamMultiplexer> map = ctrlMap.computeIfAbsent(queryAppObj, k -> new HashMap<>());
			// StreamMultiplexer in Map der anfragenden Applikation einhängen:
			map.put(queryIdx, mux);
		}
	}

	/**
	 * De-Registriert den zuvor unter dem angegebenen Anfrage-Applikations-Objekt und Anfrage-Index registrierten StreamMultiplexer für den Empfang von
	 * Steuerpaketen.
	 *
	 * @param queryAppObj Anfrage-Applikations-Objekt
	 * @param queryIdx    Anfrage-Index
	 */
	public void unsubscribe(SystemObject queryAppObj, int queryIdx) {
		synchronized(syncObj) {
			Map<Integer, StreamMultiplexer> map = ctrlMap.get(queryAppObj);
			if(map != null) map.remove(queryIdx);
		}
	}

	@SuppressWarnings("unused")
	private void log(byte[] streamTicketPacket) {
		InputStream in = new ByteArrayInputStream(streamTicketPacket);

		// Deserialisieren:
		Deserializer deserializer = SerializingFactory.createDeserializer(in);
		int indexOfStream = 0;
		int maximumStreamTicketIndex = 0;
		try {
			// Index des Streams, dessen Ticketanzahl erhöht werden soll
			indexOfStream = deserializer.readInt();
			// Eine neue Obergrenze bis zu der der Stream Daten senden darf, -1 zeigt an, dass der Stream beendet werden soll
			maximumStreamTicketIndex = deserializer.readInt();
		}
		catch(IOException e) {
			logger.error("Fehler", e);
		}

		logger.info("Payload: Stream-Idx=" + indexOfStream + " MaxStreamTicketIdx=" + maximumStreamTicketIndex);
	}
	@SuppressWarnings("unused")
	private void log(final SystemObject queryAppObj, final int queryIdx, byte[] streamTicketPacket) {
		InputStream in = new ByteArrayInputStream(streamTicketPacket);

		// Deserialisieren:
		Deserializer deserializer = SerializingFactory.createDeserializer(in);
		int indexOfStream = 0;
		int maximumStreamTicketIndex = 0;
		try {
			// Index des Streams, dessen Ticketanzahl erhöht werden soll
			indexOfStream = deserializer.readInt();
			// Eine neue Obergrenze bis zu der der Stream Daten senden darf, -1 zeigt an, dass der Stream beendet werden soll
			maximumStreamTicketIndex = deserializer.readInt();
		}
		catch(IOException e) {
			logger.error("Fehler", e);
		}

		if(maximumStreamTicketIndex <= 0) logger.info("FlowControlTask: app: " + queryAppObj.getId() + ", query: " + queryIdx + " streamIndex: " + indexOfStream + " MaxStreamTicketIdx: " + maximumStreamTicketIndex);
	}
}
