/*
 * Copyright 2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.tasks;

import com.google.common.base.Joiner;
import com.google.common.collect.Multimap;
import com.google.common.collect.MultimapBuilder;
import com.google.common.collect.Multimaps;
import de.bsvrz.ars.ars.mgmt.ArchiveManager;
import de.bsvrz.ars.ars.mgmt.LongTermTaskStatePublisher;
import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SyncKey;
import de.bsvrz.ars.ars.mgmt.tasks.base.TaskManager;
import de.bsvrz.ars.ars.mgmt.tasks.base.TaskState;
import de.bsvrz.ars.ars.persistence.*;
import de.bsvrz.ars.ars.persistence.directories.ActivePersistenceDirectory;
import de.bsvrz.ars.ars.persistence.directories.PersistenceDirectory;
import de.bsvrz.ars.ars.persistence.directories.mgmt.lock.DirectoryStatus;
import de.bsvrz.ars.ars.persistence.directories.mgmt.lock.LockFileManager;
import de.bsvrz.ars.ars.persistence.util.FileSync;
import de.bsvrz.ars.ars.persistence.util.SyncResult;
import de.bsvrz.ars.ars.persistence.walk.DataIdentificationDirAction;
import de.bsvrz.ars.ars.persistence.walk.DataIdentificationDirWalk;
import de.bsvrz.ars.ars.persistence.walk.internal.StatusPrinter;
import de.bsvrz.dav.daf.util.cron.CronDefinition;
import de.bsvrz.dav.daf.util.cron.CronScheduler;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.*;
import java.time.Duration;
import java.util.*;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * {@link Task}-Implementierung vom datenkonsistenten Backup.
 *
 * <p>Grundgedanke des datenkonsitenten Backups ist, dass es ausreichend ist, alle Containerverzeichnisse (bzw. Datenidentifikationen) einzeln in einem konsistenten Zustand zu sichern.</p>
 * <p>Zur Umsetzung dieses Konzepts würde im Archivsystem ein weiterer Hintergrundtask integriert werden, der alle Containerverzeichnisse nacheinander durchgeht und einzeln sichert. Eine parallele Bearbeitung von mehreren Verzeichnissen ist prinzipiell problemlos möglich, könnte jedoch die Systemlast erhöhen und sollte daher nicht standardmäßig aktiviert werden.
 * Der neue Task wird regelmäßig folgende Aktionen ausführen:</p>
 * <ul>
 * <li>Für jedes Containerdateienverzeichnis:<ol>
 * <li>Schreibe die Indexe, leere den Cache und unterdrücke weitere Schreibvorgänge</li>
 * <li>Stelle sicher, dass das entsprechende Backup-Verzeichnis auf demselben Stand ist, wie das aktuelle Archivdatenverzeichnis im Persistenzverzeichnis. Das entsprechende Unterverzeichnis im Backup-Ordner wird angelegt, wenn es noch nicht existiert. </li>
 * <li>Daten, die im Backup-Verzeichnis noch nicht existieren, werden kopiert, Daten, die auf einem alten bzw. abweichenden Stand sind, werden aktualisiert und Daten, die nicht mehr existieren (z. B. weil sie aus dem Archiv ausgelagert wurden), werden im Backup-Verzeichnis ebenfalls gelöscht.</li>
 * <li>Hebe die Blockierung wieder auf </li>
 * </ol>
 * </li>
 * <li>Schreibe am Ende eine _startUpInfo.property und _restartTime.property, die den Stand der gesicherten Daten angibt
 * (die Inhalte können bei der Sicherung aggregiert werden).</li>
 * <li>Starte (auf Wunsch) einen vom Anwender konfigurierbaren Prozess, der die Daten aus diesem Verzeichnis noch einmal an einen anderen Ort sichert oder beispielsweise einen Schnappschuss erstellt.</li>
 * <li>Warte ggf. auf die Beendigung dieses Prozesses</li>
 * </ul>
 *
 * @author Kappich Systemberatung
 */
public final class MirrorTask extends CronTask {

	private static final Debug _debug = Debug.getLogger();
	private static final String[] BYTE_UNIT = {" Byte", " KiB", " MiB", " GiB", " TiB", " PiB", " EiB"};

	/**
	 * {@link CronScheduler} zum regelmäßigen Aktualisieren der Definitionsdateien von der Festplatte
	 */
	private final CronScheduler _refreshScheduler = new CronScheduler();

	private final de.bsvrz.ars.ars.mgmt.tasks.base.Task _longTermTaskStatePublisherTask;

	/**
	 * Benutzt für die Lock-Dateien in den Zielverzeichnissen
	 */
	private final LockFileManager lockFileManager = new LockFileManager();

	/**
	 * Konstruktor
	 *
	 * @param archiveMgr                 Archivmanager
	 * @param longTermTaskStatePublisher Publisher
	 */
	public MirrorTask(final ArchiveManager archiveMgr, final LongTermTaskStatePublisher longTermTaskStatePublisher) {
		super(archiveMgr);
		setName("Datenkonsistentes Backup");
		_refreshScheduler.schedule(this::refreshTasks, CronDefinition.EVERY_HOUR);
		refreshTasks();
		_longTermTaskStatePublisherTask = longTermTaskStatePublisher.registerTask("Datenkonsistentes Backup");
		_longTermTaskStatePublisherTask.setTaskState(TaskState.IDLE);
	}

	private void refreshTasks() {
		setCronTaskDefinitions(refreshCronTasks());
	}

	@Override
	public void terminateTask() {
		super.terminateTask();
		_refreshScheduler.shutdownNow();
		_longTermTaskStatePublisherTask.setTaskState(TaskState.TERMINATED);
	}

	private List<CronTaskDefinition> refreshCronTasks() {
		try {
			Path backupDir = getPersistenceManager().getBackupConfigurationDirectory();
			Files.createDirectories(backupDir);
			List<MirrorDefinition> definitions = new ArrayList<>();
			try (Stream<Path> list = Files.list(backupDir)) {
				list.forEach(new Consumer<>() {
					@Override
					public void accept(final Path path) {
						if (path.getFileName().toString().toLowerCase(Locale.ROOT).endsWith(".properties")) {
							Properties properties = new Properties();
							try (InputStream inStream = Files.newInputStream(path)) {
								properties.load(inStream);
							} catch (IOException e) {
								_debug.warning("Datei " + path + " konnte nicht gelesen werden.", e);
								return;
							}
							String cronString = properties.getProperty("cron");
							String targetString = properties.getProperty("ziel");
							if (cronString == null) {
								_debug.warning("Datei " + path + " konnte nicht gelesen werden: Sie hat keinen Parameter \"cron\".");
								return;
							}
							if (targetString == null) {
								_debug.warning("Datei " + path + " konnte nicht gelesen werden: Sie hat keinen Parameter \"ziel\".");
								return;
							}

							Path targetBaseDir;
							try {
								targetBaseDir = Paths.get(targetString.trim());
							} catch (InvalidPathException e) {
								_debug.warning("Das in Datei \"" + path + "\" angegebene Ziel \"" + targetString + "\" ist kein gültiger Pfadname. Die Backupdefinition wird ignoriert.", e);
								return;
							}

							String commandString = properties.getProperty("cmd");
							String[] commandLine = getCommand(commandString);

							CronDefinition cronDefinition;
							try {
								cronDefinition = new CronDefinition(cronString);
							} catch (IllegalArgumentException e) {
								_debug.warning("Der in Datei \"" + path + "\" angegebene Cron-String \"" + cronString + "\" ist ungültig. Die Backupdefinition wird ignoriert.", e);
								return;
							}

							MirrorDefinition e = new MirrorDefinition(cronDefinition, targetBaseDir, commandLine);
							definitions.add(e);
						}
					}

					private String[] getCommand(final String command) {
						if (command == null) {
							return new String[0];
						}
						if (System.getProperty("os.name", "unknown").startsWith("Windows")) {
							return new String[]{"cmd", "/C", command};
						}
						return new String[]{"sh", "-c", command};
					}
				});
			}
			return definitions.stream().map(this::createMirrorCronTask)
					.collect(Collectors.toList());
		} catch (IOException e) {
			_debug.warning("Kann Backup-Definitionen nicht einlesen", e);
		}
		return Collections.emptyList();
	}

	/**
	 * Erstellt aus einer {@link MirrorDefinition} eine {@link de.bsvrz.ars.ars.mgmt.tasks.CronTask.CronTaskDefinition}
	 *
	 * @param mirrorDefinition Enthält Einstellungen aus einer Einstellungsdatei (backup/???.properties)
	 * @return CronTaskDefinition, die an den {@link de.bsvrz.dav.daf.util.cron.CronScheduler} übergeben werden kann
	 */
	private CronTaskDefinition createMirrorCronTask(final MirrorDefinition mirrorDefinition) {
		return new CronTaskDefinition(mirrorDefinition.getCronDefinition(), () -> {
			TaskManager.run(_longTermTaskStatePublisherTask, (tpi) -> {
				doMirror(mirrorDefinition.getTargetDirectory(), mirrorDefinition.getCommandLine());
			});
			_longTermTaskStatePublisherTask.setTaskState(TaskState.IDLE);
		});
	}

	/**
	 * Führt einen Backup-lauf durch
	 *
	 * @param targetBaseDir Zielverzeichnis
	 * @param commandLine   Kommando, das nach dem erfolgreichen Ausführen ausgeführt wird.
	 */
	public void doMirror(final Path targetBaseDir, final String[] commandLine) {
		if (shouldTerminate()) {
			return;
		}
		try {
			Thread.currentThread().setName("Datenkonsistentes Backup nach " + targetBaseDir);
			Files.createDirectories(targetBaseDir);
			Path startUpInfoFile = targetBaseDir.resolve(StartupProperties.STARTUP_PROP_FILE_NAME);
			Path restartTimeFile = targetBaseDir.resolve(PersistenceManager.RESTART_TIME_FILE_NAME);

			Files.deleteIfExists(startUpInfoFile);
			Files.deleteIfExists(restartTimeFile);

			Path sourceRootPath = getPersistenceManager().getRootPath();

			MirrorAction lastResult = null;

			List<? extends PersistenceDirectory> persistenceDirectories = getPersistenceManager().getPersistenceDirectories(0);
			for (PersistenceDirectory persistenceDirectory : persistenceDirectories) {
				Path relativePath = sourceRootPath.relativize(persistenceDirectory.getBasePath());
				Path targetPath = targetBaseDir.resolve(relativePath);

				MirrorAction result = doPersistenceDirectoryMirror(persistenceDirectory, targetPath);
				lastResult = result != null && result.isSuccessful() ? result : null;
			}
			new FileSync().syncDirectories(sourceRootPath.resolve(PersistenceManager.BACKUP_DIR_NAME), targetBaseDir.resolve(PersistenceManager.BACKUP_DIR_NAME));
			if (shouldTerminate()) {
				return;
			}
			if (lastResult != null) {
				writeStartUpInfo(lastResult, targetBaseDir);
				writeRestartTime(lastResult, targetBaseDir);
			}
			if (commandLine.length > 0) {
				runExternalProcess(targetBaseDir, commandLine);
			}
		} catch (Exception e) {
			_debug.error("Unerwarteter Fehler beim konsistenten Backup", e);
		}
	}

	private MirrorAction doPersistenceDirectoryMirror(PersistenceDirectory sourceDirectory, Path targetDirectory) throws Exception {
		DirectoryStatus status;
		if (Files.exists(targetDirectory)) {
			try {
				status = lockFileManager.open(targetDirectory);
			} catch (FileAlreadyExistsException ex) {
				// isActive-Datei ist vorhanden, also vermutlich nicht fertig synchronisiert
				lockFileManager.forceWritable(targetDirectory);
				status = DirectoryStatus.Writable;
			}
		} else {
			lockFileManager.createWritable(targetDirectory);
			status = DirectoryStatus.Writable;
		}

		if (status == DirectoryStatus.Finalized) {
			// Bereits vollständig synchronisiert, nichts zu tun
			return null;
		}
		MirrorAction action = new MirrorAction(sourceDirectory, targetDirectory);

		DataIdentificationDirWalk walk = DataIdentificationDirWalk.allDirectories(sourceDirectory);
		walk.setPublisherTask(_longTermTaskStatePublisherTask);
		walk.setStatusPrinter(new StatusPrinter() {
			@Override
			public String getStatusMessage(final String actionName, final Duration runtime, final ApproximationType approximationType, final long allContainerDirs, final long visitedContainerDirs, final long visitedContainerFiles) {
				return super.getStatusMessage(actionName, runtime, approximationType, allContainerDirs, visitedContainerDirs, visitedContainerFiles) +
						getMessage(action, runtime);
			}

			@Override
			public String getSuccessMessage(final String actionName, final Duration runtime, final long visitedContainerDirs, final long visitedContainerFiles) {
				return super.getSuccessMessage(actionName, runtime, visitedContainerDirs, visitedContainerFiles) +
						getMessage(action, runtime);
			}
		});
		walk.execute("Datenkonsistentes Backup von " + sourceDirectory + " nach " + targetDirectory, 1, action);


		if (action.isSuccessful()) {
			if (!(sourceDirectory instanceof ActivePersistenceDirectory)) {
				// Abgeschlossenes Verzeichnis wurde vollständig synchronisiert
				lockFileManager.writeProtectPermanently(targetDirectory);
			}
			lockFileManager.close(targetDirectory);
		}

		return action;
	}

	/**
	 * Formatiert eine Debug-Meldung über den Fortschritt
	 *
	 * @param action  Aktuelle Spiegelungs-Aktion (enthält Statisktik-Daten)
	 * @param runtime Bisherige Laufzeit
	 * @return Formatierter String
	 */
	@NotNull
	private static String getMessage(final MirrorAction action, final Duration runtime) {
		String byteSize = formatByteSize(action.getNumBytesCopied());
		String speed = formatSpeed(action.getNumBytesCopied() / (runtime.getSeconds() + runtime.getNano() / 1000000000.0));
		return "\n " + getFileCountString(action.getNumFilesCopied()) + " (" + byteSize + "; " + speed + ") kopiert."
				+ "\n " + getFileCountString(action.getNumFilesDeleted()) + " gelöscht.";
	}

	/**
	 * Formatiert einen Byte-Pro-Sekunde-Wert
	 *
	 * @param bytesPerSecond Byte pro Sekunde
	 * @return String-Darstellung
	 */
	private static String formatSpeed(final double bytesPerSecond) {
		if (!Double.isFinite(bytesPerSecond)) return "?";

		double bytes_double = bytesPerSecond;
		for (final String unit : BYTE_UNIT) {
			if (bytes_double < 1024L) {
				return String.format("%.1f", bytes_double) + unit + "/s";
			}
			bytes_double /= 1024;
		}
		return bytesPerSecond + BYTE_UNIT[0] + "/s";
	}

	/**
	 * Formatiert eine Datenmenge in Bytes
	 *
	 * @param bytes Bytes
	 * @return String-Darstellung
	 */
	@NotNull
	private static String formatByteSize(final long bytes) {
		double bytes_double = bytes;
		for (final String unit : BYTE_UNIT) {
			if (bytes_double < 1024L) {
				return String.format("%.1f", bytes_double) + unit;
			}
			bytes_double /= 1024;
		}
		return bytes + BYTE_UNIT[0];
	}

	/**
	 * Formatiert eine Dateianzahl
	 *
	 * @param numFiles Anzahl dateien (>= 0)
	 * @return einen Wert wie "3 Dateien"
	 */
	@NotNull
	private static String getFileCountString(final long numFiles) {
		if (numFiles == 0) {
			return "Keine Dateien";
		} else if (numFiles == 1) {
			return "Eine Datei";
		}
		return numFiles + " Dateien";
	}

	/**
	 * Schreibt eine StartUpInfo.properties-Datei ins Spiegelungsverzeichnis. Die Werte werden aus dem aktuellen Mirror-Lauf aggregiert
	 *
	 * @param action        Mirrorlauf
	 * @param targetBaseDir Zielverzeichnis (wurzel)
	 * @throws PersistenceException Bei Schreibfehlern
	 */
	private void writeStartUpInfo(final MirrorAction action, final Path targetBaseDir) throws PersistenceException {
		StartupProperties startupProperties = new StartupProperties(targetBaseDir);
		startupProperties.setVal(StartupProperties.STUP_LAST_ATIME, action.getLastATime());
		startupProperties.setVal(StartupProperties.STUP_MAX_CONT_ID, action.getMaxContainerId());
		startupProperties.writeStartUpProperties();
	}

	/**
	 * Schreibt die Datei, die enthält bis zu welchem Archivzeitpunkt die angegebenen Daten gültig waren (um beim Neustart Datenlücken korrekt zu bilden)
	 *
	 * @param action        Mirrorlauf
	 * @param targetBaseDir Zielverzeichnis (wurzel)
	 */
	private void writeRestartTime(final MirrorAction action, final Path targetBaseDir) {
		getPersistenceManager().saveUnsubscriptionTime(action.getArchiveTimes(), targetBaseDir.resolve(PersistenceManager.RESTART_TIME_FILE_NAME));
	}

	/**
	 * Führt einen externen Prozess aus
	 *
	 * @param targetBaseDir Zielverzeichnis
	 * @param commandLine   Kommandozeile
	 * @throws IOException          IO-Fehler
	 * @throws InterruptedException Interrupted beim Warten auf Beendigung
	 */
	private static void runExternalProcess(final Path targetBaseDir, final String[] commandLine) throws IOException, InterruptedException {

		_debug.info("Führe Befehl aus: '" + Joiner.on("' '").join(commandLine) + "'");

		ProcessBuilder processBuilder = new ProcessBuilder().command(commandLine);
		processBuilder.directory(targetBaseDir.toFile());
		processBuilder.inheritIO();
		Process process = processBuilder.start();
		process.waitFor();
	}

	/**
	 * Eigentliche Implementierung der Spiegelungs-Aktion
	 */
	private final class MirrorAction implements DataIdentificationDirAction {

		/**
		 * Zielverzeichnis
		 */
		private final Path _targetBaseDir;

		/**
		 * Statistik
		 */
		private final AtomicLong _lastATime = new AtomicLong();

		/**
		 * Statistik
		 */
		private final AtomicLong _maxContainerId = new AtomicLong();

		/**
		 * Statistik
		 */
		private final AtomicLong _numFilesDeleted = new AtomicLong();

		/**
		 * Statistik
		 */
		private final AtomicLong _numBytesCopied = new AtomicLong();

		/**
		 * Statistik
		 */
		private final AtomicLong _numFilesCopied = new AtomicLong();

		/**
		 * Wird auf false gesetzt, falls ein Fehler auftrat
		 */
		private boolean _successful = true;

		private final Multimap<Long, IdDataIdentification> _archiveTimes = Multimaps.synchronizedMultimap(MultimapBuilder.treeKeys().arrayListValues().build());

		private final PersistenceDirectory _persistenceDirectory;

		/**
		 * Erzeugt eine neue Instanz
		 *
		 * @param sourceDirectory Quellverzeichnis
		 * @param targetDirectory Zielverzeichnis
		 */
		private MirrorAction(PersistenceDirectory sourceDirectory, final Path targetDirectory) {
			_persistenceDirectory = sourceDirectory;
			this._targetBaseDir = targetDirectory;
		}

		/**
		 * Wird für jedes Datenidentifikationsverzeichnis ausgeführt
		 *
		 * @param dataIdentificationDir DataIdentificationDir
		 * @param walk                  Aktueller Durchlauf
		 */
		@Override
		public void run(final DataIdentificationDir dataIdentificationDir, final DataIdentificationDirWalk walk) {
			if (shouldTerminate()) {
				_successful = false;
				walk.terminate();
				return;
			}
			suspendTaskIfNecessary();
			_archiveTimes.put(System.currentTimeMillis(), dataIdentificationDir.getDataIdentification());
			try (SyncKey<IdDataIdentification> indexLock = getPersistenceManager().lockIndex(dataIdentificationDir.getDataIdentification())) {
				for (LockedContainerDirectory directory : dataIdentificationDir.getContainerDirectories(indexLock)) {
					new MirrorDirectoryAction(_persistenceDirectory, directory).execute();
				}
			} catch (IOException e) {
				_successful = false;
				_debug.warning("Verzeichnis " + dataIdentificationDir + " konnte nicht vollständig gesichert werden. Fehler beim Kopieren der Dateien", e);
			} catch (Exception e) {
				_successful = false;
				_debug.warning("Verzeichnis " + dataIdentificationDir + " konnte nicht vollständig gesichert werden.", e);
			}
		}

		public long getLastATime() {
			return _lastATime.get();
		}

		public long getMaxContainerId() {
			return _maxContainerId.get();
		}

		public Multimap<Long, IdDataIdentification> getArchiveTimes() {
			return _archiveTimes;
		}

		public long getNumFilesCopied() {
			return _numFilesCopied.get();
		}

		public long getNumFilesDeleted() {
			return _numFilesDeleted.get();
		}

		public long getNumBytesCopied() {
			return _numBytesCopied.get();
		}

		public boolean isSuccessful() {
			return _successful;
		}

		/**
		 * Aktion für ein Containerdateienverzeichnis, hier findet die eigentliche Kopier-Aktion statt
		 */
		private class MirrorDirectoryAction {
			private final LockedContainerDirectory containerDirectory;
			private final PersistenceDirectory persistenceDirectory;

			/**
			 * Konstruktor
			 *
			 * @param persistenceDirectory Persistenzverzeichnis
			 * @param containerDirectory   Referenz auf die gelockte Datenidentifikation und Datenart für den Zugriff auf Containerdaten
			 */
			public MirrorDirectoryAction(PersistenceDirectory persistenceDirectory,
			                             LockedContainerDirectory containerDirectory) {
				this.persistenceDirectory = persistenceDirectory;
				this.containerDirectory = containerDirectory;
			}

			/**
			 * Führt das aktualisieren und kopieren der daten durch
			 *
			 * @throws Exception diverse Fehler
			 */
			public void execute() throws Exception {
				if (persistenceDirectory instanceof ActivePersistenceDirectory activePersistenceDirectory) {
					// Siehe Feinkonzept
					flushCache(activePersistenceDirectory);
					refreshIndex(activePersistenceDirectory);

					_lastATime.accumulateAndGet(activePersistenceDirectory.maxATime(containerDirectory), Math::max);
					_maxContainerId.accumulateAndGet(persistenceDirectory.maxContainerId(containerDirectory), Math::max);
				}

				// Zugriff sperren ist bereits durch Lock erfolgt
				copyFiles();
			}

			/**
			 * Schribt den Containerdateien-Schreibcache auf die Festplatte
			 *
			 * @throws Exception IO-Fehler
			 */
			private void flushCache(ActivePersistenceDirectory persistenceDirectory) throws Exception {
				long openContID = persistenceDirectory.getOpenContID(containerDirectory);
				if (openContID != -1) {
					CacheManager.Cache cache = CacheManager.getInstance().getCache(openContID);
					if (cache != null) {
						cache.flush();
					}
				}
			}

			/**
			 * Schreibt die Indexdateien im aktuellen Stand auf die Festplatte.
			 *
			 * @param persistenceDirectory Persistenzverzeichnis
			 */
			private void refreshIndex(ActivePersistenceDirectory persistenceDirectory) {
				persistenceDirectory.flushIndexes(containerDirectory);
			}

			/**
			 * Kopiert die eigentlichen Dateien ins zielverzeichnis (ähnlich rsync)
			 *
			 * @throws IOException IO-Fehler
			 */
			private void copyFiles() throws IOException {
				Path path = persistenceDirectory.getPath(containerDirectory);
				Path targetPath = _targetBaseDir.resolve(persistenceDirectory.getBasePath().relativize(path));

				Files.createDirectories(targetPath);

				// Flag-Datei anlegen, damit Zielverzeichnis als Inkonsistent betrachtet wird, falls beim Kopieren das Archivsystem abstürzt o.ä.
				Path rebuildFlagFile = targetPath.resolve(ActivePersistenceDirectory.REBUILD_INDEX_FILE_FLAG_NAME);
				if (!Files.exists(rebuildFlagFile)) {
					Files.createFile(rebuildFlagFile);
				}

				SyncResult syncResult = new FileSync() {
					@Override
					protected boolean shouldIgnore(final String fileName) {
						if (ActivePersistenceDirectory.REBUILD_INDEX_FILE_FLAG_NAME.equals(fileName) || "_backupindex.idx".equals(fileName)) {
							return true;
						}
						return fileName.endsWith(".defect");
					}
				}.syncDirectories(path, targetPath);

				_numFilesCopied.addAndGet(syncResult.getFilesCopied());
				_numFilesDeleted.addAndGet(syncResult.getFilesRemoved());
				_numBytesCopied.addAndGet(syncResult.getBytesCopied());

				// Flag-Datei wieder löschen, falls alles bis hier geklappt hat
				Files.deleteIfExists(rebuildFlagFile);
			}
		}
	}

}
