/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2019 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.tasks;

import de.bsvrz.ars.ars.mgmt.ArchiveManager;
import de.bsvrz.ars.ars.persistence.util.SignalingQueue;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.sys.funclib.losb.ringbuffer.RingBuffer;

import java.util.ArrayList;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * Verwaltet eine Warteschlange und eine Liste von Tasks. Ein Auftrag aus der Warteschlage wird dem nächsten freien Task zugewiesen. Wenn kein Task frei ist,
 * wartet der MultiTaskManager.
 *
 * @author beck et al. projects GmbH
 * @author Alexander Schmidt
 * @version $Revision$ / $Date$ / ($Author$)
 */
public final class MultiTaskManager extends QueueTask<ResultData> {

	/**
	 * Warteschlange mit Sub-Tasks, die aktuell nichts tun.
	 */
	private final RingBuffer<MultiTask> freeTaskQueue;

	/**
	 * Liste mit allen Sub-Tasks
	 */
	private final ArrayList<MultiTask> taskList = new ArrayList<>();

	/** So viele Tasks sind beim nächsten Durchlauf zu entfernen */
	private final AtomicInteger adjustTaskNumFactor = new AtomicInteger(0);

	/**
	 * Erstellt einen neuen MultiTaskManager
	 * @param archiveMgr Archiv-Manager
	 * @param inputData Queue, in die von einem anderne THread zu verarbeitende Daten eingefügt werden
	 * @param taskName Name
	 */
	public MultiTaskManager(ArchiveManager archiveMgr, SignalingQueue<ResultData> inputData, String taskName) {
		super(archiveMgr, inputData);
		setName(getName() + "[" + taskName + "]");
		freeTaskQueue = new RingBuffer<>(16, RingBuffer.UNBOUNDED_SIZE);	// waechst/schrumpft blockweise um 16
	}

	@Override
	public void start() {
		// Auch die Kind-Tasks starten
		synchronized(taskList) {
			for(MultiTask task : taskList) {
				task.start();
			}
		}
		super.start();
	}

	/**
	 * Fuegt ein MultiTask hinzu und startet ihn, falls der MultitaskManager selbst schon gestartet ist.
	 *
	 * @param task Kind-Task
	 */
	public synchronized void addTask(MultiTask task) {
		if(shouldTerminate()) return;

		synchronized(taskList) {
			taskList.add(task);
			task.setTaskIndex(taskList.size() - 1);
		}

		// Initialisierung der neuen Tasks
		ArchiveManager archMgr = getArchMgr();
		if (archMgr != null && archMgr.getDataModel() != null) {
			task.getObjectsFromDav();
		}

		// nur starten, wenn der MultitaskManager selbst schon gestartet ist:
		if(!(getState() == Thread.State.NEW || getState() == Thread.State.TERMINATED)) task.start();
		pushFreeTask(task);
	}

	/** Entfernt einen Subtask aus der Liste der verfuegbaren Tasks. Es verbleibt immer mindestens ein Task. */
	public synchronized void removeTask() {
		if(getTaskNum() > 1) adjustTaskNumFactor.incrementAndGet();		// ...und bei der nächsten Zuweisung überspringen
	}

	/**
	 * Gibt die Anzahl der aktuell verwalteten Sub-Tasks zurück
	 * @return Anzahl
	 */
	public int getTaskNum() {
		synchronized(taskList) {
			return taskList.size() - adjustTaskNumFactor.get();		// die zu entfernenden abziehen...
		}
	}

	/**
	 * Diese Methode wird von einem Task aufgerufen, wenn dieser Task mit der Bearbeitung eines Auftrages fertig ist und einen neuen Auftrag entgegennehmen kann.
	 *
	 * @param task Freier Task
	 */
	void pushFreeTask(MultiTask task) {
		if(!shouldTerminate()) freeTaskQueue.push(task);
	}

	@Override
	protected void cleanUp() {
		// Kind-Tasks erst dann terminieren, wenn die eigene Queue abgearbeitet wurde
		// daher hier terminieren und nicht in this.terminateTask
		synchronized(taskList) {
			for(MultiTask task : taskList) {
				task.terminateTask();
			}
		}
	}

	@Override
	protected void work(final ResultData step) throws InterruptedException {
		MultiTask nextWorkingTask;
		while(true) {
			nextWorkingTask = freeTaskQueue.pop();		// synchronized auf freeTaskQueue, blockiert falls leer

			if(adjustTaskNumFactor.get() > 0) {			// Wenn Tasks entfernt werden sollen, werden sie hier
				adjustTaskNumFactor.decrementAndGet();	// übersprungen und beendet.
				nextWorkingTask.terminateTask();
				synchronized(taskList) {
					taskList.remove(nextWorkingTask);	// aus der statischen Liste entfernen
				}
				continue;
			}

			if(nextWorkingTask.shouldTerminate()) {
				// Keinem abgebrochenen Task eine Aufgabe zuweisen!!
				continue;
			}
			break;
		}
		nextWorkingTask.submit(step);						// nextWorkingTask aufwecken

		taskStepDone();
	}

	@Override
	public void join() throws InterruptedException {
		// Darauf warten, dass alle Kind-Tasks sich beendet haben.
		ArrayList<MultiTask> tmp;
		synchronized(taskList) {
			tmp = new ArrayList<>(taskList);
		}
		for(MultiTask multiTask : tmp) {
			multiTask.join();
		}
		super.join();
	}

	public String toString() {
		StringBuilder sb = new StringBuilder(2048);
		synchronized(taskList) {
			for(MultiTask mt : taskList) {
				sb.append(mt != null ? (mt.getName() + ":  " + mt.getState()) : "null");
				sb.append("\n");
			}
		}
		return sb.toString();
	}

	@Override
	public boolean isAlive() {
		// Der MultiTaskManager lebt noch, solange er selbst oder mindestens eins der Subtasks läuft
		return super.isAlive() || taskList.stream().anyMatch(AbstractTask::isAlive);
	}

	public void getObjectsFromDav() {
		// Weiterleiten
		synchronized(taskList) {
			for(MultiTask multiTask : taskList) {
				multiTask.getObjectsFromDav();
			}
		}
	}
}
