/*
 * Copyright 2019-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.mgmt.tasks;

import de.bsvrz.ars.ars.mgmt.ArchiveManager;
import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SynchronizationFailedException;
import de.bsvrz.ars.ars.persistence.*;
import de.bsvrz.ars.ars.persistence.iter.DataIterator;
import de.bsvrz.ars.ars.persistence.iter.QueryDataSequence;
import de.bsvrz.ars.ars.persistence.writer.ArchiveTask;
import de.bsvrz.dav.daf.main.DataState;
import de.bsvrz.dav.daf.main.archive.*;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.bsvrz.sys.funclib.losb.util.Util;

import java.util.Arrays;

/**
 * Klasse, die eine einzelne Unterabfrage ({@link ArchiveDataSpecification}) aus der Archivanfrage und die zugehoerigen Daten für die Archivantwort kapselt.
 */
public class Query implements AutoCloseable {

	/**
	 * Die Archivanfrage.
	 */
	private final ArchiveDataSpecification arcDataSpec;

	/**
	 * Der Zeit-/Indexbereich der Archivanfrage.
	 */
	private final ArchiveTimeSpecification timeSpec;

	/**
	 * Flag, ob Anfrage bereits erledigt ist. Dies bedeutet, es sind alle Ergebnis-Datensätze und der Kennzeichen-Datensatz am Ende der Antwort versandt worden.
	 */
	volatile boolean done;

	/**
	 * Enthält den letzten Datensatz einer aktuellen Datensatzart für den Deltavergleich. War der letzte versandte Datensatz ein nachgelieferter, ist hier
	 * {@code null} enthalten.
	 */
	private SimpleArchiveData lastDataForDelta;

	/**
	 * Temporäres Result-Objekt
	 */
	private final ContainerDataResult _result = new ContainerDataResult();

	/**
	 * Ist schon der Ende-Archiv-Datensatz verschickt worden?
	 */
	private boolean _eoaReturned;

	/**
	 * Persistenz
	 */
	private final PersistenceManager persMgr;

	private final DataIterator _iterator;

	/**
	 * Erzeugt eine neue Query mit dem angegebenen Index zur Bearbeitung der angegebenen Unterabfrage.
	 *
	 * @param archMgr                Archivmanager
	 * @param ads                    Unterabfrage, die diese Query bearbeitet
	 * @param requestedIntervalError Zeigt an, ob die Grenzen des Abfrage-Intervalls vertauscht sind
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	Query(ArchiveManager archMgr, ArchiveDataSpecification ads, boolean requestedIntervalError) throws PersistenceException {
		this.arcDataSpec = ads;
		this.timeSpec = ads.getTimeSpec();
		this.persMgr = archMgr.getPersistenceManager();

		IdDataIdentification dataIdentification = new IdDataIdentification(ads);
		try {
			_iterator = getDataIterator(requestedIntervalError, dataIdentification);
		} catch (Exception e) {
			throw new PersistenceException(e);
		}
	}

	private DataIterator getDataIterator(final boolean requestedIntervalError,
	                                     final IdDataIdentification idDataIdentification) throws PersistenceException, SynchronizationFailedException {
		final DataIterator iterator;
		if (requestedIntervalError) {
			iterator = DataIterator.empty();
		} else {
			final QueryDataSequence sequence = new QueryDataSequence(persMgr, arcDataSpec.getDataKinds(), timeSpec, arcDataSpec.getSortOrder(), idDataIdentification);
			iterator = sequence.iterator();
		}
		return iterator;
	}

	/**
	 * Liefert den nächsten Ergebnis-Datensatz der Unterabfrage zurück. Dabei werden die
	 * Ergebnis-Datensätze der verschiedenen Datensatzarten gemischt.
	 *
	 * @param dataIterator Daten-Iterator, der Datensätze zurückgibt
	 * @return Nächster Ergebnisdatensatz, {@code null} falls keiner mehr zu liefern ist.
	 */
	@Nullable
	private SimpleArchiveData getNextData(DataIterator dataIterator) {
		if (arcDataSpec.getRequestOption() == ArchiveRequestOption.DELTA) {
			SimpleArchiveData tmp = getNextDataRegular(dataIterator);
			while (containSameData(lastDataForDelta, tmp)) {
				// Wegen Deltaanfrage ignorieren
				tmp = getNextDataRegular(dataIterator);
			}
			lastDataForDelta = tmp;
			return tmp;
		} else {
			return getNextDataRegular(dataIterator);
		}
	}

	/**
	 * Liefert den nächsten Ergebnis-Datensatz der Unterabfrage zurück. Dabei werden die ErgebnisDatensätze der verschiedenen Datensatzarten gemischt.
	 *
	 * @return Nächster Ergebnisdatensatz, {@code null} falls keiner mehr zu liefern ist.
	 */
	@Nullable
	private SimpleArchiveData getNextDataRegular(DataIterator iterator) {
		boolean deleted;
		final String medId;
		try {
			if (iterator.isEmpty()) {
				return getEndOfArchive(iterator); // Oder null wenn bereits gesendet
			}
			deleted = iterator.getContainerManagementData().getContainerHeaderParamAsBoolean(ContainerManagementInformation.CHP_DELETED);
			medId = iterator.getContainerManagementData().getContainerHeaderParamAsString(ContainerManagementInformation.CHP_MEDIUM_ID);

			iterator.peek(_result);
			iterator.remove();
		} catch (PersistenceException | SynchronizationFailedException e) {
			Task._debug.error("Fehler beim Durchführen der Archivanfrage", e);
			return null;
		}

		if (deleted) {
			DataState dataState;
			if (medId.equals(ContainerHdr.MEDIUM_UNSAVED) || medId.equals(ContainerHdr.MEDIUM_DESTROYED)) {
				dataState = DataState.DELETED_BLOCK;
			} else {
				dataState = DataState.UNAVAILABLE_BLOCK;
			}
			return new SimpleArchiveData(
					_result.getArchiveTime(),
					_result.getDataTime(),
					_result.getDataIndex(),
					_result.getDataKind(),
					dataState,
					false,
					null
			);
		}
		return new SimpleArchiveData(
				_result.getArchiveTime(),
				_result.getDataTime(),
				_result.getDataIndex(),
				_result.getDataKind(),
				_result.getDataState(),
				_result.isCompressed(),
				_result.getData()
		);

	}

	@Nullable
	private SimpleArchiveData getEndOfArchive(DataIterator iterator) throws PersistenceException, SynchronizationFailedException {
		if (!_eoaReturned) {
			_eoaReturned = true;
			ContainerDataResult next = iterator.peekNext();
			if (next != null) {
				// Es gibt einen Datensatz, der auf den letzten gesendeten Datensatz folgt
				// Dieser wird benutzt, um das Gültigkeitsende des zuletzt gelieferten Datensatzes zu bestimmen
				// und als Ende-Archiv zurückzuliefern
				long endATime = next.getArchiveTime();
				long endDTime = next.getDataTime();
				long endIndex = _result.getDataIndex();

				if (timeSpec.getTimingType() == TimingType.ARCHIVE_TIME) {
					endATime = Math.min(endATime, timeSpec.getIntervalEnd());
				}
				if (timeSpec.getTimingType() == TimingType.DATA_TIME) {
					endDTime = Math.min(endDTime, timeSpec.getIntervalEnd());
				}
				if (timeSpec.getTimingType() == TimingType.DATA_INDEX) {
					endIndex = Math.min(endIndex, timeSpec.getIntervalEnd());
				}
				return new SimpleArchiveData(
						endATime,
						endDTime,
						Util.dIdxSetArSBit(endIndex),
						next.getDataKind(),
						DataState.END_OF_ARCHIVE,
						false,
						null
				);
			} else {
				if (_result.getDataKind() == null) {
					// es wurde noch kein Datensatz geliefert
					ArchiveDataKindCombination dataKinds = arcDataSpec.getDataKinds();
					ArchiveDataKind dataKind = dataKinds.isEmpty() ? ArchiveDataKind.ONLINE : dataKinds.iterator().next();
					long time = ArchiveTask.getLastArchiveTime();
					/* Wenn zu der entsprechenden Datenidentifikation noch nie ein Datensatz archiviert wurde,
					   wird eine Ende-Kennung übertragen, die als Archiv und Datenzeitstempel die aktuelle Zeit
					   und als Datenindex nur das gesetzte Archivbit enthält.*/
					return new SimpleArchiveData(
							time,
							time,
							Util.dIdxSetArSBit(0),
							dataKind,
							DataState.END_OF_ARCHIVE,
							false,
							null
					);
				}
				long endATime = ArchiveTask.getLastArchiveTime();
				long endDTime = _result.getDataTime();
				long endIndex = _result.getDataIndex();

				if (timeSpec.getTimingType() == TimingType.ARCHIVE_TIME) {
					endATime = Math.min(endATime, timeSpec.getIntervalEnd());
				}
				if (timeSpec.getTimingType() == TimingType.DATA_TIME) {
					endDTime = Math.min(endDTime, timeSpec.getIntervalEnd());
				}
				if (timeSpec.getTimingType() == TimingType.DATA_INDEX) {
					endIndex = Math.min(endIndex, timeSpec.getIntervalEnd());
				}
				return new SimpleArchiveData(
						endATime,
						endDTime,
						Util.dIdxSetArSBit(endIndex),
						_result.getDataKind(),
						DataState.END_OF_ARCHIVE,
						false,
						null
				);
			}

		}
		return null;
	}

	/**
	 * Bricht die Query ab und markiert sie als erledigt. Noch geoeffnete Container werden geschlossen.
	 */
	@Override
	public void close() {
		_iterator.close();
		done = true;
	}

	/**
	 * Vergleicht die beiden angegebenen Ergebnis-Datensätze auf Gleichheit der Nutzdaten für die Deltaantwort.
	 *
	 * @param arcData1 Erster Datensatz
	 * @param arcData2 Zweiter Datensatz
	 * @return Flag, ob die beiden Datensätze gleich sind
	 */
	private static boolean containSameData(SimpleArchiveData arcData1, SimpleArchiveData arcData2) {
		return arcData1 != null
				&& arcData2 != null
				&& arcData1.dataState().equals(arcData2.dataState())
				&& arcData1.dataState() == DataState.DATA
				&& !arcData1.dataKind().isDelayed()
				&& !arcData2.dataKind().isDelayed()
				&& Arrays.equals(arcData1.rawData(), arcData2.rawData());
	}


	public SimpleArchiveData take() {
		return getNextData(_iterator);
	}
}
