/*
 * Copyright 2019-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.mgmt.tasks;

import de.bsvrz.ars.ars.mgmt.ArchiveManager;
import de.bsvrz.dav.daf.main.archive.ArchiveDataSpecification;
import de.bsvrz.sys.funclib.communicationStreams.StreamMultiplexer;
import de.bsvrz.sys.funclib.debug.Debug;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

/**
 * Anfrage die aus mehreren Datenspezifikationen besteht und grundsätzlich beantwortet werden kann
 */
public final class QueryList implements QueryData {

	private static final Debug _debug = Debug.getLogger();

	private final List<Query> queries = new ArrayList<>();

	/**
	 * Erstellt eine neue QueryList.
	 *
	 * @param archMgr Archiv-manager
	 * @param ads     Archivdatenspezifikationen
	 */
	QueryList(ArchiveManager archMgr, ArchiveDataSpecification[] ads) {
		try {
			for (int i = 0; i < ads.length; i++) {
				boolean intervalErr = ads[i].getTimeSpec().getIntervalStart() > ads[i].getTimeSpec().getIntervalEnd();
				queries.add(i, new Query(archMgr, ads[i], intervalErr));
			}
		} catch (Throwable e) {
			close();
			_debug.error("Fehler beim Initialisieren der QueryList", e);
			throw new RuntimeException(e);
		}
	}


	@Override
	public void close() {
		// Synchronisiert sich mit der Methode dataRequest und verhindert, dass bereits die Verarbeitung einer neuen Anfrage begonnen wird
		// während die Sendesteuerung noch verarbeitet wird.
		synchronized (queries) {
			// Alle Unterabfragen verwerfen
			for (Query query : queries) {
				try {
					query.close();
				} catch (Exception e) {
					_debug.warning("Fehler beim Schließen eines Archivanfragen-Iterators", e);
				}
			}
		}
	}

	@Override
	public int size() {
		return queries.size();
	}

	@Override
	public void sendReply(ArchiveQueryTask.ResponseSender sender, StreamMultiplexer mux) throws IOException, InterruptedException {
		sender.sendInitialResponse(true, "");
		if (mux != null) {
			mux.sendAllStreamData();
		}
	}

	@Override
	public Query get(int indexOfStream) {
		return queries.get(indexOfStream);
	}
}
