/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2019 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.tasks;

import de.bsvrz.ars.ars.mgmt.ArchiveManager;
import de.bsvrz.ars.ars.mgmt.tasks.base.TimeoutManager;
import de.bsvrz.dav.daf.main.*;
import de.bsvrz.dav.daf.main.archive.*;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.dataSerializer.Deserializer;
import de.bsvrz.sys.funclib.dataSerializer.NoSuchVersionException;
import de.bsvrz.sys.funclib.dataSerializer.Serializer;
import de.bsvrz.sys.funclib.dataSerializer.SerializingFactory;
import de.bsvrz.sys.funclib.losb.datk.MessageType;
import de.bsvrz.sys.funclib.losb.datk.PidQuery;
import de.bsvrz.sys.funclib.losb.kernsoftware.ConnectionManager;
import de.bsvrz.sys.funclib.losb.util.Util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;

/**
 * Oberklasse für Tasks, die Archivanfragen oder Archiv-Informationsanfragen bearbeiten.
 *
 * @author beck et al. projects GmbH
 * @author Alexander Schmidt
 * @version $Revision$ / $Date$ / ($Author$)
 */
public abstract class QueryTask extends MultiTask implements ClientSenderInterface {

	/**
	 * Wartet darauf, dass es eine positive Sendesteuerung für das Verschicken der Antworten gibt.
	 * Diese Methode wartet so lange, wie eine negative Sendesteuerung besteht.
	 *
	 * @return In der Regel {@link SendingState#SEND}, außer es tritt ein Fehler oder Timeout auf.
	 * Der Aufrufer muss den Rückgabewert auswerten und entsprechend reagieren.
	 * @throws InterruptedException Unterbrochen
	 */
	protected SendingState waitForSendControl() throws InterruptedException {
		final SendingState state;// Nach X Sekunden ohne Sendesteuerung wird der Auftrag abgebrochen
		final long timeoutTime = System.currentTimeMillis() + TimeoutManager.STANDARD_TIMEOUT;
		synchronized (syncObj) {
			while (sendingState == SendingState.STOP) {
				final long delta = timeoutTime - System.currentTimeMillis();
				if (delta <= 0) break;
				syncObj.wait(delta);
			}
			state = sendingState;
		}
		return state;
	}

	/**
	 * Zustand der Sendesteuerung (Senden / Anhalten / Abbrechen).
	 */
	protected enum SendingState {

		SEND,
		STOP,
		ABORT
	}

	/**
	 * Aktueller Zustand der Sendesteuerung.
	 */
	protected SendingState sendingState;

	/**
	 * Spezielles Objekt zur Synchronisiserung. Immer dieses anstatt this verwenden!
	 */
	protected final Object syncObj = new Object();

	/**
	 * Applikations-Objekt des Absenders der Archiv-Anfrage.
	 */
	protected SystemObject queryAppObj;

	/**
	 * Attributgruppe für Archiv-Anfragen.
	 */
	protected AttributeGroup atgQuery;

	/**
	 * Datenidentifikation der Archiv-Antwort.
	 */
	protected DataDescription ddResponse;

	/**
	 * Anfrage-Index der anfragenden Applikation zur Archiv-Anfrage.
	 */
	protected int queryIdx;

	/**
	 * Serialisiererversion der anfragenden Applikation, mit der auch die Antwort erstellt wird.
	 */
	protected int serializerVersion;

	/**
	 * Dieses Datenobjekt wird zum Senden des Ergebnisses verwendet
	 */
	protected Data gData;

	/**
	 * Wird zum Senden verwendet (muss nur einmal angelegt werden)
	 */
	protected ResultData gResultData;

	/**
	 * Wiederverwendbarer ByteArrayOutputStream.
	 */
	protected final ByteArrayOutputStream bosResult = new ByteArrayOutputStream(16 * 1024);

	/**
	 * Der Serialisierer aus DAF.
	 */
	protected Serializer serializer;

	/**
	 * Der Deserialisierer aus DAF.
	 */
	protected Deserializer deserializer;


	/**
	 * Konstruktor für den QueryTask.
	 *
	 * @param archiveMgr Der Archiv-Manager.
	 * @param tMgr       Der MultiTaskManager.
	 */
	public QueryTask(ArchiveManager archiveMgr, MultiTaskManager tMgr) {
		super(archiveMgr, tMgr);
	}

	@Override
	public void getObjectsFromDav() {
		DataModel dataModel = getArchMgr().getDataModel();
		if (dataModel == null) {
			throw new IllegalStateException();
		}
		atgQuery = dataModel.getAttributeGroup(PidQuery.ATG_QUERY_PID);
		ddResponse = new DataDescription(atgQuery, dataModel.getAspect(PidQuery.ASP_RESPONSE_PID));
	}

	/**
	 * Initialisiert den Task für eine neue Anfrage.
	 */
	protected void init() {
		queryAppObj = null;
		queryIdx = 0;
		serializerVersion = 0;
		sendingState = SendingState.STOP;
	}
//

	@Override
	public boolean isRequestSupported(SystemObject object, DataDescription dataDescription) {
		return true; // Sendesteuerung ist erwuenscht und wird in dataRequest() verarbeitet.
	}

	@Override
	public void dataRequest(SystemObject object, DataDescription dataDescription, byte state) {
		synchronized (syncObj) {
			if (state == ClientSenderInterface.START_SENDING) {
				sendingState = SendingState.SEND;
			} else if (sendingState == SendingState.SEND) {
				sendingState = SendingState.ABORT;
			} else {
				sendingState = SendingState.STOP;
			}
			syncObj.notifyAll();
		}
	}

	/**
	 * Meldet den Task als Sender für Archiv-Antworten an das angegebene Empfaenger-Applikations-Objekt an. Registriert den StreamMultiplexer für den Empfang
	 * von Flusskontroll-Steuerungspaketen.
	 *
	 * @throws OneSubscriptionPerSendData Anmeldung besteht bereits
	 */
	protected void subscribeSender() throws OneSubscriptionPerSendData {
		ConnectionManager.subscribeSender(getArchMgr().getDavCon(), this, queryAppObj, ddResponse, SenderRole.sender());
	}

	/**
	 * Meldet den Task als Sender für Archiv-Antworten an das angegebene Empfaenger-Applikations-Objekt ab.
	 */
	protected void unsubscribeSender() {
		try {
			ConnectionManager.unsubscribeSender(getArchMgr().getDavCon(), this, queryAppObj, ddResponse);
		} catch (Exception e) {
			_debug.warning(getName() + " Sende-Abmeldung fehlgeschlagen.", e);
		}
	}

	/**
	 * Legt alle Datenstrukturen an, die zu Beginn der Anfrage ermittelt werden können.
	 *
	 * @throws NoSuchVersionException Angefragte Serialisierer-Version ist nicht vorhanden
	 */
	protected void createQueryData() throws NoSuchVersionException {
		queryAppObj = resultData.getData().getReferenceValue(PidQuery.ATT_SENDER_NAME).getSystemObject();
		queryIdx = resultData.getData().getUnscaledValue(PidQuery.ATT_QUERY_IDX_NAME).intValue();

		gData = getArchMgr().getDavCon().createData(atgQuery);
		gData.getReferenceValue(PidQuery.ATT_SENDER_NAME).setSystemObject(getArchMgr().getArchiveObject());
		gData.getItem(PidQuery.ATT_QUERY_IDX_NAME).asUnscaledValue().set(queryIdx);

		byte[] dataArray = resultData.getData().getUnscaledArray(PidQuery.ATT_DATA_NAME).getByteArray();
		serializerVersion = Util.getSerVersion(dataArray); // Erste 4 Bytes = Serialisierer-Version
		bosResult.reset();
		serializer = SerializingFactory.createSerializer(serializerVersion, bosResult);

		InputStream stream = new ByteArrayInputStream(dataArray, 4, dataArray.length - 4);
		deserializer = SerializingFactory.createDeserializer(serializerVersion, stream);

		gResultData = new ResultData(queryAppObj, ddResponse, 0, gData);    // Datenzeit wird spaeter gesetzt
	}

	/**
	 * Sendet das übergebene Byte-Array in einem Ergebnis-Datensatz ({@link ResultData}) an das Empfaenger-Applikations-Objekt.
	 *
	 * @param resultBytes Zu versendendes Byte-Array
	 * @param msgType     ArchivNachrichtenTyp wie definiert in {@link MessageType}
	 */
	protected void sendResultData(byte[] resultBytes, int msgType) {
		try {
			gData.getItem(PidQuery.ATT_MESSAGE_TYP_NAME).asUnscaledValue().set(msgType);
			Data.Array data = gData.getArray(PidQuery.ATT_DATA_NAME);
			data.setLength(resultBytes.length);
			for (int i = 0; i < resultBytes.length; i++) {
				data.getItem(i).asUnscaledValue().set(resultBytes[i]);
			}

			gResultData.setDataTime(System.currentTimeMillis());
			getArchMgr().getDavCon().sendData(gResultData);
		} catch (Exception e) {
			_debug.error(getName() + " Daten-Versand an Empfaenger-Applikations-Objekt fehlgeschlagen", e);
		}
	}

	/**
	 * Interpretiert die empfangene Archivanfrage oder Archiv-Informationsanfrage und zerlegt sie in einzelne Unterabfragen vom Typ ArchiveDataSpecification.
	 *
	 * @param ds        Deserialisierer mit der Anfrage.
	 * @param dataModel Datenmodell
	 * @return Feld von Unterabfragen vom Typ ArchiveDataSpecification.
	 * @throws IOException Allgemeiner IO-Fehler
	 */
	protected static ArchiveDataSpecification[] parseArchiveDataSpec(Deserializer ds, DataModel dataModel) throws IOException {
		// Analyse der empfangenen Daten erfolgt hier gemäß
		// Datenkatalog Kapitel 7.2.3.12 ArchivAnfrageSchnittstelle

		int anzQueries = ds.readInt();        // Größe der Liste mit den ArchiveDataSpecification
		ArchiveDataSpecification[] result = new ArchiveDataSpecification[anzQueries];

		for (int i = 0; i < anzQueries; i++) {
			TimingType tt = Util.getTimingType(ds.readByte());               // Typ der Timingangabe

			boolean relative = ds.readByte() == 1;                // Intervallstart relativ
			long start = ds.readLong();                // Intervallstart
			long end = ds.readLong();                  // Intervallende

			ArchiveTimeSpecification timeSpec = new ArchiveTimeSpecification(tt, relative, start, end);

			boolean oa = ds.readByte() == 1;             // Online aktuell
			boolean on = ds.readByte() == 1;             // Online nachgeliefert
			boolean na = ds.readByte() == 1;             // Nachgefordert aktuell
			boolean nn = ds.readByte() == 1;             // Nachgefordert nachgeliefert

			ArchiveDataKindCombination adkComb = Util.getADKCombination(oa, on, na, nn);

			int orderCode = ds.readInt();
			ArchiveOrder order = orderCode != 0 ? ArchiveOrder.getInstance(orderCode) : null;

			int reqOptCode = ds.readInt();
			ArchiveRequestOption reqOpt = reqOptCode != 0 ? ArchiveRequestOption.getInstance(reqOptCode) : null;

			AttributeGroup atg = (AttributeGroup) ds.readObjectReference(dataModel);
			Aspect asp = (Aspect) ds.readObjectReference(dataModel);
			short sv = ds.readShort();
			SystemObject obj = ds.readObjectReference(dataModel);

			result[i] = new ArchiveDataSpecification(timeSpec, adkComb, order, reqOpt, new DataDescription(atg, asp, sv), obj);
		}
		return result;
	}
}
