/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.tasks;

import de.bsvrz.ars.ars.mgmt.ArchiveManager;
import de.bsvrz.ars.ars.persistence.util.SignalingQueue;
import de.bsvrz.ars.ars.persistence.util.TerminatedException;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;

import java.util.concurrent.Executors;

/**
 * Abstrakte Basisklasse für die abstrakten Task-Basisklassen {@link de.bsvrz.ars.ars.mgmt.tasks.SingleTask} und {@link de.bsvrz.ars.ars.mgmt.tasks.MultiTask}.
 * <p>
 * Ein QueueTask ist ein Thread, der laufend eine bestimmte Aktion ausführt. Er ähnelt einem {@link Executors#newSingleThreadExecutor()}, die einzelnen Aufgaben
 * werden aber nicht als Runnable o.ä. übergeben, sondern der Task holt sich selbst die durchzuführenden Arbeitsschritte aus einer Queue. Die Klasse, auf der
 * ein Task arbeitet wird über den generischen Typ T definiert.
 *
 * @author beck et al. projects GmbH
 * @author Thomas Schaefer
 * @author Alexander Schmidt
 * @author Kappich Systemberatung
 * @version $Revision$ / $Date$ / ($Author$)
 */
public abstract class QueueTask<T> extends RepeatingTask {

	private final SignalingQueue<T> _queue;

	protected QueueTask(ArchiveManager archiveMgr, SignalingQueue<T> queue) {
		super(archiveMgr);
		_queue = queue;
	}

	/**
	 * Fügt einen abzuarbeitenden Auftrag in die Queue ein.
	 *
	 * @param step Datensatz
	 */
	public void submit(@NotNull T step) {
		_queue.add(step);
	}

	/**
	 * Terminiert den Task. Diese Methode fügt zusätzlich in die Queue einen speziellen Token ein, mit dem der Aufruf von
	 * {@code _queue.take()} in der {@link #step()}-Methode aufgeweckt wird, falls die Methode dort gerade auf einen neuen Auftrag warten.
	 */
	@Override
	public void terminateTask() {
		super.terminateTask();
		_queue.terminateNow();
	}

	@Override
	public final void step() throws InterruptedException {
		try {
			work(_queue.take());
		} catch (TerminatedException e) {
			return;
		}

		taskStepDone();
	}

	/**
	 * Vom konkreten Task zu implementierende Methode, die den Auftrag abarbeitet. Diese Methode sollte
	 * nur Intern aufgerufen werden, ist für Testfälle aber public. Normalerweise sollten Aufträge aber
	 * Asynchron mit submit() übergeben werden
	 *
	 * @param step Auftrag/Arbeitsschritt
	 * @throws InterruptedException Thread unterbrochen
	 */
	protected abstract void work(T step) throws InterruptedException;
}
