/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.tasks;

import com.google.common.collect.Range;
import de.bsvrz.ars.ars.mgmt.ArchiveManager;
import de.bsvrz.ars.ars.mgmt.datatree.DataIdentNode;
import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SynchronizationFailedException;
import de.bsvrz.ars.ars.mgmt.tasks.base.TaskManager;
import de.bsvrz.ars.ars.mgmt.tasks.base.TimeoutManager;
import de.bsvrz.ars.ars.mgmt.tasks.query.QueryHandler;
import de.bsvrz.ars.ars.persistence.IdDataIdentification;
import de.bsvrz.ars.ars.persistence.PersistenceException;
import de.bsvrz.ars.ars.persistence.gap.GapFile;
import de.bsvrz.ars.ars.persistence.gap.RemoteArchive;
import de.bsvrz.ars.ars.persistence.iter.DataIterator;
import de.bsvrz.ars.ars.persistence.iter.QueryDataSequence;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.DataState;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.archive.*;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.dataSerializer.Deserializer;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.bsvrz.sys.funclib.losb.datk.MessageType;
import de.bsvrz.sys.funclib.losb.datk.PidQuery;
import de.bsvrz.sys.funclib.losb.util.Util;

import java.io.IOException;
import java.util.*;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;

/**
 * Task zur Nachforderung fehlender Archivdaten aus anderen Archivsystemen gemäß Parametrierung.
 *
 * @author beck et al. projects GmbH
 * @author Alexander Schmidt
 * @version $Revision$ / $Date$ / ($Author$)
 */
public class RequestGapTask extends SingleTask {

	/** Prioritaet der Nachforderungsanfragen. */
	private static final ArchiveQueryPriority REQUEST_PRIO = ArchiveQueryPriority.LOW;

	private final ArchiveDataKindCombination requestAdk = ArchiveDataKindCombination.all();

	private ArchiveDataSpecification[] archDataSpecs;

	private List<RemoteArchive> archiveList;

	private final AtomicInteger totalGapsFound = new AtomicInteger(0);

	/**
	 * Zeitspezifikation, die beim Nachfordern verwendet werden soll oder {@code null}, falls keine Einschränkung berücksichtigt werden soll.
	 */
	private ArchiveTimeSpecification archTimeSpec;

	private final ReentrantLock archiveRequestedDataFinishedLock = new ReentrantLock();

	private final Condition archiveRequestedDataFinishedCondition = archiveRequestedDataFinishedLock.newCondition();

	private boolean requestedDataFinished = false;

	/**
	 * Erzeugt einen neuen Task zur Nachforderung fehlender Archivdaten.
	 * @param archiveMgr Der Archiv-Manager.
	 */
	public RequestGapTask(ArchiveManager archiveMgr) {
		super(archiveMgr);
	}

	/** 
	 * Liefert den Nachforderungsanfrage-Datensatz. 
	 * @return Dummy-Datensatz, der Nachforderung auslöst
	 */
	public static RequestGapResultData getRequestGapResultData() {
		return new RequestGapResultData();
	}

	/**
	 * Startet das automatische oder manuelle Nachfordern entsprechend dem übergebenen Nachforderungs-Datensatz.
	 *
	 * @param resultData Nachforderungs-Datensatz.
	 */
	@Override
	protected void work(ResultData resultData) {
		long t = Util.startTimer();
		totalGapsFound.set(0);
		long anzReqDS = getArchMgr().getInQueuesMgr().getQueuesCountRequested();
		if(resultData instanceof RequestGapResultData) {
			_debug.info(getName() + ": Automatische Nachforderung gestartet.");
			archTimeSpec = null;
			archDataSpecs = new ArchiveDataSpecification[0];
			archiveList = new ArrayList<>(0);
			try {
				performRequest();
				anzReqDS = getArchMgr().getInQueuesMgr().getQueuesCountRequested() - anzReqDS;
				_debug.info(
						getName() + ": Automatische Nachforderung beendet (Zeit: " + Util.stopTimer(t) + ", Luecken: " + totalGapsFound.get()
								+ ", erhaltene Datensaetze: " + anzReqDS + ")"
				);
			}
			catch(Exception e) {
				_debug.warning("Nachforderung fehlgeschlagen.", e);
			}
		}
		else {
			if(!resultData.hasData()) return;
			int msgType = resultData.getData().getUnscaledValue(PidQuery.ATT_MESSAGE_TYP_NAME).intValue();
			if(msgType == MessageType.REQUEST_DID) {
				int messageTypeResult = MessageType.REQUEST_DID_RESULT;
				_debug.info(getName() + ": Manuelle Nachforderung gestartet.");
				QueryHandler queryHandler = new QueryHandler(getName(), getArchMgr(), messageTypeResult, resultData.getData());
				try {
					archTimeSpec = null;
					Deserializer deserializer = queryHandler.getDeserializer();

					// ====== Analyse der empfangenen Daten erfolgt hier gemäß ======
					// ====== Datenkatalog Kapitel 7.2.3.12 ArchivAnfrageSchnittstelle ======

					archDataSpecs = QueryTask.parseArchiveDataSpec(deserializer, getArchMgr().getDataModel());
					archiveList = parseArchiveList(deserializer);
					performRequest();
					queryHandler.sendSuccessResponse(messageTypeResult);
					anzReqDS = getArchMgr().getInQueuesMgr().getQueuesCountRequested() - anzReqDS;
					_debug.info(
							getName() + ": Manuelle Nachforderung beendet (Zeit: " + Util.stopTimer(t) + ", Luecken: " + totalGapsFound.get()
									+ ", erhaltene Datensaetze: " + anzReqDS + ")"
					);
				}
				catch(Exception e) {
					_debug.warning("Nachforderung fehlgeschlagen.", e);
					queryHandler.sendErrorResponse(messageTypeResult, "Nachforderung fehlgeschlagen: " + e.getMessage());
				}
			}
			else if(msgType == MessageType.REQUEST_AUTOM) {
				int messageTypeResult = MessageType.REQUEST_AUTOM_RESULT;
				QueryHandler queryHandler = new QueryHandler(getName(), getArchMgr(), messageTypeResult, resultData.getData());
				try {
					_debug.info(getName() + ": Manueller Anstoß der automatischen Nachforderung.");
					archDataSpecs = new ArchiveDataSpecification[0];
					Deserializer deserializer = queryHandler.getDeserializer();
					final long startTime = deserializer.readLong();
					final long endTime = deserializer.readLong();
					archTimeSpec = new ArchiveTimeSpecification(TimingType.ARCHIVE_TIME, false, startTime, endTime);
					archiveList = parseArchiveList(deserializer);
					performRequest();
					queryHandler.sendSuccessResponse(messageTypeResult);
					anzReqDS = getArchMgr().getInQueuesMgr().getQueuesCountRequested() - anzReqDS;
					_debug.info(
							getName() + ": Manueller Anstoß der automatischen Nachforderung beendet (Zeit: " + Util.stopTimer(t) + ", Luecken: " + totalGapsFound
									.get()
									+ ", erhaltene Datensaetze: " + anzReqDS + ")"
					);
				}
				catch(Exception e) {
					_debug.warning("Nachforderung fehlgeschlagen.", e);
					queryHandler.sendErrorResponse(messageTypeResult, "Manueller Anstoß der automatischen Nachforderung fehlgeschlagen: " + e.getMessage());
				}
			}
			else {
				_debug.error("Ungültiger Nachrichtentyp für die Nachforderung: " + msgType);
			}
		}
		archDataSpecs = null;
	}


	/**
	 * Extrahiert die Liste der Archivsysteme, bei denen nachgefordert werden soll. Der Deserialisierer muss genau vor dem Längenbyte der Liste stehen.
	 * Doppelte Einträge werden gefiltert, Nachfordern bei sich selbst ebenso.
	 *
	 * @param ds Deserialisierer der Anfrage
	 * @return Liste mit Archivsystem-Objekten, leere Liste, falls keine angegeben sind
	 * @throws IOException Allgemeiner IO-Fehler
	 */
	private List<RemoteArchive> parseArchiveList(Deserializer ds) throws Exception {
		int anzArchives = ds.readInt(); // Größe der Liste mit den Archiven
		List<RemoteArchive> result = new ArrayList<>(anzArchives);

		HashSet<SystemObject> singleEntries = new HashSet<>();
		for(int i = 0; i < anzArchives; i++) {
			SystemObject foreignArS = ds.readObjectReference(getArchMgr().getDataModel());
			if(foreignArS == null) throw new IllegalArgumentException("Archivsystem-Objekt[" + i + "] == null");

			// doppelte Anfragen vermeiden, nicht bei sich selber nachfragen
			if (!singleEntries.contains(foreignArS) && !getArchMgr().getArchiveObject().getPid().equals(foreignArS.getPid())) {
				result.add(RemoteArchive.open(foreignArS));
				singleEntries.add(foreignArS);
			}
		}
		return result;
	}

	/**
	 * Führt die Nachforderung durch
	 */
	private void performRequest() throws PersistenceException {

		if(archDataSpecs.length == 0) {
			for (DataIdentNode dataIdentNode : getDidTree()) {
				if (dataIdentNode.isArSParameterized() && dataIdentNode.arSParamIsNachfordern()) {
					requestDID(dataIdentNode.getDataIdentification(), dataIdentNode, archTimeSpec);
				}
			}
		} else {
			for (ArchiveDataSpecification archDataSpec : archDataSpecs) {
				requestDID(new IdDataIdentification(archDataSpec), null, archDataSpec.getTimeSpec());
			}
		}
	}

	/**
	 * Diese Klasse sieht aus wie ein ResultData, wird aber nur intern verwendet, um einen über den Scheduler angestoßenen Sicherungslauf von einem extern
	 * angestoßenen zu unterscheiden
	 *
	 * @author beck et al. projects GmbH
	 * @author Phil Schrettenbrunner
	 */
	public static class RequestGapResultData extends ResultData {

		/** Erzeugt ein neues leeres RequestGapResultData-Objekt. */
		public RequestGapResultData() {
			super(null, null, 0, null);
		}

	}

	/**
	 * Fordert die angegebene Datenidentifikation innerhalb des angegebenen Bereichs nach.
	 */
	private void requestDID(IdDataIdentification dataIdentification, @Nullable DataIdentNode dinIfKnown, @Nullable ArchiveTimeSpecification ats) throws PersistenceException {
		try {
			suspendTaskIfNecessary();    //Hält den Task an, falls das Archivsystem im kritischen Bereich.

			DataIdentNode din;
			if (dinIfKnown != null) {
				din = dinIfKnown;
			} else {
				din = getDidTree().getIfPresent(dataIdentification);
				if(din == null) {
					_debug.fine("DIN ist nicht parametriert für " + dataIdentification + ".");
					return;    // Überhaupt nicht parametriert
				}
			}

			if(!din.arSParamIsNachfordern()) {
				_debug.fine("Nachfordern ist nicht parametriert für " + dataIdentification + ".");
				return;    // Nachfordern nicht parametriert
			}

			// Nur für die parametrierten Archive nachfordern
			List<RemoteArchive> requestArchives;

			if (!archiveList.isEmpty()) {
				// Wenn eine ArchivListe übergeben wurde, die anzufragenden Archive darauf beschränken
				requestArchives = archiveList;
			}
			else {
				requestArchives = copyReqArSFromDIN(din);
			}

			// Alle Archive ignorieren, die nicht erreichbar sind
			requestArchives.removeIf(it -> {
				ArchiveRequestManager requestManager = it.createRequestManager(getArchMgr().getDavCon());
				return requestManager == null || !requestManager.isArchiveAvailable();
			});

			if(requestArchives.isEmpty()) {
				_debug.fine("Es ist kein Archiv verfügbar, um " + dataIdentification + " nachzufordern.");
				return;
			}

			AttributeGroup atg = getArchMgr().getAtg(dataIdentification.getAtgId());
			Aspect asp = getArchMgr().getAsp(dataIdentification.getAspectId());
			final SystemObject object = getArchMgr().getObj(dataIdentification.getObjectId());
			final DataDescription dataDesc = new DataDescription(atg, asp);

			DataIndexRangeSet gaps = TaskManager.compute("Nachfordern: Ermittlung von Datenlücken",
					(it) -> getGapRanges(dataIdentification, ats));
			if (gaps == null) return;
			if (gaps.isEmpty()) {
				_debug.fine("Keine Datenlücken vorhanden für '" + dataIdentification + "'");
				return;
			}
			requestGaps(gaps, requestArchives, dataIdentification, dataDesc, object);
		} catch (Exception e) {
			_debug.warning("Fehler beim Nachfordern von '" + dataIdentification, e);
		}

		try {
			_debug.fine("Warte auf Fertigstellung der Nachforderung.");

			getArchMgr().getInQueuesMgr().insertRequestedDataFinishedNotification();
			awaitNotification();

			_debug.fine("Nachforderung fertig.");
		} catch (InterruptedException e) {
			throw new PersistenceException("Warten auf Nachfordern wurde unterbrochen");
		}
	}

	/**
	 * Wartet auf die Benachrichtigung, dass alle Daten archiviert wurden. Siehe {@link #signalNotification()}.
	 */
	private void awaitNotification() throws InterruptedException {
		archiveRequestedDataFinishedLock.lock();
		try {
			while (!requestedDataFinished) {
				archiveRequestedDataFinishedCondition.await();
			}
			// hier schon zurücksetzen
			requestedDataFinished = false;
		} finally {
			archiveRequestedDataFinishedLock.unlock();
		}
	}

	/**
	 * Sendet die Benachrichtigung, dass alle Daten archiviert wurden. Siehe {@link #awaitNotification()}.
	 */
	public void signalNotification() {
		archiveRequestedDataFinishedLock.lock();
		try {
			requestedDataFinished = true;
			archiveRequestedDataFinishedCondition.signalAll();
		} finally {
			archiveRequestedDataFinishedLock.unlock();
		}
	}

	/**
	 * Ermittelt alle Lücken
	 *
	 * @param dataIdentification Datenidentifikation
	 * @param ats                Zeitspezifikation, die untersucht werden soll, oder null für alle Daten.
	 * @return Lücken
	 * @throws PersistenceException           Lesefehler in Persistenz
	 * @throws SynchronizationFailedException Synchronisierung fehlgeschlagen
	 */
	@Nullable
	private DataIndexRangeSet getGapRanges(IdDataIdentification dataIdentification, @Nullable ArchiveTimeSpecification ats) throws PersistenceException, SynchronizationFailedException {
		DataIndexRangeSet gaps = new DataIndexRangeSet();

		QueryDataSequence dataSequence = new QueryDataSequence(getPersistenceManager(), ArchiveDataKindCombination
				.all(), ats, ArchiveOrder.BY_INDEX, dataIdentification);

		try (DataIterator iterator = dataSequence.iteratorWithoutTimeFilter()) {

			if (iterator.isEmpty()) {
				_debug.fine("Es sind keine bestehenden Daten archiviert, " + dataIdentification + " kann daher nicht nachgefordert werden.");
				return null;
			}

			TimingType timingType = ats == null ? TimingType.DATA_INDEX : ats.getTimingType();

			long lowDidx = -1;
			long lowFilterValue = -1;

			while (!iterator.isEmpty()) {

				if (shouldTerminate()) return null;

				long hiDidx = iterator.peekDataIndex();
				long highFilterValue = peekFilterValue(timingType, iterator);

				if (isGap(lowDidx, hiDidx)) {
					// Es gibt eine Lücke

					if (isInRange(ats, lowFilterValue, highFilterValue)) {
						// Die Lücke liegt im Anfragebereich

						totalGapsFound.incrementAndGet();

						gaps.addRange(noModBits(lowDidx) + 1, noModBits(hiDidx));
					}

				}
				lowDidx = hiDidx;
				lowFilterValue = highFilterValue;
				iterator.remove();
			}
		}
		return gaps;
	}

	private static boolean isInRange(@Nullable final ArchiveTimeSpecification ats, final long a, final long b) {
		if(ats == null) return true;
		long start = ats.getIntervalStart();
		long end = ats.getIntervalEnd();
		if(a <= b) {
			return a <= end && b >= start;
		}
		return b <= end && a >= start;
	}

	private static long peekFilterValue(final TimingType timingType, final DataIterator iterator) {
		if(timingType == TimingType.DATA_TIME) {
			return iterator.peekDataTime();
		}
		if(timingType == TimingType.ARCHIVE_TIME) {
			return iterator.peekArchiveTime();
		}
		if(timingType == TimingType.DATA_INDEX) {
			return iterator.peekDataIndex();
		}
		throw new IllegalArgumentException("TimingType: " + timingType);
	}

	private void requestGaps(final DataIndexRangeSet gaps, final List<RemoteArchive> archivesToRequestFrom, final IdDataIdentification dataIdentification, final DataDescription dataDesc, final SystemObject object) {
		_debug.fine("Starte Nachforderung von " + gaps.size() + " Lücken bei " + archivesToRequestFrom);
		DataIndexRangeSet remainingGaps = new DataIndexRangeSet(gaps);
		DataIndexRangeSet allClosedGaps = new DataIndexRangeSet();

		GapFile gapFile = GapFile.getInstance(getPersistenceManager(), dataIdentification);
		
		final Map<RemoteArchive, DataIndexRangeSet> previouslyFailedGapsPerArs = new HashMap<>();
		
		gapFile.readGaps(previouslyFailedGapsPerArs);
		
		final Map<RemoteArchive, DataIndexRangeSet> failedGapsPerArs = new HashMap<>(previouslyFailedGapsPerArs);
		
		for(RemoteArchive archiveObject : archivesToRequestFrom) {
			DataIndexRangeSet previouslyFailedGaps = previouslyFailedGapsPerArs.get(archiveObject);
			if(previouslyFailedGaps == null) {
				previouslyFailedGaps = new DataIndexRangeSet();
			}
			DataIndexRangeSet gapsToRequest = new DataIndexRangeSet(remainingGaps);
			
			// Keine Daten anfragen, wo wir schon wissen, dass das Archivsystem diese nicht besitzt.
			// Ausnahme: Wenn Archivsysteme explizit angegeben sind, wird dort auch dann nachgefragt, wenn dies
			// bereits als nicht erfolgreich vermerkt wurde (TAnfArS 5.1.2.12.2)
			if(archiveList.isEmpty()) {
				gapsToRequest.removeAll(previouslyFailedGaps);
			}

			// Daten nachfordern
			DataIndexRangeSet closedGaps = requestGapsFromArchive(gapsToRequest, archiveObject, dataDesc, object);

			if (closedGaps == null) {
				// Abfrage bei diesem Archivsystem hat nicht funktioniert, ignorieren
				// requestGapsFromArchive hat bereits Debug-Warnung erzeugt.
				continue;
			}
			
			// Lücken aktualisieren
			allClosedGaps.addAll(closedGaps);
			DataIndexRangeSet failedGaps = new DataIndexRangeSet(gapsToRequest);
			failedGaps.addAll(previouslyFailedGaps);
			failedGaps.removeAll(closedGaps);
			failedGapsPerArs.put(archiveObject, failedGaps);
			remainingGaps.removeAll(closedGaps);
			if(remainingGaps.isEmpty()) {
				_debug.fine("Alle Lücken sind beim Nachfordern geschlossen worden, weitere Archivsysteme werden ignoriert.");
				break;
			}
			if (shouldTerminate()) {
				_debug.fine("Nachforderung abgebrochen.");
				break;
			}
		}

		/*
		 Datenlücken die geschlossen wurden, brauchen nicht mehr im Gapfile gespeichert zu werden, da wir uns nicht mehr dafür interessieren
		 (Sie würden eh nicht neu angefragt, weil die Daten ja jetzt da sind).
		 Sollte beim Archivieren was schiefgegangen sein, dann werden die verbliebenen Datenlücken beim nächsten Durchlauf geschlossen, da sie
		 ja nicht mehr im Gapfile enthalten sind.
		*/
		for (DataIndexRangeSet value : failedGapsPerArs.values()) {
			value.removeAll(allClosedGaps);
		}
		
		gapFile.writeGaps(failedGapsPerArs);
		
	}

	/**
	 * Fordert Daten von einem Archivsystem nach
	 *
	 * @param gaps          Liste mit Datenlücken
	 * @param archiveSystem Archivsystem, von dem nachgefordert werden soll
	 * @param dataDesc      {@link DataDescription}, die nachgefordert werden soll
	 * @param object        Objekt, für das nachgefordert werden soll
	 * @return Alle Datenindexbereiche der Datenlücken, die mit diesem Archivsystem geschlossen werden konnten
	 * oder {@code} null, wenn keine Anfrage gestellt wurde, weil das Archivsystem nicht erreichbar war oder Fehler
	 * aufgetreten sind. 
	 */
	@Nullable
	private DataIndexRangeSet requestGapsFromArchive(final DataIndexRangeSet gaps, final RemoteArchive archiveSystem, final DataDescription dataDesc, final SystemObject object) {
		            
		DataIndexRangeSet closedGaps = new DataIndexRangeSet();
		if(gaps.isEmpty()) {
			return closedGaps; // Nichts zu tun, keine Lücken da
		}

		ArchiveRequestManager arm = archiveSystem.createRequestManager(getArchMgr().getDavCon());
		
		if(arm == null) {
			_debug.info("Nachfordern: Archivsystem '" + archiveSystem + "' ist nicht erreichbar");
			// Archivsystem nicht erreichbar
			return null;
		}

		for (Range<Long> gap : gaps) {

			suspendTaskIfNecessary();

			long lowBnd = Util.dIdxAppendZeroModBits(gap.lowerEndpoint());
			long upBnd = Util.dIdxAppendZeroModBits(gap.upperEndpoint());

			try {
				DataIndexRangeSet requestedGaps = requestSingleGapFromArchive(archiveSystem, dataDesc, object, arm, lowBnd, upBnd);
				if (requestedGaps == null) {
					// Warnung wurde schon von requestSingleGapFromArchive ausgegeben
					return null;
				}
				closedGaps.addAll(requestedGaps);
			} catch (IllegalStateException | TimeoutException | InterruptedException e) {
				_debug.warning("Nachfordern: Problem bei Anfrage an '" + archiveSystem + "'", e);
				return null;
			}

			if (shouldTerminate()) {
				break;
			}
		}
		return closedGaps;
	}

	@Nullable
	private DataIndexRangeSet requestSingleGapFromArchive(RemoteArchive archiveSystem, DataDescription dataDesc, SystemObject object, ArchiveRequestManager arm, long lowBnd, long upBnd) throws TimeoutException, InterruptedException {
		return TimeoutManager.callWithTimeout(timeout -> {
			DataIndexRangeSet innerClosedGaps = new DataIndexRangeSet();
			ArchiveDataQueryResult res = arm.request(REQUEST_PRIO, buildRequest(lowBnd, upBnd, dataDesc, object));
			if (!res.isRequestSuccessful()) {
				_debug.warning("Nachforderungsanfrage an Archivsystem '" + archiveSystem + "' fehlgeschlagen", res.getErrorMessage());
				return null;
			}
			timeout.tick();
			ArchiveDataStream ads = res.getStreams()[0];    // nur einer war angelegt
			try {
				ArchiveData ad;
				while ((ad = ads.take()) != null) {
					timeout.tick();
					if (idx(ad) >= upBnd) {
						// hinter dem letzten angefragten Datensatz 
						// → Abbrechen
						break;
					}

					if (idx(ad) < lowBnd) {
						// vor dem ersten gewünschten Datensatz (z. B. Anfangszustand oder Potgap)
						// → Ignorieren
						continue;
					}

					// Datenindex passt. Nur speichern, wenn korrekte Datenart. Lückenmarkierungen,
					// die einen fehlerhaften Index haben, werden dadurch nicht gespeichert.
					if (properDataState(ad)) {
						getArchMgr().getInQueuesMgr().archiveRequestedData(System.currentTimeMillis(), ad);
						innerClosedGaps.addRange(noModBits(ad.getDataIndex()), noModBits(ad.getDataIndex()) + 1);
					}
				}
			} catch (IOException e) {
				_debug.warning("Fehler beim Empfang eines Nachforderungs-Datensatzes von '" + archiveSystem + "'", e);
			} finally {
				ads.abort();
			}
			return innerClosedGaps;
		});
	}

	/**
	 * Kopiert die Liste der Nachforderungs-Archive aus dem DataIdentNode. Jedes Archivsystem wird nur einmal in die Ergebnisliste aufgenommen.
	 *
	 * @param din Der betreffende Datenknoten
	 * @return Liste von Archiv-Objekten
	 */
	private List<RemoteArchive> copyReqArSFromDIN(DataIdentNode din) {
		ArrayList<RemoteArchive> otherArchives = new ArrayList<>();
		HashSet<String> singleEntries = new HashSet<>();
		for(int i = 0; i < din.arSParamGetAnzNachfordern(); i++) {
			String pid = din.arSParamGetNachfordern(i);

			if (pid == null || pid.isEmpty() || pid.equals("0")) continue;    // DAV erlaubt hier Nullwerte

			// doppelte Anfragen vermeiden, nicht bei sich selber nachfragen
			if (!singleEntries.contains(pid) && !getArchMgr().getConfigAuth().getPid().equals(pid)) {
				otherArchives.add(RemoteArchive.open(pid));
				singleEntries.add(pid);
			}
		}
		return otherArchives;
	}

	// ***************************************************************************************************
	// **********  Abkuerzende Methoden (um den Hauptalgorithmus einfach zu halten) **********************
	// ***************************************************************************************************

	private ArchiveDataSpecification buildRequest(long startIdx, long endIdx, final DataDescription dataDesc, final SystemObject object) {
		return new ArchiveDataSpecification(
				new ArchiveTimeSpecification(TimingType.DATA_INDEX, false, startIdx, endIdx),
				requestAdk,
				ArchiveOrder.BY_INDEX,
				ArchiveRequestOption.NORMAL,
				dataDesc,
				object
		);
	}

	private static boolean isGap(long start, long end) {
		return start != -1 && noModBits(end) - noModBits(start) > 1;
	}

	private static boolean properDataState(ArchiveData ad) {
		DataState ds = ad.getDataType();
		return ds.equals(DataState.DATA) || ds.equals(DataState.NO_DATA) || ds.equals(DataState.NO_SOURCE);
	}

	private static long idx(ArchiveData ad) {
		return ad.getDataIndex();
	}

	private static long noModBits(long dataIndex) {
		return Util.dIdxNoModBits(dataIndex);
	}
}
