/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.mgmt.tasks;

import de.bsvrz.ars.ars.persistence.ContainerFile;
import de.bsvrz.ars.ars.persistence.writer.AsyncSerializableDataset;

import java.util.Arrays;
import java.util.zip.Deflater;

public class SerializeTaskHelper {
	final DavDataSerializer serializer = new DavDataSerializer();
	final Deflater deflater = new Deflater();
	/**
	 * Puffer zum Schreiben von komprimierten Daten
	 */
	final byte[] writeBuf;

	public SerializeTaskHelper() {
		this.writeBuf = new byte[ContainerFile.BUFFER_SIZE];
	}

	/**
	 * Liefert den übergebenen Puffer, falls dessen Größe ausreichend ist, andernfalls einen neu angelegten.
	 * Zweck: sowenig Arbeit für die Speicherverwaltung wie möglich.
	 *
	 * @param defaultBuffer Vorhandener Puffer
	 * @param desiredSize   Gewünschte PufferGröße
	 * @return Byte-Array-Puffer
	 */
	static byte[] getBuf(byte[] defaultBuffer, int desiredSize) {
		return (desiredSize <= defaultBuffer.length) ? defaultBuffer : new byte[desiredSize];
	}

	public void complete(AsyncSerializableDataset dataset) {
		byte[] originalData = serializer.serializeData(dataset.originalData());

		if (originalData.length > ContainerFile.MAX_UNCOMPRESSED) {
			byte[] dest = getBuf(writeBuf, originalData.length);

			deflater.reset();
			deflater.setInput(originalData);
			deflater.finish();
			int compressedDataLength = deflater.deflate(dest);

			if (compressedDataLength < originalData.length && deflater.finished()) {
				// Komprimieren hat was gebracht 
				dataset.complete(Arrays.copyOf(dest, compressedDataLength), originalData.length);
				return;
			}
		}
		// Es wurde nicht komprimiert
		dataset.complete(originalData, ContainerFile.NOT_COMPRESSED);
	}
}
