/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.mgmt.tasks.base;

import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

/**
 * Task-Verwaltung im Datenverteiler. Ein Task ist eine Operation, die im Hintergrund im Archivsystem ausgeführt wird,
 * und insbesondere alte (bereits archivierte) Daten betrifft. Die laufende Archivierung von neuen Daten ist kein
 * Task.
 * <p>
 * Diese Klasse erfüllt primär die Aufgabe, zu überwachen, welche Tasks gerade laufen, und exklusive Tasks
 * von der Ausführung von anderen Tasks zu trennen. Exklusive Tasks werden nur ausgeführt, wenn gerade keine anderen
 * Tasks laufen, was bei bestimmten Operationen erforderlich ist. Tasks, die z. B. alte Archivverzeichnisse löschen
 * oder anderweitig strukturell verändern dürfen nicht nebenläufig mit anderen Tasks ausgeführt werden.
 * <p>
 * Die Methoden in dieser Klassen führen Tasks aus und besitzen optional ein {@link Task}-Argument, bit dem bereits
 * vorhandene Task-Implementierungen übergeben werden können
 * (siehe z. B. {@link de.bsvrz.ars.ars.mgmt.ArchiveLongTermTaskStatePublisher#registerTask(String)}).
 * Dadurch wird der Task-Fortschritt sowohl diesem {@link TaskManager} als auch dem
 * {@link de.bsvrz.ars.ars.mgmt.LongTermTaskStatePublisher} bekannt gemacht.
 * <p>
 * Für temporäre, kurzzeitige Tasks kann die {@link #compute(String, TaskProducer)}-Methode jedoch auch ohne
 * Task-Argument ausgeführt werden.
 * <p>
 * Es ist wichtig, dass alle Operationen, die auf alte Archivdaten irgendwie zugreifen, als Task ausgeführt werden.
 * Nur so ist sichergestellt, dass kein nebenläufiger exklusiver Task zwischenzeitlich Daten verändert
 * oder löscht. Dies wird jedoch nicht im Code überprüft, da viele Operationen über mehrere Threads laufen, und
 * so eine Zuordnung von Thread zu Task nicht trivial ist,
 * und die Weitergabe von Task-Objekten, Zugrifftokens o.ä. den Code sehr unübersichtlich machen würde.
 */
public class TaskManager {

	/**
	 * Gerade laufende (exklusive und nicht-exklusive) Tasks, mit {@link #taskLock} geschützt.
	 */
	private static final Set<Task> activeTasks = new HashSet<>();

	/**
	 * Gerade laufender exklusiver Task, mit {@link #taskLock} geschützt.
	 */
	private static Task exclusiveTask = null;

	/**
	 * Lock-Objekt, das diese Klasse threadsicher macht
	 */
	private static final Lock taskLock = new ReentrantLock();

	/**
	 * Bedingung, die ausgelöst wird, wenn activeTasks leer ist, also erneut geprüft werden kann, ob Tasks
	 * ausgeführt werden dürfen.
	 * <p>
	 * Normale Tasks werden dann ausgeführt, wenn kein exklusiver Task läuft,
	 * ansonsten muss über diese Bedingung gewartet werden, bis der exklusive Tasks aus dem activeTasks-Set raus ist.
	 * <p>
	 * Exklusive Tasks können nur dann gestartet werden, wenn das activeTasks-Set leer ist.
	 * <p>
	 * Es ist also in jedem
	 * Fall ausreichend darauf zu warten, dass activeTasks leer ist.
	 */
	private static final Condition tasksEmpty = taskLock.newCondition();

	/**
	 * Sorgt dafür, dass ein Task ausgeführt wird. Nebenläufige andere Tasks sind möglich, sofern diese nicht
	 * als {@linkplain #computeExclusively(Task, TaskProducer) exklusiver Task} gestartet worden sind.
	 *
	 * @param task     Optional: Task-Objekt (falls ein bestehender Task genutzt werden soll)
	 * @param executor Funktionales Interface, das die eigentliche implementierung des Tasks enthält.
	 * @param <T>      generischer Exception-Typ, der von der Funktion geworfen werden kann (wird an Aufrufer weitergegeben)
	 * @param <R>      Typ des Ergebnisses
	 * @return Ergebnis der executor-Funktion
	 * @throws T                    Fehler, die von der Funktion generiert werden, werde an den Aufrufer weitergegeben
	 * @throws InterruptedException Wenn die Funktion beim Warten auf exklusive Task-Ausführung unterbrochen wurde
	 */
	public static <R, T extends Exception> R compute(Task task, TaskProducer<R, T> executor) throws T, InterruptedException {
		task.setTaskState(TaskState.START_WORK);
		taskLock.lock();
		if (activeTasks.contains(task)) {
			// activeTasks ist ein Set, denselben Task mehrmals einzufügen würde der Logik der Klasse widersprechen
			// und Probleme verursachen, da der Task zu füh aus dem Set entfernt werden würde.
			throw new IllegalStateException("Task bereits registriert.");
		}
		// Warten, bis kein exklusiver Task mehr läuft
		try {
			while (exclusiveTask != null) {
				tasksEmpty.await();
			}
			// Task registrieren
			activeTasks.add(task);
		} catch (InterruptedException e) {
			Thread.currentThread().interrupt();
			throw e;
		} finally {
			taskLock.unlock();
		}
		try {
			// Diesen Task ausführen
			task.setTaskState(TaskState.WORKING);
			return executor.run(task);
		} finally {
			task.setTaskState(TaskState.FINISH_WORK);

			// und aus der Liste der laufenden Tasks entfernen
			taskLock.lock();
			try {
				activeTasks.remove(task);
				if (activeTasks.isEmpty()) {
					// Signal auslösen, das andere wartende Threads aufweckt
					tasksEmpty.signal();
				}
			} finally {
				taskLock.unlock();
			}
		}
	}

	/**
	 * Sorgt dafür, dass ein Task exklusiv, also ohne nebenläufige andere Tasks zu beeinflussen, ausgeführt wird.
	 *
	 * @param task     Optional: Task-Objekt (falls ein bestehender Task genutzt werden soll)
	 * @param executor Funktionales Interface, das die eigentliche implementierung des Tasks enthält.
	 * @param <T>      generischer Exception-Typ, der von der Funktion geworfen werden kann (wird an Aufrufer weitergegeben)
	 * @param <R>      Typ des Ergebnisses
	 * @return Ergebnis der executor-Funktion
	 * @throws T                    Fehler, die von der Funktion generiert werden, werde an den Aufrufer weitergegeben
	 * @throws InterruptedException Wenn die Funktion beim Warten auf exklusive Task-Ausführung unterbrochen wurde
	 */
	public static <R, T extends Exception> R computeExclusively(Task task, TaskProducer<R, T> executor) throws T, InterruptedException {
		task.setTaskState(TaskState.START_WORK);
		taskLock.lock();
		if (activeTasks.contains(task)) {
			// activeTasks ist ein Set, denselben Task mehrmals einzufügen würde der Logik der Klasse widersprechen
			// und Probleme verursachen, da der Task zu füh aus dem Set entfernt werden würde.
			throw new IllegalStateException("Task bereits registriert.");
		}

		// Warten, bis kein Task mehr läuft
		try {
			while (!activeTasks.isEmpty()) {
				tasksEmpty.await();
			}

			// Task registrieren
			activeTasks.add(task);
			exclusiveTask = task;
		} catch (InterruptedException e) {
			Thread.currentThread().interrupt();
			throw e;
		} finally {
			taskLock.unlock();
		}

		try {
			// Diesen Task ausführen
			task.setTaskState(TaskState.WORKING);
			return executor.run(task);
		} finally {
			task.setTaskState(TaskState.FINISH_WORK);

			// und aus der Liste der laufenden Tasks entfernen
			taskLock.lock();
			try {
				activeTasks.remove(task);
				exclusiveTask = null;
				assert activeTasks.isEmpty();
				// Signal auslösen, das andere wartende Threads aufweckt
				tasksEmpty.signal();
			} finally {
				taskLock.unlock();
			}
		}
	}

	/**
	 * Sorgt dafür, dass ein Task ausgeführt wird. Nebenläufige andere Tasks sind möglich, sofern diese nicht
	 * als {@linkplain #computeExclusively(String, TaskProducer) exklusiver Task} gestartet worden sind.
	 *
	 * @param name     Beschreibender deutscher Name des Tasks (für Debug-Ausgaben, Task-Übersicht etc.)
	 * @param executor Funktionales Interface, das die eigentliche implementierung des Tasks enthält.
	 * @param <T>      generischer Exception-Typ, der von der Funktion geworfen werden kann (wird an Aufrufer weitergegeben)
	 * @param <R>      Typ des Ergebnisses
	 * @return Ergebnis der executor-Funktion
	 * @throws T                    Fehler, die von der Funktion generiert werden, werde an den Aufrufer weitergegeben
	 * @throws InterruptedException Wenn die Funktion beim Warten auf exklusive Task-Ausführung unterbrochen wurde
	 */
	public static <R, T extends Exception> R compute(String name, TaskProducer<R, T> executor) throws T, InterruptedException {
		return compute(new ShortTermTask(name), executor);
	}

	/**
	 * Sorgt dafür, dass ein Task exklusiv, also ohne nebenläufige andere Tasks zu beeinflussen, ausgeführt wird.
	 *
	 * @param name     Beschreibender deutscher Name des Tasks (für Debug-Ausgaben, Task-Übersicht etc.)
	 * @param executor Funktionales Interface, das die eigentliche implementierung des Tasks enthält.
	 * @param <T>      generischer Exception-Typ, der von der Funktion geworfen werden kann (wird an Aufrufer weitergegeben)
	 * @param <R>      Typ des Ergebnisses
	 * @return Ergebnis der executor-Funktion
	 * @throws T                    Fehler, die von der Funktion generiert werden, werde an den Aufrufer weitergegeben
	 * @throws InterruptedException Wenn die Funktion beim Warten auf exklusive Task-Ausführung unterbrochen wurde
	 */
	public static <R, T extends Exception> R computeExclusively(String name, TaskProducer<R, T> executor) throws T, InterruptedException {
		return computeExclusively(new ShortTermTask(name), executor);
	}

	/**
	 * Sorgt dafür, dass ein Task ausgeführt wird. Nebenläufige andere Tasks sind möglich, sofern diese nicht
	 * als {@linkplain #computeExclusively(Task, TaskProducer) exklusiver Task} gestartet worden sind.
	 *
	 * @param task     Optional: Task-Objekt (falls ein bestehender Task genutzt werden soll)
	 * @param executor Funktionales Interface, das die eigentliche implementierung des Tasks enthält.
	 * @param <T>      generischer Exception-Typ, der von der Funktion geworfen werden kann (wird an Aufrufer weitergegeben)
	 * @throws T Fehler, die von der Funktion generiert werden, werde an den Aufrufer weitergegeben
	 */
	public static <T extends Exception> void run(Task task, TaskExecutor<T> executor) throws T {
		try {
			compute(task, (TaskProducer<Void, T>) tpi -> {
				executor.run(tpi);
				return null;
			});
		} catch (InterruptedException e) {
			Thread.currentThread().interrupt();
		}
	}

	/**
	 * Sorgt dafür, dass ein Task exklusiv, also ohne nebenläufige andere Tasks zu beeinflussen, ausgeführt wird.
	 *
	 * @param task     Optional: Task-Objekt (falls ein bestehender Task genutzt werden soll)
	 * @param executor Funktionales Interface, das die eigentliche implementierung des Tasks enthält.
	 * @param <T>      generischer Exception-Typ, der von der Funktion geworfen werden kann (wird an Aufrufer weitergegeben)
	 * @throws T Fehler, die von der Funktion generiert werden, werde an den Aufrufer weitergegeben
	 */
	public static <T extends Exception> void runExclusively(Task task, TaskExecutor<T> executor) throws T {
		try {
			computeExclusively(task, (TaskProducer<Void, T>) tpi -> {
				executor.run(tpi);
				return null;
			});
		} catch (InterruptedException e) {
			Thread.currentThread().interrupt();
		}
	}

	/**
	 * Sorgt dafür, dass ein Task ausgeführt wird. Nebenläufige andere Tasks sind möglich, sofern diese nicht
	 * als {@linkplain #computeExclusively(String, TaskProducer) exklusiver Task} gestartet worden sind.
	 *
	 * @param name     Beschreibender deutscher Name des Tasks (für Debug-Ausgaben, Task-Übersicht etc.)
	 * @param executor Funktionales Interface, das die eigentliche implementierung des Tasks enthält.
	 * @param <T>      generischer Exception-Typ, der von der Funktion geworfen werden kann (wird an Aufrufer weitergegeben)
	 * @throws T Fehler, die von der Funktion generiert werden, werde an den Aufrufer weitergegeben
	 */
	public static <T extends Exception> void run(String name, TaskExecutor<T> executor) throws T {
		try {
			compute(name, (TaskProducer<Void, T>) tpi -> {
				executor.run(tpi);
				return null;
			});
		} catch (InterruptedException e) {
			Thread.currentThread().interrupt();
		}
	}

	/**
	 * Sorgt dafür, dass ein Task exklusiv, also ohne nebenläufige andere Tasks zu beeinflussen, ausgeführt wird.
	 *
	 * @param name     Beschreibender deutscher Name des Tasks (für Debug-Ausgaben, Task-Übersicht etc.)
	 * @param executor Funktionales Interface, das die eigentliche implementierung des Tasks enthält.
	 * @param <T>      generischer Exception-Typ, der von der Funktion geworfen werden kann (wird an Aufrufer weitergegeben)
	 * @throws T Fehler, die von der Funktion generiert werden, werde an den Aufrufer weitergegeben
	 */
	public static <T extends Exception> void runExclusively(String name, TaskExecutor<T> executor) throws T {
		try {
			computeExclusively(name, (TaskProducer<Void, T>) tpi -> {
				executor.run(tpi);
				return null;
			});
		} catch (InterruptedException e) {
			Thread.currentThread().interrupt();
		}
	}

	/**
	 * Gibt alle aktuell laufenden tasks zurück (z. B. für Debug-Ausgaben)
	 *
	 * @return alle aktuell laufenden Tasks
	 */
	public static Set<Task> getActiveTasks() {
		taskLock.lock();
		try {
			return Set.copyOf(activeTasks);
		} finally {
			taskLock.unlock();
		}
	}
}
