/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.mgmt.tasks.base;

import de.bsvrz.sys.funclib.debug.Debug;

import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;

/**
 * Hilfsklasse, die dabei hilft, bestimmte Aktionen mit Timeout auszuführen. Es geht hier insbesondere darum,
 * Blockierung zu vermeiden, wenn die Nachforderung von Archivdaten beim Warten auf Antworten des anderen
 * Archivsystem hängenbleibt.
 */
public class TimeoutManager {

	private static final Debug _debug = Debug.getLogger();
	private static final ScheduledExecutorService checkExecutor = Executors.newSingleThreadScheduledExecutor();

	/**
	 * Standard-Timeout im Archivsystem
	 */
	public static final long STANDARD_TIMEOUT = 60 * 1000;

	/**
	 * Führt den angegebenen Lambda-Ausdruck mit einem Timeout aus. Der Timeout ist {@link #STANDARD_TIMEOUT}.
	 * <p>
	 * Wenn der Timeout abläuft und die Funktion sich nicht nicht beendet hat, wird der Thread,
	 * der die Funktion ausführt (dies ist der gleiche Thread, auf dem diese Funktion aufgerufen wird),
	 * mit {@link Thread#interrupt()} unterbrochen. Das heißt, mit dem {@link TimeoutManager} können nur
	 * Funktionen mit einem Timeout versehen werden, die sauber auf Interrupts reagieren und nicht z. B.
	 * in Endlosschleifen festhängen.
	 * </p>
	 *
	 * @param func Funktion, die ausgeführt wird. Der Funktion wird als Parameter ein {@link TimeoutCallback} übergeben,
	 *             über den mit {@link TimeoutCallback#tick()} der Timeout verlängert werden kann.
	 * @param <T>  Ergebnis-Typ
	 * @return Ergebnis der übergebenen Funktion
	 * @throws TimeoutException     Wenn der Timeout abgelaufen ist.
	 * @throws InterruptedException Wenn dieser Thread unterbrochen wurde, bevor der Timeout abgelaufen ist
	 */
	public static <T> T callWithTimeout(TimeoutFunction<T> func) throws TimeoutException, InterruptedException {
		return callWithTimeout(func, STANDARD_TIMEOUT);
	}

	/**
	 * Führt den angegebenen Lambda-Ausdruck mit einem Timeout aus.
	 * <p>
	 * Wenn der Timeout abläuft und die Funktion sich nicht nicht beendet hat, wird der Thread,
	 * der die Funktion ausführt (dies ist der gleiche Thread, auf dem diese Funktion aufgerufen wird),
	 * mit {@link Thread#interrupt()} unterbrochen. Das heißt, mit dem {@link TimeoutManager} können nur
	 * Funktionen mit einem Timeout versehen werden, die sauber auf Interrupts reagieren und nicht z. B.
	 * in Endlosschleifen festhängen.
	 * </p>
	 *
	 * @param func          Funktion, die ausgeführt wird. Der Funktion wird als Parameter ein {@link TimeoutCallback} übergeben,
	 *                      über den mit {@link TimeoutCallback#tick()} der Timeout verlängert werden kann.
	 * @param timeoutMillis Timeout in Millisekunden
	 * @param <T>           Ergebnis-Typ
	 * @return Ergebnis der übergebenen Funktion
	 * @throws TimeoutException     Wenn der Timeout abgelaufen ist.
	 * @throws InterruptedException Wenn dieser Thread unterbrochen wurde, bevor der Timeout abgelaufen ist
	 */
	public static <T> T callWithTimeout(TimeoutFunction<T> func, long timeoutMillis) throws TimeoutException, InterruptedException {

		AtomicLong timeLimit = new AtomicLong(System.currentTimeMillis() + timeoutMillis);
		AtomicBoolean finished = new AtomicBoolean(false);

		Thread executingThread = Thread.currentThread();

		checkExecutor.schedule(() -> {
			checkDelay(finished, timeLimit, executingThread);
		}, timeoutMillis, TimeUnit.MILLISECONDS);

		TimeoutCallback callback = new TimeoutCallback() {
			@Override
			public void tick() {
				timeLimit.set(System.currentTimeMillis() + timeoutMillis);
			}

			@Override
			public void cancel() {
				finished.set(true);
			}
		};

		try {
			T result = func.call(callback);
			finished.set(true);
			if (Thread.interrupted()) {
				if (System.currentTimeMillis() >= timeLimit.get()) {
					// Wenn der Timeout gerade jetzt abgelaufen ist, die Methode aber fertig ist,
					// interrupted-Status zurücksetzen, da er nicht mehr gebraucht wird
					_debug.fine("Cleared interrupted status");
				} else {
					throw new InterruptedException();
				}
			}
			return result;
		} catch (InterruptedException e) {
			if (System.currentTimeMillis() >= timeLimit.get()) {
				throw new TimeoutException();
			}
			throw e;
		}
	}

	/**
	 * Interne Funktion die vom Executor aufgerufen wird, sobald der ursprüngliche Timeout (ungefähr) abgelaufen ist.
	 *
	 * @param finished        Funktion hat sich beendet?
	 * @param timeLimit       Aktuelles Zeitlimit (Epoch Millis)
	 * @param executingThread Bei Ablauf des Timeouts zu unterbrechender Thread
	 */
	private static void checkDelay(AtomicBoolean finished, AtomicLong timeLimit, Thread executingThread) {
		long limit = timeLimit.get();
		long remainingDelay = limit - System.currentTimeMillis();
		if (finished.get()) {
			// Timeout ist abgelaufen, aber das ist egal, da die Aktion erfolgreich beendet wurde
			return;
		}

		if (remainingDelay <= 0) {
			executingThread.interrupt();
		} else {
			checkExecutor.schedule(() -> {
				checkDelay(finished, timeLimit, executingThread);
			}, remainingDelay, TimeUnit.MILLISECONDS);
		}
	}
}
