/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.mgmt.tasks.query;

import de.bsvrz.ars.ars.mgmt.ArchiveManager;
import de.bsvrz.ars.ars.mgmt.tasks.base.TimeoutManager;
import de.bsvrz.dav.daf.main.*;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.dataSerializer.Deserializer;
import de.bsvrz.sys.funclib.dataSerializer.NoSuchVersionException;
import de.bsvrz.sys.funclib.dataSerializer.Serializer;
import de.bsvrz.sys.funclib.dataSerializer.SerializingFactory;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.bsvrz.sys.funclib.losb.datk.PidQuery;
import de.bsvrz.sys.funclib.losb.kernsoftware.ConnectionManager;
import de.bsvrz.sys.funclib.losb.util.ByteIO;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;

/**
 * Klasse, die einzelne Archivanfragen bearbeitet und beantwortet.
 */
public final class QueryHandler implements ClientSenderInterface {

	private static final Debug _debug = Debug.getLogger();

	private int serializerVersion = 0;

	/**
	 * Zustand der Sendesteuerung (Senden / Anhalten / Abbrechen).
	 */
	protected enum SendingState {

		SEND,
		STOP,
		ABORT
	}

	/**
	 * Aktueller Zustand der Sendesteuerung.
	 */
	private SendingState sendingState = SendingState.STOP;

	/**
	 * Zeit in Millisekunden, die auf eine positive Sendesteuerung gewartet wird
	 */
	public static final int SENDCONTROL_TIMEOUT = 30000;
	/**
	 * Spezielles Objekt zur Synchronisiserung. Immer dieses anstatt this verwenden!
	 */
	private final Object syncObj = new Object();
	/**
	 * Attributgruppe der Archiv-Anfrage und -Antwort.
	 */
	private AttributeGroup atgQuery;
	/**
	 * DataDescription der Archiv-Antwort.
	 */
	private DataDescription ddResponse;
	/**
	 * Applikations-Objekt des Absenders der Archiv-Anfrage.
	 */
	private SystemObject queryAppObj;
	/**
	 * Anfrage-Index der anfragenden Applikation zur Archiv-Anfrage.
	 */
	private int queryIdx;
	private final ByteArrayOutputStream resultStream = new ByteArrayOutputStream();
	private Serializer serializer;
	private Deserializer deserializer;
	private final String name;
	private final ArchiveManager archMgr;
	private final int messageTypeResult;

	public QueryHandler(String name, ArchiveManager archMgr, int messageTypeResult, @Nullable Data data) {
		this.name = name;
		this.archMgr = archMgr;
		this.messageTypeResult = messageTypeResult;
		DataModel dataModel = this.archMgr.getDataModel();

		if (dataModel == null) {
			// Datenverteiler getrennt, Anfragen-Bearbeitung sinnlos...
			return;
		}

		atgQuery = dataModel.getAttributeGroup(PidQuery.ATG_QUERY_PID);
		ddResponse = new DataDescription(atgQuery, dataModel.getAspect(PidQuery.ASP_RESPONSE_PID));
		if (data == null) return;

		queryAppObj = data.getReferenceValue(PidQuery.ATT_SENDER_NAME).getSystemObject();
		queryIdx = data.getUnscaledValue(PidQuery.ATT_QUERY_IDX_NAME).intValue();

		byte[] dataArray = data.getUnscaledArray(PidQuery.ATT_DATA_NAME).getByteArray();
		/*
		  Serialisierer-Version der anfragenden Applikation, mit der auch die Antwort erstellt werden muss.
		 */
		// Erste 4 Bytes = Serialisierer-Version
		serializerVersion = ByteIO.readSignedInt4Bytes(dataArray, 0);
		InputStream bis = new ByteArrayInputStream(dataArray, 4, dataArray.length - 4);

		// Beide Teile bereits hier erzeugen, damit eine falsche Version zu Beginn bemerkt wird
		try {
			deserializer = SerializingFactory.createDeserializer(serializerVersion, bis);
		} catch (NoSuchVersionException ignored) {
		}
		try {
			serializer = SerializingFactory.createSerializer(serializerVersion, resultStream);
		} catch (NoSuchVersionException ignored) {
		}
	}

	public SystemObject getQueryApplication() {
		return queryAppObj;
	}

	/**
	 * Wartet darauf, dass es eine positive Sendesteuerung für das Verschicken der Antworten gibt.
	 * Diese Methode wartet so lange, wie eine negative Sendesteuerung besteht.
	 *
	 * @return In der Regel {@link SendingState#SEND}, außer es tritt ein Fehler oder Timeout auf.
	 * Der Aufrufer muss den Rückgabewert auswerten und entsprechend reagieren.
	 * @throws InterruptedException Unterbrochen beim Warten
	 */
	private SendingState waitForSendControl() throws InterruptedException {
		final SendingState state;// Nach X Sekunden ohne Sendesteuerung wird der Auftrag abgebrochen
		final long timeoutTime = System.currentTimeMillis() + TimeoutManager.STANDARD_TIMEOUT;
		synchronized (syncObj) {
			while (sendingState == SendingState.STOP) {
				final long delta = timeoutTime - System.currentTimeMillis();
				if (delta <= 0) break;
				syncObj.wait(delta);
			}
			state = sendingState;
		}
		return state;
	}

	/**
	 * Sendet eine Fehlerantwort
	 *
	 * @param errorMessage Fehlermeldung
	 */
	public void sendErrorResponse(@NotNull String errorMessage) {
		sendResponse(false, errorMessage);
	}

	/**
	 * Sendet eine positive Antwort
	 */
	public void sendSuccessResponse() {
		sendResponse(true, "");
	}

	/**
	 * Sendet eine Fehlerantwort
	 *
	 * @param messageTypeResult Nachrichtentyp der Ergebnismeldung
	 * @param errorMessage      Fehlermeldung
	 */
	public void sendErrorResponse(int messageTypeResult, @NotNull String errorMessage) {
		sendResponse(messageTypeResult, false, errorMessage);
	}

	/**
	 * Sendet eine positive Antwort
	 *
	 * @param messageTypeResult Nachrichtentyp der Ergebnismeldung
	 */
	public void sendSuccessResponse(int messageTypeResult) {
		sendResponse(messageTypeResult, true, "");
	}

	/**
	 * Sendet eine beliebige Antwort
	 *
	 * @param success Rückmeldung erfolgreich?
	 * @param msg     Eventuelle Fehlermeldung oder ein Leerstring
	 */
	public void sendResponse(boolean success, @NotNull String msg) {
		sendResponse(messageTypeResult, success, msg);
	}

	/**
	 * Sendet eine beliebige Antwort
	 *
	 * @param messageTypeResult Nachrichtentyp der Ergebnismeldung
	 * @param success           Rückmeldung erfolgreich?
	 * @param msg               Eventuelle Fehlermeldung oder ein Leerstring
	 */
	public void sendResponse(int messageTypeResult, boolean success, @NotNull String msg) {
		sendResponse(messageTypeResult, success, msg, (it) -> {
		});
	}

	/**
	 * Sendet eine beliebige Antwort
	 *
	 * @param messageTypeResult Nachrichtentyp der Ergebnismeldung
	 * @param success           Rückmeldung erfolgreich?
	 * @param msg               Eventuelle Fehlermeldung oder ein Leerstring
	 * @param additionalData    Objekt, das per Callback noch weitere Nutzdaten in die Nachricht schreiben kann
	 */
	public void sendResponse(int messageTypeResult, boolean success, @NotNull String msg, DataWriter additionalData) {
		try {
			if (serializer == null) {
				_debug.warning(
						name + ": Kann keine Antwort senden, unbekannte Serializer-Version: " + serializerVersion
				);
				return;
			}

			try {
				subscribeSender();
			} catch (OneSubscriptionPerSendData e) {
				throw new AssertionError(e);
			}

			SendingState state = waitForSendControl();

			if (state != SendingState.SEND) {
				// Antwort kann nicht gesendet werden, weil keine Sendesteuerung innerhalb der Timeoutzeit empfangen wurde
				_debug.warning(
						name + ": Keine positive Sendesteuerung zum Verschicken der Antwort nach " + (SENDCONTROL_TIMEOUT / 1000)
								+ " Sekunden erhalten."
				);
				return;
			}

			Data data = archMgr.getDavCon().createData(atgQuery);
			data.getReferenceValue(PidQuery.ATT_SENDER_NAME).setSystemObject(archMgr.getArchiveObject());
			data.getUnscaledValue(PidQuery.ATT_QUERY_IDX_NAME).set(queryIdx);
			data.getUnscaledValue(PidQuery.ATT_MESSAGE_TYP_NAME).set(messageTypeResult);

			resultStream.reset();
			serializer.writeByte(success ? 1 : 0);
			if (!msg.isEmpty()) serializer.writeString(msg);
			else if (success) {
				additionalData.accept(serializer);
			}

			byte[] message = resultStream.toByteArray();
			Data.NumberArray dataBytes = data.getUnscaledArray(PidQuery.ATT_DATA_NAME);
			dataBytes.set(message);

			ResultData rd = new ResultData(queryAppObj, ddResponse, System.currentTimeMillis(), data);
			archMgr.getDavCon().sendData(rd);
			resultStream.reset();
		} catch (InterruptedException e) {
			_debug.warning("Archivsystem beendet. Versenden der Antwort abgebrochen.");
		} catch (Exception e) {
			_debug.warning("Versenden der Antwort fehlgeschlagen.", e);
		} finally {
			unsubscribeSender();
		}
	}

	@NotNull
	public Deserializer getDeserializer() throws NoSuchVersionException {
		if (serializer == null) {
			throw new NoSuchVersionException("Deserializer Serializer-Version: " + serializerVersion);
		}
		return deserializer;
	}

	/**
	 * Meldet den Task als Sender für Archiv-Antworten an das angegebene Empfaenger-Applikations-Objekt an. Registriert den StreamMultiplexer für den Empfang von
	 * Flusskontroll-Steuerungspaketen.
	 *
	 * @throws OneSubscriptionPerSendData Falls schon eine Anmeldung besteht
	 */
	private void subscribeSender() throws OneSubscriptionPerSendData {
		sendingState = SendingState.STOP;
		ConnectionManager.subscribeSender(archMgr.getDavCon(), this, queryAppObj, ddResponse, SenderRole.sender());
	}

	/**
	 * Meldet den Task als Sender für Archiv-Antworten an das angegebene Empfaenger-Applikations-Objekt ab.
	 */
	private void unsubscribeSender() {
		try {
			ConnectionManager.unsubscribeSender(archMgr.getDavCon(), this, queryAppObj, ddResponse);
		} catch (Exception e) {
			_debug.warning(name + " Sende-Abmeldung fehlgeschlagen.", e);
		}
	}// ////////////////////////////////////////////////////////////////////////////////////////////////////////

	// ClientSenderInterface Methoden
	// ////////////////////////////////////////////////////////////////////////////////////////////////////////
	@Override
	public void dataRequest(SystemObject object, DataDescription dataDescription, byte state) {
		synchronized (syncObj) {
			if (state == ClientSenderInterface.START_SENDING) {
				sendingState = SendingState.SEND;
			} else if (sendingState == SendingState.SEND) {
				sendingState = SendingState.ABORT;
			} else {
				sendingState = SendingState.STOP;
			}
			syncObj.notifyAll();
		}
	}

	@Override
	public boolean isRequestSupported(SystemObject object, DataDescription dataDescription) {
		return true; // Sendesteuerung ist erwuenscht und wird in dataRequest() verarbeitet.
	}
}
