/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence;

import de.bsvrz.ars.ars.persistence.iter.DataIterator;
import de.bsvrz.dav.daf.main.DataState;
import de.bsvrz.dav.daf.util.CloseableRandomAccessFile;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.bsvrz.sys.funclib.losb.util.ByteIO;
import de.bsvrz.sys.funclib.losb.util.Util;

import java.io.File;
import java.io.IOException;
import java.util.LinkedHashSet;
import java.util.NoSuchElementException;
import java.util.concurrent.atomic.AtomicInteger;

import static de.bsvrz.ars.ars.persistence.ContainerFile.*;

/**
 * Diese Klasse liest aus einem einzelnen Container Datensätze
 *
 * @author Kappich Systemberatung
 */
public final class ContainerDataIterator implements DataIterator {

	/**
	 * Maximalanzahl der gleichzeitig offenen Containerdateien
	 */
	private static int OPEN_FILE_LIMIT = 500;

	/**
	 * true, wenn über das Ende hinaus gelesen wurde
	 */
	private boolean finished;

	/**
	 * Aktuelle Container-Datei
	 */
	private final ContainerManagementData containerManagementData;

	/**
	 * Hier wird für z. B. die {@link #peek(ContainerDataResult)}-Funktion der nächste Datensatz gespeichert.
	 */
	private final ContainerDataResult next = new ContainerDataResult();

	/**
	 * Flag, ob bereits der nächste Datensatz gelesen wurde (für {@link #peek(ContainerDataResult)})
	 */
	private boolean dataRead;

	/**
	 * Puffer zum Einlesen von Längenbytes etc.
	 */
	private final byte[] byte8Buf = new byte[8];

	/**
	 * Container-Datei. Ist ein RandomAccessFile, da Rückspruenge noetig sind, wenn der letzte Datensatz nur teilweise in den Puffer eingelesen wurde.
	 */
	private final MyCloseableRandomAccessFile randomAccessFile;

	/**
	 * Länge des nächsten Datensatzes (wird in iterReadNextData() besetzt)
	 */
	private int nextDataLen;

	/**
	 * Anzahl gelesene Datensätze
	 */
	private int counter;

	/**
	 * Dateilänge, wird hier aus Performancegründen gecacht,
	 * da _randomAccessFile.length() laut profiling einige Zeit benötigt.
	 */
	private final long length;

	public static int getOpenFileLimit() {
		return OPEN_FILE_LIMIT;
	}

	public static void setOpenFileLimit(final int openFileLimit) {
		OPEN_FILE_LIMIT = openFileLimit;
	}

	/**
	 * Erstellt einen neuen DataIteratorAdapter
	 *
	 * @param containerFileHandle Datenquelle
	 */
	ContainerDataIterator(final StandaloneContainerFileHandle containerFileHandle) throws PersistenceException {
		containerManagementData = new ContainerHeaders(containerFileHandle);
		randomAccessFile = new MyCloseableRandomAccessFile(containerFileHandle.getFile().toFile());
		try {
			length = randomAccessFile.length();
			skipContainerHeader();
			finished = !readNextDataHdr();
		} catch (IOException e) {
			throw new PersistenceException("Container " + containerFileHandle + " ist nicht lesbar", e);
		}
		if (finished) {
			close();
		}
	}

	/**
	 * Springt im übergebenen Eingabestrom an die Stelle hinter dem Header der Container-Datei. Voraußetzung ist, dass das RandomAccessFile am Anfang der
	 * Datei steht.
	 *
	 * @throws IOException Allgemeiner IO-Fehler
	 */
	private void skipContainerHeader() throws IOException {
		randomAccessFile.readFully(byte8Buf, 0, VERSION_STRING.length + ByteIO.INT4B_LEN);
		randomAccessFile.skipBytes(ByteIO.readSignedInt4Bytes(byte8Buf, VERSION_STRING.length));
	}

	@Override
	public void peek(final ContainerDataResult result) throws PersistenceException {
		if (finished) {
			throw new NoSuchElementException();
		}
		if (!dataRead) {
			try {
				next.setData(readRawData());
			} catch (IOException e) {
				throw new PersistenceException("Datensatz im Container " + containerManagementData + " ist nicht lesbar", e);
			}
			dataRead = true;
		}
		next.copyTo(result);
	}

	@Nullable
	@Override
	public ContainerDataResult peekNext() {
		return null;
	}

	@Override
	public long peekDataIndex() {
		if (finished) {
			throw new NoSuchElementException();
		}
		return next.getDataIndex();
	}

	@Override
	public long peekDataTime() {
		if (finished) {
			throw new NoSuchElementException();
		}
		return next.getDataTime();
	}

	@Override
	public long peekArchiveTime() {
		if (finished) {
			throw new NoSuchElementException();
		}
		return next.getArchiveTime();
	}

	@Override
	public void remove() throws PersistenceException {
		if (finished) {
			throw new NoSuchElementException();
		}
		if (!dataRead) {
			try {
				skipData();
			} catch (IOException e) {
				throw new PersistenceException("Datensatz im Container " + containerManagementData + " konnte nicht übersprungen werden", e);
			}
		} else {
			next.setData(null);
			dataRead = false;
		}
		try {
			finished = !readNextDataHdr();
		} catch (IOException e) {
			throw new PersistenceException("Nächster Datensatz im Container " + containerManagementData + " ist nicht lesbar", e);
		}
		if (finished) {
			close();
		}
	}

	@Override
	public boolean isEmpty() {
		return finished;
	}

	@Override
	public void close() {
		try {
			randomAccessFile.close();
		} catch (IOException e) {
			Debug.getLogger().error("Container konnte nicht geschlossen werden: " + containerManagementData, e);
		}
		next.setData(null);
	}

	@NotNull
	@Override
	public ContainerManagementData getContainerManagementData() {
		return containerManagementData;
	}

	/**
	 * Liest den Datensatzheader
	 *
	 * @return true, wenn erfolgreich gelesen, false wenn am Containerende
	 * @throws IOException          Wenn beim Lesen ein Fehler aufgetreten ist
	 * @throws PersistenceException Wenn beim Lesen ein Fehler aufgetreten ist
	 */
	private boolean readNextDataHdr() throws IOException, PersistenceException {
		if (randomAccessFile.position() == length) return false; // EOF

		nextDataLen = randomAccessFile.readInt();

		next.setDataKind(containerManagementData.getLocation().archiveDataKind());
		next.setContainerID(containerManagementData.getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_CONT_ID));

		next.setArchiveTime(randomAccessFile.read6ByteUnsigned());
		next.setDataTime(randomAccessFile.read6ByteUnsigned());
		next.setDataIndex(randomAccessFile.readLong());
		int checkSum = randomAccessFile.readInt();

		int calcCheckSum = (int) (nextDataLen + next.getArchiveTime() + next.getDataTime() + next.getDataIndex());
		if (checkSum != calcCheckSum) {
			throw new IOException(
					"Prüfsumme des Datensatzes an Pos. " + counter + " fehlerhaft: " + checkSum + " != dx=" + next.getDataIndex()
							+ "+dt=" + next.getDataTime() + "+at=" + next.getArchiveTime() + "+len=" + nextDataLen + "=" + calcCheckSum + "\n" + this
			);
		}

		next.setDataSize(nextDataLen - COMPRESS_LEN - DATAHDR_LEN - ByteIO.SEPARATOR.length);
		int dataUncompressedSize = randomAccessFile.readInt();
		next.setDataUncompressedSize(dataUncompressedSize);
		next.setCompressed(dataUncompressedSize != NOT_COMPRESSED);
		DataState state;
		state = DataState.DATA;
		if (next.getDataSize() == 6) {
			long savedPosition = randomAccessFile.position();
			byte[] buffer = new byte[6];
			randomAccessFile.readFully(buffer);
			randomAccessFile.position(savedPosition);
			if (Util.cmpBytes(buffer, 0, NO_DATA)) {
				state = DataState.NO_DATA;
			} else if (Util.cmpBytes(buffer, 0, NO_SOURCE)) {
				state = DataState.NO_SOURCE;
			} else if (Util.cmpBytes(buffer, 0, NO_RIGHTS)) {
				state = DataState.NO_RIGHTS;
			} else if (Util.cmpBytes(buffer, 0, POT_GAP)) {
				state = DataState.POSSIBLE_GAP;
			}
		}
		next.setDataState(state);

		counter++;
		return true;
	}


	@Nullable
	private byte[] readRawData() throws IOException {
		int netDataLen = nextDataLen - COMPRESS_LEN - DATAHDR_LEN - ByteIO.SEPARATOR.length;

		if (!next.isData()) {
			randomAccessFile.skip(netDataLen + ByteIO.SEPARATOR.length);
			return null;
		} else {
			byte[] buf = new byte[netDataLen];
			randomAccessFile.readFully(buf);
			randomAccessFile.skip(ByteIO.SEPARATOR.length);
			return buf;
		}
	}

	private void skipData() throws IOException {
		int netDataLen = nextDataLen - COMPRESS_LEN - DATAHDR_LEN;
		randomAccessFile.skip(netDataLen);
	}

	/**
	 * Erweiterung von {@link CloseableRandomAccessFile}, die sich mit {@link #openFilesInCurrentThread} und numOpenFiles
	 * merkt, wie viele Dateien aktuell geöffnet sind, und bei zu viel Dateien die älteste Datei temporär schließt
	 */
	private static class MyCloseableRandomAccessFile extends CloseableRandomAccessFile {

		private static final ThreadLocal<LinkedHashSet<MyCloseableRandomAccessFile>> openFilesInCurrentThread = ThreadLocal.withInitial(LinkedHashSet::new);

		private static final AtomicInteger numOpenFiles = new AtomicInteger(0);

		MyCloseableRandomAccessFile(final File file) {
			super(file);
		}

		@Override
		protected void openFile() throws IOException {
			while (numOpenFiles.get() > getOpenFileLimit()) {
				MyCloseableRandomAccessFile first = CollectionUtilities.removeOldest(openFilesInCurrentThread.get());
				if (first != null) {
					first.closeFile(false);
				} else {
					break;
				}
			}

			// Offene Dateien zählen
			numOpenFiles.incrementAndGet();
			openFilesInCurrentThread.get().add(this);
			super.openFile();
		}

		@Override
		public void closeFile() throws IOException {
			closeFile(true);
		}

		void closeFile(final boolean removeFromQueue) throws IOException {
			if (!isOpen()) return;
			// Datei nicht mehr zählen, da geschlossen
			numOpenFiles.decrementAndGet();
			if (removeFromQueue) {
				openFilesInCurrentThread.get().remove(this);
			}
			super.closeFile();
		}


		long read6ByteUnsigned() throws IOException {
			return ((readUnsignedShort() & 0xFFFFL) << 32) | (readInt() & 0xFFFFFFFFL);
		}
	}
}
