/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence;

import de.bsvrz.dav.daf.main.DataState;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;

import java.util.Arrays;

/**
 * Klasse, in die Ergebnisse beim Iterieren von Containerdaten abgelegt werden. Diese Klasse dient als "Container" für
 * alle zurückgegebenen daten, ohne dass für jeden Methodenaufruf ein eigenes Objekt angelegt werden muss. Beispielcode:
 *
 * <pre>{@code
 * ContainerDataResult result = new ContainerDataResult();
 * while(!dataIter.isEmpty()) {
 *	 dataIter.peek(result); // Kopieren der Daten in die Variable "result"
 *	 // tue was mit den Daten in result
 *	 dataIter.remove(); // Nächsten Datensatz bearbeiten
 * }
 * }</pre>
 *
 * @author Kappich Systemberatung
 */
@SuppressWarnings("AssignmentOrReturnOfFieldWithMutableType")
public final class ContainerDataResult {
	private ArchiveDataKind _dataKind;
	@Nullable
	private byte[] _data;
	private boolean _isCompressed;
	private long _archiveTime;
	private long _dataIndex;
	private long _dataTime;
	private int _dataSize;
	private int _dataUncompressedSize;
	private long _containerID;
	private DataState _dataState;

	/**
	 * Liefert den serialisierten Datensatz
	 *
	 * @return Datensatz
	 */
	@Nullable
	public byte[] getData() {
		return _data;
	}

	/** @return Ob der Datensatz im Header als komprimiert gekennzeichnet wurde */
	public boolean isCompressed() {
		return _isCompressed;
	}

	/**
	 * Liefert den Datenindex des Datensatzes.
	 *
	 * @return Datenindex
	 */
	public long getDataIndex() {
		return _dataIndex;
	}

	/**
	 * Wird von der Persistenz benutzt, um den Datenindex des eingelesenen Datensatzes in diesem Iterator zu setzen.
	 *
	 * @param dataIndex Datenindex
	 */
	public void setDataIndex(long dataIndex) {
		this._dataIndex = dataIndex;
	}

	/**
	 * Liefert den Datenzeitstempel des Datensatzes.
	 *
	 * @return Datenzeitstempel in Millisekunden seit 1970.
	 */
	public long getDataTime() {
		return _dataTime;
	}

	/**
	 * Wird von der Persistenz benutzt, um den Datenzeitstempel des eingelesenen Datensatzes in diesem Iterator zu setzen.
	 *
	 * @param dataTime Datenzeitstempel
	 */
	public void setDataTime(long dataTime) {
		this._dataTime = dataTime;
	}

	/**
	 * Liefert die Markierung für diesen Datensatz.
	 *
	 * @return Datenmarkierung
	 */
	public DataState getDataState() {
		return _dataState;
	}

	/** @return Wahr, falls der Datensatz keine spezielle Markierung besitzt */
	public boolean isData() {
		return _dataState == DataState.DATA;
	}

	/**
	 * Gibt an, ob der eingelesene Datensatz eine "keine Daten"-Markierung besitzt.
	 *
	 * @return Wahr, falls der Datensatz eine "keine Daten"-Markierung besitzt
	 */
	public boolean isNoData() {
		return _dataState == DataState.NO_DATA;
	}

	/**
	 * Gibt an, ob der eingelesene Datensatz eine "keine Rechte"-Markierung besitzt.
	 *
	 * @return Wahr, falls der Datensatz eine "keine Rechte"-Markierung besitzt
	 */
	public boolean isNoRights() {
		return _dataState == DataState.NO_RIGHTS;
	}

	/**
	 * Gibt an, ob der eingelesene Datensatz eine "keine Quelle"-Markierung besitzt.
	 *
	 * @return Wahr, falls der Datensatz eine "keine Quelle"-Markierung besitzt
	 */
	public boolean isNoSource() {
		return _dataState == DataState.NO_SOURCE;
	}

	/**
	 * Gibt an, ob der eingelesene Datensatz als "potentielle Datenlücke" gekennzeichnet
	 * wurde.
	 *
	 * @return Wahr, falls der Datensatz als "potentielle Datenlücke" gekennzeichnet wurde
	 */
	public boolean isPotDataGap() {
		return _dataState == DataState.POSSIBLE_GAP;
	}

	public int getDataSize() {
		return _dataSize;
	}

	public void setDataSize(int dataSize) {
		this._dataSize = dataSize;
	}

	public int getDataUncompressedSize() {
		return _dataUncompressedSize;
	}

	public void setDataUncompressedSize(int dataUncompressedSize) {
		this._dataUncompressedSize = dataUncompressedSize;
	}

	public ArchiveDataKind getDataKind() {
		return _dataKind;
	}

	public long getArchiveTime() {
		return _archiveTime;
	}

	public void setDataKind(final ArchiveDataKind dataKind) {
		this._dataKind = dataKind;
	}

	public void setData(@Nullable final byte[] data) {
		_data = data;
	}

	public void setCompressed(final boolean compressed) {
		_isCompressed = compressed;
	}

	public void setArchiveTime(final long archiveTime) {
		_archiveTime = archiveTime;
	}

	public long getContainerID() {
		return _containerID;
	}

	public void setContainerID(final long containerID) {
		_containerID = containerID;
	}

	public void setDataState(final DataState dataState) {
		_dataState = dataState;
	}

	@Override
	public String toString() {
		return "ContainerDataResult{" +
				"_dataKind=" + _dataKind +
				", _data=" + Arrays.toString(_data) +
				", _dataState=" + _dataState +
				", _isCompressed=" + _isCompressed +
				", _archiveTime=" + _archiveTime +
				", _dataIndex=" + _dataIndex +
				", _dataTime=" + _dataTime +
				", _dataSize=" + _dataSize +
				", _dataUncompressedSize=" + _dataUncompressedSize +
				", _containerID=" + _containerID +
				'}';
	}

	/**
	 * Legt die Daten dieses Objekts im anderen Objekt ab.
	 * @param result anderes Objekt, das mit den Daten dieses Objekts überschrieben werden soll.
	 */
	public void copyTo(final ContainerDataResult result) {
		result.setDataState(_dataState);
		result.setArchiveTime(_archiveTime);
		result.setDataUncompressedSize(_dataUncompressedSize);
		result.setDataTime(_dataTime);
		result.setDataSize(_dataSize);
		result.setDataIndex(_dataIndex);
		result.setDataKind(_dataKind);
		result.setCompressed(_isCompressed);
		result.setContainerID(_containerID);
		result.setData(_data);
	}
}
