/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence;

import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;

/**
 * Interface für die Identifikation eines Containerverzeichnisses bzw. einer Datenidentifikation mit Datenart.
 * <p>
 * Dieses Interface identifiziert nur die Daten und mache keine Aussage, in welchem konkreten Verzeichnis auf dem
 * Dateisystem die Daten liegen, auch nicht welche(s) Wochenverzeichnis(se) ggf. berücksichtigt werden sollen.
 * <p>
 * Als Standard-Implementierung kann {@link IdContainerFileDir} verwendet werden, aber andere Klassen
 * können dieses Interface implementieren.
 */
public interface ContainerDirectory extends Comparable<ContainerDirectory> {
	/**
	 * Gibt die Objekt-ID zurück
	 *
	 * @return die Objekt-ID
	 */
	default long getObjectId() {
		return dataIdentification().getObjectId();
	}

	/**
	 * Gibt die Attributgruppen-ID zurück
	 *
	 * @return die Attributgruppen-ID
	 */
	default long getAtgId() {
		return dataIdentification().getAtgId();
	}

	/**
	 * Gibt die Aspekt-ID zurück
	 *
	 * @return die Aspekt-ID
	 */
	default long getAspectId() {
		return dataIdentification().getAspectId();
	}

	/**
	 * Gibt die Simulationsvariante zurück
	 *
	 * @return die Simulationsvariante
	 */
	default int getSimVariant() {
		return dataIdentification().getSimVariant();
	}

	/**
	 * Gibt die Archivdatenart zurück
	 *
	 * @return Archivdatenart
	 */
	ArchiveDataKind archiveDataKind();

	/**
	 * Gibt die Datenidentifikation zurück.
	 *
	 * @return die Datenidentifikation
	 */
	IdDataIdentification dataIdentification();

	@Override
	default int compareTo(@NotNull final ContainerDirectory o) {
		int tmp = dataIdentification().compareTo(o.dataIdentification());
		if (tmp != 0) {
			return tmp;
		}
		return Integer.compare(archiveDataKind().getCode(), o.archiveDataKind().getCode());
	}
}
