/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence;

import de.bsvrz.ars.ars.mgmt.datatree.DataIdentNode;
import de.bsvrz.ars.ars.mgmt.tasks.ArchiveDataSerializer;
import de.bsvrz.ars.ars.persistence.directories.PersistenceDirectory;
import de.bsvrz.ars.ars.persistence.layout.PersistenceDirectoryLayoutInstance;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;

import java.io.File;
import java.nio.file.Path;
import java.util.Objects;

/**
 * Klasse für den Zugriff auf eine Container-Datei. Diese Klasse implementiert {@link ContainerManagementData} um die Header-informationen aus der
 * Container-Datei zu lesen und {@link de.bsvrz.ars.ars.persistence.iter.DataSequence} zum Zugriff auf die Daten.
 * <p>
 * Außerdem implementiert die Klasse {@link AutoCloseable} um sicherzustellen, dass die Containerdatei zeitnah wieder geschlossen wird.
 *
 * @author Kappich Systemberatung
 */
public class ContainerFileHandle extends BasicContainerFileHandle {

	private final PersistenceDirectoryLayoutInstance persistenceDirectoryLayoutInstance;

	/**
	 * Erstellt eine neue Instanz zum Zugriff auf die Containerdatei
	 * @param containerDirectory Referenz auf die gelockte Datenidentifikation und Datenart für den Zugriff auf Containerdaten
	{@link DataIdentNode Datenidentifikations-Knoten} zum synchronisierten Zugriff auf die Datei
	 * @param containerId Container-ID, die geöffnet werden soll
	 * @param readonly Schreibgeschützter Zugriff? (Benötigt weniger Synchronisierung)
	 * @param persistenceDirectoryLayoutInstance Persistenzverzeichnis
	 * @throws PersistenceException Fehler beim Zugriff
	 */
	public ContainerFileHandle(final @NotNull ContainerDirectory containerDirectory, final long containerId, final boolean readonly, final @NotNull PersistenceDirectoryLayoutInstance persistenceDirectoryLayoutInstance) throws PersistenceException {
		super(containerDirectory);
		Objects.requireNonNull(containerDirectory, "containerDirectory == null");
		Objects.requireNonNull(persistenceDirectoryLayoutInstance, "persistenceDirectoryLayoutInstance == null");
		this.persistenceDirectoryLayoutInstance = persistenceDirectoryLayoutInstance;
		Path contDir = createDirectoriesIfAbsent();
		getContainerFile().accessContainer(containerId, contDir, readonly);
	}

	/**
	 * Liefert den Archiv-Pfad für die angegebene Datenidentifikation. Wenn das Verzeichnis nicht existiert, wird es angelegt (lazy-Verhalten).
	 *
	 * @return Archiv-Pfad
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	private Path createDirectoriesIfAbsent() throws PersistenceException {
		Path path = persistenceDirectoryLayoutInstance.getPath(getLocation());
		return PersistenceDirectory.createDirectoriesIfAbsent(path);
	}

	/**
	 * Prüft ob bereits eine Container-Datei existiert.
	 *
	 * @return Kennzeichen
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	public boolean existsContainer() throws PersistenceException {
		return getContainerFile().existsContainer();
	}


	/**
	 * Erzeugt eine neue Container-Datei für den mit spezifizierten Container. Schreibt den Container-Header und befuellt diesen mit den vorher gesetzten
	 * Parametern.
	 *
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	public void createContainer() throws PersistenceException {
		getContainerFile().createContainer(getLocation());
	}

	//////////////////////////////////////////////////////////////////////////////////////////////////////////
	// Container: Header
	//////////////////////////////////////////////////////////////////////////////////////////////////////////


	/**
	 * Nachdem mit {@link #readContainerHeader()} der Header einer Container-Datei eingelesen worden ist, kann mit dieser Methode der Wert eines Parameters im
	 * Container-Header geändert werden. Wirft eine {@link PersistenceException}, wenn die Änderung nicht durchgefuehrt werden kann.
	 *
	 * @param param Container-Header-Parameter
	 * @param val   Wert des Container-Header-Parameters
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	public void setContainerHeaderParam(KeyValParam param, String val) throws PersistenceException {
		getContainerFile().setContHdrParam(param, val);
	}

	/**
	 * Nachdem mit {@link #readContainerHeader()} der Header einer Container-Datei eingelesen worden ist, kann mit dieser Methode der Wert eines numerischen
	 * Parameters im Container-Header geändert werden. Wirft eine {@link PersistenceException}, wenn die Änderung nicht durchgefuehrt werden kann.
	 *
	 * @param param Container-Header-Parameter
	 * @param val   Numerischer Wert des Container-Header-Parameters
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	public void setContainerHeaderParam(KeyValParam param, long val) throws PersistenceException {
		getContainerFile().setContHdrParam(param, val);
	}

	/**
	 * Die mit {@link #setContainerHeaderParam(KeyValParam, String)} gesetzten Parameter werden im Container-Header persistiert.
	 * <p> 
	 * Diese Methode aktualisiert den Verwaltungsdatenindex nicht und kann z. B. benutzt werden, währen ein Container repariert wird bevor der Index aufgebaut ist.
	 *
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	public void writeContainerHeaderWithoutUpdatingIndex() throws PersistenceException {
		getContainerFile().writeContainerHeader();
	}


	//////////////////////////////////////////////////////////////////////////////////////////////////////////
	// Container: Datensätze
	//////////////////////////////////////////////////////////////////////////////////////////////////////////

	/**
	 * Ermittelt die Gesamtlänge des mit {@code accessContainer()}- im Zugriff befindlichen Containers.
	 *
	 * @return Länge der Container-Datei
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	public long getContainerSize() throws PersistenceException {
		return getContainerFile().getContainerSize();
	}

	/**
	 * Haengt den bereits übergebenen und serialisierten Datensatz an die aktuelle Container-Datei an.
	 *
	 * @param serializer Serialisierer
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	public void appendSerializedData(ArchiveDataSerializer serializer) throws PersistenceException {
		getContainerFile().appendSerializedData(serializer, getLocation().dataIdentification());
	}
}
