/*
 *
 * Copyright 2018-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence;

import com.google.common.collect.ImmutableList;
import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SyncKey;
import de.bsvrz.ars.ars.persistence.directories.PersistenceDirectory;
import de.bsvrz.ars.ars.persistence.walk.DataIdentificationDirAction;
import de.bsvrz.ars.ars.persistence.walk.internal.StandardDataidentificationDirWalk;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKindCombination;
import de.bsvrz.dav.daf.main.archive.ArchiveDataSpecification;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;

import java.nio.file.Path;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import static de.bsvrz.ars.ars.persistence.layout.DataKinds.getAvailableDataKinds;

/**
 * Klasse, die zu einem Datenidentifikationsverzeichnis auf der Festplatte die Datenidentifikation 
 * und Archivdatenart speichert, damit diese Informationen zusammen verwendet werden können,
 * vor allem bei Aufgaben die über die Persistenzdaten iterieren, z. B. {@link StandardDataidentificationDirWalk#execute(String, int, DataIdentificationDirAction)}.
 * <p>
 * D. h. im Gegensatz zu {@link IdDataIdentification} wird hier auch der Ort auf der Festplatte gespeichert,
 * was aus eine Abfrage nach vorhandenen Archivdatenarten etc. erlaubt.
 *
 * @author Kappich Systemberatung
 */
public class DataIdentificationDir {

	/**
	 * Verzeichnis
	 */
	private final Path _dataIdentificationDir;

	/**
	 * Datenidentifikation
	 */
	private final IdDataIdentification _dataIdentification;

	/**
	 * Ursprüngliche Archivdatenspezifikation (oder null)
	 */
	private Collection<ArchiveDataSpecification> _ads;

	/**
	 * Persistenzverzeichnis
	 */
	private final PersistenceDirectory _persistenceDirectory;

	/**
	 * Erstellt ein neues DataIdentificationDir-Objekt
	 * @param dataIdentificationDir Verzeichnis
	 * @param dataIdentification Datenidentifikation
	 */
	private DataIdentificationDir(Path dataIdentificationDir, IdDataIdentification dataIdentification,
	                              PersistenceDirectory persistenceDirectory) {
		_dataIdentificationDir = dataIdentificationDir;
		_dataIdentification = dataIdentification;
		_persistenceDirectory = persistenceDirectory;
	}

	/**
	 * Erstellt ein neues DataIdentificationDir-Objekt
	 *
	 * @param persistenceDirectory       (Wochen-)Verzeichnis-Objekt
	 * @param dataIdentificationFromPath Datenidentifikation
	 * @param dataIdentificationDir      Verzeichnis der Datenidentifikation auf Festplatte
	 */
	public DataIdentificationDir(PersistenceDirectory persistenceDirectory, IdDataIdentification dataIdentificationFromPath, Path dataIdentificationDir) {
		this(dataIdentificationDir, dataIdentificationFromPath, persistenceDirectory);
	}

	/**
	 * Erstellt ein neues ContainerFileDir-Objekt
	 *
	 * @param persistenceDirectory (Wochen-)Verzeichnis-Objekt
	 * @param dataIdentification   Datenidentifikation
	 */
	public DataIdentificationDir(PersistenceDirectory persistenceDirectory, IdDataIdentification dataIdentification) {
		this(persistenceDirectory.getPath(dataIdentification), dataIdentification, persistenceDirectory);
	}

	/**
	 * Erstellt ein neues ContainerFileDir-Objekt
	 * @param archiveDataSpecifications Datenspezifikationen
	 * @param persistenceDirectory (Wochen-)Verzeichnis-Objekt
	 * @param dataIdentification Datenidentifikation
	 */
	public DataIdentificationDir(final PersistenceDirectory persistenceDirectory, final IdDataIdentification dataIdentification, final Collection<ArchiveDataSpecification> archiveDataSpecifications) {
		this(persistenceDirectory, dataIdentification);
		_ads = ImmutableList.copyOf(archiveDataSpecifications);
	}

	/**
	 * Gibt das Verzeichnis zurück
	 * @return das Verzeichnis
	 */
	public Path getDataIdentificationDir() {
		return _dataIdentificationDir;
	}

	/**
	 * Gibt die Datenidentifikation zurück
	 * @return die Datenidentifikation
	 */
	public IdDataIdentification getDataIdentification() {
		return _dataIdentification;
	}
	/**
	 * Gibt die Archivdatenspezifikation zurück (falls sie beim Konstruktor übergeben wurde)
	 * @return Archivdatenspezifikation oder null
	 */
	@Nullable
	public Collection<ArchiveDataSpecification> getArchiveDataSpecifications() {
		return _ads;
	}

	@Override
	public String toString() {
		return _dataIdentification.toString();
	}

	/**
	 * Gibt die Archivdatenarten zurück, die relevant sind. Wenn dieses Objekt mit einer oder mehreren {@link ArchiveDataSpecification}-parametern erzeugt wird, werden die angefragten Archivdatenarten zurückgegeben.
	 * <p>
	 * Wenn keine {@link ArchiveDataSpecification} übergeben wurde, werden auf der Festplatte die vorhandenen
	 * Archivdatenart-Verzeichnisse ermittelt.
	 *
	 * @return Menge von Archivdatenarten
	 * @throws PersistenceException IO-Fehler beim Lesen der vorhandenen Verzeichnisse
	 */
	public ArchiveDataKindCombination getArchiveDataKinds() throws PersistenceException {
		if(_ads != null) {
			List<ArchiveDataKind> dataKinds = 
					_ads.stream()
					.flatMap(it -> it.getDataKinds().stream())
					.collect(Collectors.toList());
			
			return new ArchiveDataKindCombination(dataKinds);
		}
		return getAvailableDataKinds(this.getDataIdentificationDir());
	}

	/**
	 * Ermittelt alle relevanten Containerverzeichnisse für dieses Datenidentifikationsverzeichnis. Diese Methode gibt für
	 * jede {@linkplain #getArchiveDataKinds() Archivdatenart} das zugehörige Containerverzeichnis-Objekt
	 * zurück.
	 * @return Liste mit {@link ContainerDirectory}-Objekten
	 * @throws PersistenceException IO-Fehler beim Lesen der vorhandenen Verzeichnisse
	 */
	public Collection<? extends ContainerDirectory> getContainerDirectories() throws PersistenceException {
		IdDataIdentification dataIdentification = getDataIdentification();
		return getArchiveDataKinds().stream().map(dataIdentification::resolve).collect(Collectors.toList());
	}

	/**
	 * Ermittelt alle relevanten Containerverzeichnisse für dieses Datenidentifikationsverzeichnis. Diese Methode gibt für
	 * jede {@linkplain #getArchiveDataKinds() Archivdatenart} das zugehörige Containerverzeichnis-Objekt
	 * zurück.
	 * <p>
	 * Im Gegensatz zu {@link #getContainerDirectories()} ohne Parameter wird hier die Synchronisierung auf die zurückgegeben
	 * Objekte übertragen.
	 * @param lock Synchronisierung auf die DatenIdentifikation
	 * @return Liste mit {@link LockedContainerDirectory}-Objekten
	 * @throws PersistenceException IO-Fehler beim Lesen der vorhandenen Verzeichnisse
	 */
	public Collection<? extends LockedContainerDirectory> getContainerDirectories(SyncKey<IdDataIdentification> lock) throws PersistenceException {
		assert lock.getElement().equals(getDataIdentification());
		return getArchiveDataKinds().stream().map(it -> new LockedContainerDirectory(lock, it)).collect(Collectors.toList());
	}

	/**
	 * Gibt das Persistenzverzeichnis zurück.
	 * @return das Persistenzverzeichnis
	 */
	public PersistenceDirectory getPersistenceDirectory() {
		return _persistenceDirectory;
	}

}
