/*
 *
 * Copyright 2017-2019 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence;

import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.Dataset;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.dav.daf.main.archive.ArchiveDataSpecification;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.dataIdentificationSettings.DataIdentification;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;

/**
 * Klasse, die die IDs der Objekte einer Datenidentifikation aufnehmen kann. Diese Klasse ist <em>value based</em>,
 * <em>immutable</em> und es werden keine Aussagen über Objektidentität getroffen.
 * @author Kappich Systemberatung
 */
public final class IdDataIdentification implements Comparable<IdDataIdentification> {

	/** Id des Systemobjekts dieser Datenidentifikation */
	private final long _objectId;

	/** Id der Attributgruppe dieser Datenidentifikation */
	private final long _atgId;

	/** Id des Aspekts dieser Datenidentifikation */
	private final long _aspectId;

	/** Simulationsvariante dieser Datenidentifikation */
	private final int _simVariant;

	/**
	 * Erzeugt eine neue {@link IdDataIdentification} basierend auf der Datenidentifikation eines DataSet-Objekts
	 *
	 * @param dataset Datensatz
	 */
	public IdDataIdentification(Dataset dataset) {
		this(dataset.getObject(), dataset.getDataDescription());
	}
	
	/**
	 * Erzeugt ein neues Objekt mit den angegebenen Werten
	 * @param object Objekt
	 * @param dataDescription {@link DataDescription}
	 */
	public IdDataIdentification(final SystemObject object, DataDescription dataDescription) {
		_objectId = object.getId();
		_atgId = dataDescription.getAttributeGroup().getId();
		_aspectId = dataDescription.getAspect().getId();
		short simVariant = dataDescription.getSimulationVariant();
		if (simVariant == -1) {
			// Workaround für nachgelieferte Archivdaten, die mit SV==-1 ankommen
			simVariant = 0;
		}
		_simVariant = simVariant;
	}

	/**
	 * Erzeugt ein neues Objekt mit den angegebenen Werten
	 *
	 * @param objectId          Id des Systemobjekts dieser Datenidentifikation
	 * @param atgId             Id der Attributgruppe dieser Datenidentifikation
	 * @param aspectId          Id des Aspekts dieser Datenidentifikation
	 * @param simulationVariant Simulationsvariante dieser Datenidentifikation
	 */
	public IdDataIdentification(final long objectId, final long atgId, final long aspectId, final int simulationVariant) {
		_objectId = objectId;
		_atgId = atgId;
		_aspectId = aspectId;
		_simVariant = simulationVariant;
	}

	/**
	 * Erstellt eine neue Instanz basiert auf einer {@link ArchiveDataSpecification}
	 * @param ads {@link ArchiveDataSpecification}
	 */
	public IdDataIdentification(final ArchiveDataSpecification ads) {
		this(ads.getObject(), ads.getDataDescription());
	}

	/**
	 * Erstellt eine neue Instanz aus einer {@link DataIdentification}
	 * @param dataIdentification {@link DataIdentification}
	 */
	public IdDataIdentification(final DataIdentification dataIdentification) {
		this(dataIdentification.getObject(), dataIdentification.getDataDescription());
	}

	/**
	 * Gibt den Containerort der angegebenen Datenart zurück
	 *
	 * @param archiveDataKind Datenart
	 * @return ein IdContainerFileDir
	 */
	public IdContainerFileDir resolve(ArchiveDataKind archiveDataKind) {
		return new IdContainerFileDir(this, archiveDataKind);
	}

	@Override
	public boolean equals(final Object o) {
		if(this == o) return true;
		if(o == null || getClass() != o.getClass()) return false;

		final IdDataIdentification that = (IdDataIdentification) o;

		if(_objectId != that._objectId) return false;
		if(_atgId != that._atgId) return false;
		if(_aspectId != that._aspectId) return false;
		return _simVariant == that._simVariant;
	}

	@Override
	public int hashCode() {
		int result = (int) (_objectId ^ (_objectId >>> 32));
		result = 31 * result + (int) (_atgId ^ (_atgId >>> 32));
		result = 31 * result + (int) (_aspectId ^ (_aspectId >>> 32));
		result = 31 * result + _simVariant;
		return result;
	}

	/**
	 * Gibt die Objekt-ID zurück
	 * @return die Objekt-ID
	 */
	public long getObjectId() {
		return _objectId;
	}

	/**
	 * Gibt die Attributgruppen-ID zurück
	 * @return die Attributgruppen-ID
	 */
	public long getAtgId() {
		return _atgId;
	}

	/**
	 * Gibt die Aspekt-ID zurück
	 * @return die Aspekt-ID
	 */
	public long getAspectId() {
		return _aspectId;
	}

	/**
	 * Gibt die Simulationsvariante zurück
	 * @return die Simulationsvariante
	 */
	public int getSimVariant() {
		return _simVariant;
	}

	public String toString() {
		return "obj: " + _objectId + " atg: " + _atgId + " asp: " + _aspectId + " sv: " + _simVariant;
	}

	@Override
	public int compareTo(@NotNull final IdDataIdentification o) {
		int tmp;
		tmp = Long.compare(getObjectId(), o.getObjectId());
		if(tmp != 0) return tmp;
		tmp = Long.compare(getAtgId(), o.getAtgId());
		if(tmp != 0) return tmp;
		tmp = Long.compare(getAspectId(), o.getAspectId());
		if(tmp != 0) return tmp;
		return Integer.compare(getSimVariant(), o.getSimVariant());
	}
}
