/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence;


import de.bsvrz.sys.funclib.losb.util.Util;

/**
 * Diese Klasse definiert die Parameter im Container-Header. Dazu gehören Schluesselname, max. Länge des Wertes, Wertebereich und Default-Wert. Es gibt Text-
 * und Zahl-Parameter. Verwaltet werden diese Parameter von der Klasse {@link ContainerHdr}.
 *
 * @author beck et al. projects GmbH
 * @author Thomas Schaefer
 * @author Alexander Schmidt
 * @version $Revision$ / $Date$ / ($Author$)
 */
public class KeyValParam {

	private static final String TYPE_STRING = "String";

	private static final String TYPE_NUM = "Zahl";

	private String key;

	private String defaultValue;

	private String[] allowedStringValues;

	private int valueLength;

	private long minVal, maxVal;

	private String paramType;

	private int byteLength;

	/**
	 * Erzeugt einen neuen Container-Header-Parameter (Text).
	 *
	 * @param key          Schluessel des Parameters
	 * @param defaultValue Vorgabe-Wert
	 * @param valueLength  Länge des Werts
	 * @return Text-Parameter
	 */
	public static KeyValParam createStringParam(String key, String defaultValue, int valueLength) {
		return createStringParam(key, defaultValue, valueLength, null);
	}

	/**
	 * Erzeugt einen neuen Container-Header-Parameter.
	 *
	 * @param key           Schluessel des Parameters
	 * @param defaultValue  Vorgabe-Wert
	 * @param valueLength   Länge des Werts
	 * @param allowedValues Erlaubte Werte
	 * @return Wert-Parameter
	 */
	public static KeyValParam createStringParam(String key, String defaultValue, int valueLength, String[] allowedValues) {
		KeyValParam chp = new KeyValParam();
		chp.paramType = TYPE_STRING;
		chp.allowedStringValues = allowedValues;
		chp.key = key;
		chp.defaultValue = defaultValue;
		chp.valueLength = valueLength;
		chp.minVal = -1;
		chp.maxVal = -1;
		chp.byteLength = valueLength;
		return chp;
	}

	/**
	 * Erzeugt einen neuen Container-Header-Parameter.
	 * @param key          Schluessel des Parameters
	 * @param defaultValue Vorgabe-Wert
	 * @param valueLength  Länge des Werts
	 * @param minVal       minimaler Wert
	 * @param maxVal       maximaler Wert
	 * @param byteLength   Vorgesehene Länge in bytes
	 * @return KeyValParam
	 */
	public static KeyValParam createNumParam(String key, long defaultValue, int valueLength, long minVal, long maxVal, final int byteLength) {
		KeyValParam chp = new KeyValParam();
		chp.paramType = TYPE_NUM;
		chp.key = key;
		chp.valueLength = valueLength;
		chp.minVal = minVal;
		chp.maxVal = maxVal;
		chp.defaultValue = chp.formatVal(defaultValue);
		chp.byteLength = byteLength;

		assert chp.defaultValue.length() <= valueLength;
		assert chp.formatVal(minVal).length() <= valueLength;
		assert chp.formatVal(maxVal).length() <= valueLength;
		assert minVal <= defaultValue && defaultValue <= maxVal;
		return chp;
	}

	/**
	 * Prüft, ob ein Wert gesetzt werden könnte, wirft andernfalls eine {@link PersistenceException}.
	 *
	 * @param val zu prüfender Wert
	 * @throws PersistenceException Exception, falls Wert ungültig ist.
	 */
	public void checkVal(long val) throws PersistenceException {
		if (isNumeric()) {
			if (!(minVal <= val && val <= maxVal))
				throw new PersistenceException("Ungueltiger Parameterwert (" + val + ") fuer " + this);
		} else {
			checkStringVal(Long.toString(val));
		}
	}

	/**
	 * Prüft, ob ein Wert gesetzt werden könnte, wirft andernfalls eine {@link PersistenceException}.
	 * @param val zu prüfender Wert
	 * @throws PersistenceException Exception, falls Wert ungültig ist.
	 */
	public void checkVal(String val) throws PersistenceException {
		if(isNumeric()) {
			long lVal;
			try {
				lVal = Long.parseLong(val);
			}
			catch(NumberFormatException e) {
				throw new PersistenceException("Wert ist nicht als Zahlwert interpretierbar: '" + val + "' " + this, e);
			}
			if (!(minVal <= lVal && lVal <= maxVal))
				throw new PersistenceException("Ungueltiger Parameterwert (" + val + ") fuer " + this);
		} else {
			checkStringVal(val);
		}
	}

	/**
	 * Prüft, ob ein String-Wert in einem String-Header-Feld gesetzt werden kann, wirft andernfalls eine {@link PersistenceException}.
	 *
	 * @param stringVal zu prüfender Wert
	 * @throws PersistenceException Exception, falls Wert ungültig ist.
	 */
	private void checkStringVal(String stringVal) throws PersistenceException {
		if (stringVal.length() > valueLength) {
			throw new PersistenceException("Ungueltiger Parameterwert (" + stringVal + ") fuer " + this);
		}
		else if(allowedStringValues != null) {
			for (String value : allowedStringValues) {
				if (value.equals(stringVal)) return;
			}
			throw new PersistenceException("Ungueltiger Parameterwert (" + stringVal + ") fuer " + this);
		}
	}

	/**
	 * Liefert den Schluessel des Container-Header-Parameter.
	 *
	 * @return Schluessel
	 */
	public String getKey() {
		return key;
	}

	/**
	 * Ergaenzt den angegebenen Wert um fuehrende Nullen bis zur max. Länge des Werts dieses Parameters. Bei negativen Zahlen wird das "-"-Zeichen links vor
	 * den Nullen eingefuegt.
	 *
	 * @param val Wert
	 *
	 * @return Wert mit fuehrenden Nullen
	 */
	public String formatVal(long val) {
		if(val < 0) {
			return "-" + Util.leadZero(-val, valueLength - 1);
		}
		else {
			return Util.leadZero(val, valueLength);
		}
	}

	/**
	 * Ergaenzt den angegebenen Wert um Leerzeichen bis zur max. Länge des Werts dieses Parameters.
	 *
	 * @param val Wert
	 *
	 * @return Wert mit angehaengten Leerzeichen oder fuehrenden Nullen
	 */
	public String formatVal(String val) {
		return Util.sr(val, valueLength);
	}

	/**
	 * Liefert des Default-Wert für diesen Parameter.
	 *
	 * @return Default-Wert
	 */
	public String getDefaultVal() {
		return defaultValue;
	}

	/**
	 * Liefert die Lanege des Werts (Anzahl an Zeichen).
	 *
	 * @return Länge des Werts
	 */
	public int getValLen() {
		return valueLength;
	}

	public boolean isNumeric() {
		return paramType.equals(TYPE_NUM);
	}

	public int getByteLength() {
		return byteLength;
	}

	public String toString() {
		StringBuilder erg = new StringBuilder("Container-Parameter '" + getKey() + "' (" + paramType + ") Laenge=" + getValLen() + " ");
		if (isNumeric()) {
			erg.append("min=").append(minVal).append(", max=").append(maxVal);
		}
		else {
			if (allowedStringValues != null) {
				erg.append(", Wertebereich {");
				for (String value : allowedStringValues) {
					erg.append(value).append(",");
				}
				erg = new StringBuilder(Util.removeLastChar(erg.toString()) + "}");
			}
		}
		return erg.toString();
	}
}
