/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2019 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence;

import java.util.Properties;

/**
 * Klasse zum Lesen/Schreiben von Key/Value-Paaren. Die Values haben stets feste Länge..
 *
 * @author beck et al. projects GmbH
 * @author Alexander Schmidt
 * @version $Revision$ / $Date$ / ($Author$)
 */
public abstract class KeyValProps {

	protected final Properties properties;

	// --------------------------------------------------------------------------------------------------------
	// Längen der Parameter in Textdarstellung:

	public static final int BYTES4_STR_LEN = 10;	// reicht für -1 .. Integer.MAXVALUE

	public static final int BYTES5_STR_LEN = 13;

	public static final int BYTES6_STR_LEN = 15;

	public static final int BYTES8_STR_LEN = 19;

	/** Länge eines Boolschen Werts bei textueller Repraesentation mit '0' und '1'. */
	public static final int BOOL_STR_LEN = 1;

	// --------------------------------------------------------------------------------------------------------
	// Wertebereiche der numerischen Parameter:

	protected static final long BYTE4_MAXVAL = Integer.MAX_VALUE;	// erlaubt Zuweisung auf int-Variable

	protected static final long BYTE5_MAXVAL = 0xFFFFFFFFFFL;

	protected static final long BYTE6_MAXVAL = 0xFFFFFFFFFFFFL;

	protected static final long BYTE8_MAXVAL = Long.MAX_VALUE;


	/** Textuelle Repraesentation des boolsches Kennzeichen für 'Wahr'. */
	protected static final String TRUE = "1";

	/** Textuelle Repraesentationdes boolsches Kennzeichen für 'Falsch'. */
	protected static final String FALSE = "0";

	/** Delimiter für Parameter im Container-Header. */
	protected static final String CH_DELIM = "\n";


	public KeyValProps() {
		properties = new Properties();
	}

	public void clear() {
		properties.clear();
	}

	public void setVal(KeyValParam chp, boolean val) throws PersistenceException {
		String txtVal = val ? TRUE : FALSE;
		chp.checkVal(txtVal);
		properties.setProperty(chp.getKey(), chp.formatVal(txtVal));
	}

	public void setVal(KeyValParam chp, long val) throws PersistenceException {
		chp.checkVal(String.valueOf(val));
		properties.setProperty(chp.getKey(), chp.formatVal(val));
	}

	public void setVal(KeyValParam chp, String val) throws PersistenceException {
		chp.checkVal(val);
		properties.setProperty(chp.getKey(), chp.formatVal(val));
	}

	public void setDefaultVal(KeyValParam chp) {
		properties.setProperty(chp.getKey(), chp.getDefaultVal());
	}

	public String getValAsString(KeyValParam chp) {
		return properties.getProperty(chp.getKey());
	}

	public Long getValAsLong(KeyValParam chp) throws PersistenceException {
		if(!chp.isNumeric()) {
			throw new PersistenceException("ContainerHeaderParam ist kein Zahlwert: " + chp + " " + this);
		}
		else {
			String val = properties.getProperty(chp.getKey());
			return (val == null) ? null : Long.valueOf(val);
		}
	}

	public Boolean getValAsBool(KeyValParam chp) throws PersistenceException {
		if(chp.isNumeric()) {
			throw new PersistenceException("Parameter ist numerisch und nicht als Bool interpretierbar: " + chp + " " + this);
		}
		else {
			String val = properties.getProperty(chp.getKey());
			return (val == null) ? null : val.equals(TRUE);
		}
	}

	public Integer getValAsInt(KeyValParam chp) throws PersistenceException {
		Long l = getValAsLong(chp);
		if(l > BYTE4_MAXVAL) {
			throw new PersistenceException("Parameter ist nicht als Int (4 Bytes) interpretierbar: '" + l + "' " + chp + " " + this);
		}
		else {
			return l.intValue();
		}
	}


}
