/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence;

import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SyncKey;
import de.bsvrz.ars.ars.mgmt.tasks.AbstractTask;
import de.bsvrz.ars.ars.mgmt.tasks.base.TaskManager;
import de.bsvrz.ars.ars.persistence.directories.ActivePersistenceDirectory;
import de.bsvrz.ars.ars.persistence.directories.PersistenceDirectory;
import de.bsvrz.ars.ars.persistence.index.ArchiveTimeIndexImpl;
import de.bsvrz.ars.ars.persistence.index.ContainerManagementIndex;
import de.bsvrz.ars.ars.persistence.index.DataIndexIndexImpl;
import de.bsvrz.ars.ars.persistence.index.DataTimeIndexImpl;
import de.bsvrz.ars.ars.persistence.index.result.IndexResult;
import de.bsvrz.ars.ars.persistence.walk.DataIdentificationDirWalk;
import de.bsvrz.ars.ars.persistence.walk.internal.StatusPrinter;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.losb.util.Util;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.util.List;
import java.util.concurrent.atomic.AtomicLong;

/**
 * Diese Klasse teilt das Persistenzverzeichnis in disjunkte Teilbaeume auf und stellt alle Verwaltungsinformationen wieder her.
 *
 * @author beck et al. projects GmbH
 * @author Alexander Schmidt
 * @version $Revision$ / $Date$ / ($Author$)
 * @see ActivePersistenceDirectory#REBUILD_INDEX_FILE_FLAG_NAME
 */
public final class RestorePersDirTsk extends AbstractTask {

	/**
	 * damit man nach dem cleanup noch zu testzwecken auf die ergebnisse zugreifen kann:
	 */
	public static volatile long test_totalDID;
	public static volatile long test_badDID;
	public static volatile long test_goodDID;

	private final RestoreWorker restoreWorker;

	/**
	 * Erstellt einen neuen RestorePersDirTsk.
	 *
	 * @param persistenceManager  Archivmanager
	 * @param rebuildMode Wiederherstellungs-Modus
	 */
	public RestorePersDirTsk(PersistenceManager persistenceManager, String taskName, RebuildMode rebuildMode) {
		super(persistenceManager);
		restoreWorker = new RestoreWorker(taskName, rebuildMode, getPersistenceManager(), persistenceManager.getNumCheckPersistenceThreads());
		start();
	}

	/**
	 * Durchlaueft mit Hilfe eines {@link DataIdentificationDirWalk} jedes Containerverzeichnis und stellt die Verwaltungsinformation wieder her.
	 */
	@Override
	public void execute() {
		List<? extends PersistenceDirectory> directories = getPersistenceManager().getPersistenceDirectories(0);
		if (!directories.isEmpty()) {
			restoreWorker.doRestore(directories.get(directories.size() - 1));
		}
		super.execute();
	}

	@Override
	public void terminateTask() {
		super.terminateTask();
		restoreWorker.shouldTerminate = true;
	}

	public RestoreWorker getWorker() {
		return restoreWorker;
	}

	/**
	 * Eigentliche Klasse, die den Wiederherstellungslauf durchführt
	 */
	public static class RestoreWorker {

		private final AtomicLong maxArchiveTime = new AtomicLong();
		private final AtomicLong maxContID = new AtomicLong();

		private final AtomicLong totalDID = new AtomicLong();
		private final AtomicLong badDID = new AtomicLong();
		private final AtomicLong goodDID = new AtomicLong();        // Zähler für DID

		/**
		 * Zähler für unerwartete RuntimeExceptions
		 */
		private final AtomicLong _errorCount = new AtomicLong();

		private final AtomicLong containerRead = new AtomicLong();
		private final AtomicLong containerIgnored = new AtomicLong();
		private final AtomicLong indexesDeleted = new AtomicLong();

		private final String taskName;
		private final RebuildMode rebuildMode;

		private final int numThreads;
		private final DataIdentificationManager dataIdentificationManager;

		private volatile boolean shouldTerminate;

		public RestoreWorker(String taskName, RebuildMode rebuildMode, DataIdentificationManager dataIdentificationManager, int numCheckPersistenceThreads) {
			this.taskName = taskName;
			this.rebuildMode = rebuildMode;
			this.dataIdentificationManager = dataIdentificationManager;
			numThreads = numCheckPersistenceThreads;
		}

		/**
		 * Gibt den benutzten Speicher frei (insbes. den von den einzelnen Tasks zum Wiederherstellen der Indexe benoetigten Speicher)
		 */
		private void cleanUpTask() {
			// damit man nach dem cleanup noch zu testzwecken auf die ergebnisse zugreifen kann:
			test_totalDID = getTotalDID();
			test_badDID = getBadDID();
			test_goodDID = getGoodDID();
		}

		public long getMaxArchiveTime() {
			return maxArchiveTime.get();
		}

		public long getMaxContID() {
			return maxContID.get();
		}

		public long getTotalDID() {
			return totalDID.get();
		}

		public long getBadDID() {
			return badDID.get();
		}

		public long getGoodDID() {
			return goodDID.get();
		}

		public long getContainerRead() {
			return containerRead.get();
		}

		public long getContainerIgnored() {
			return containerIgnored.get();
		}

		public long getIndexesDeleted() {
			return indexesDeleted.get();
		}

		public long getErrorCount() {
			return _errorCount.get();
		}

		/**
		 * @return Wahr, falls kein Verzeichnis nicht wiederhergestellt werden konnte, falsch sonst
		 */
		public boolean success() {
			return getBadDID() == 0 && getErrorCount() == 0;
		}

		/**
		 * Indexdateien wiederherstellen
		 *
		 * @param persistenceDirectory Persistenzverzeichnis
		 */
		public void doRestore(PersistenceDirectory persistenceDirectory) {
			try {
				if (persistenceDirectory == null) {
					return;
				}
				DataIdentificationDirWalk containerDirWalk = DataIdentificationDirWalk.allDirectories(persistenceDirectory);
				containerDirWalk.setStatusPrinter(new StatusPrinter() {
					@Override
					public String getStatusMessage(final String actionName, final Duration runtime, final ApproximationType approximationType, final long allContainerDirs, final long visitedContainerDirs, final long visitedContainerFiles) {
						long goodDirs = getGoodDID();
						long badDirs = getBadDID();
						long sum = goodDirs + badDirs;
						return super.getStatusMessage(actionName, runtime, approximationType, allContainerDirs, visitedContainerDirs, visitedContainerFiles) + "\n"
								+ getProgressMessage(goodDirs, badDirs, sum);
					}

					@Override
					public String getSuccessMessage(final String actionName, final Duration runtime, final long visitedContainerDirs, final long visitedContainerFiles) {
						long goodDirs = getGoodDID();
						long badDirs = getBadDID();
						long sum = goodDirs + badDirs;
						return super.getSuccessMessage(actionName, runtime, visitedContainerDirs, visitedContainerFiles) + "\n"
								+ getProgressMessage(goodDirs, badDirs, sum);
					}
				});

				TaskManager.runExclusively(taskName, tpi -> {
					containerDirWalk.setPublisherTask(tpi);
					doWalk(containerDirWalk, persistenceDirectory);
				});
			} catch (PersistenceException e) {
				_errorCount.getAndIncrement();
				_debug.error(taskName + ": Unbehandelter Fehler in RestorePersDirTsk", e);
			} finally {
				cleanUpTask();
			}
		}

		private void doWalk(DataIdentificationDirWalk containerDirWalk, PersistenceDirectory directory) throws PersistenceException {
			containerDirWalk.execute(
					taskName,
					numThreads,
					(dataIdentificationDir, standardContainerDirWalk) -> {
						Path didDir = dataIdentificationDir.getDataIdentificationDir();
						IdDataIdentification dataIdentification = dataIdentificationDir.getDataIdentification();
						if (shouldTerminate) {
							containerDirWalk.terminate();
						}
						try (SyncKey<IdDataIdentification> lock = dataIdentificationManager.lockIndex(dataIdentification)) {
							for (LockedContainerDirectory containerDirectory : dataIdentificationDir.getContainerDirectories(lock)) {
								try {
									totalDID.getAndIncrement();

									// Indexe ggf. wiederherstellen:

									Path changedFlagFile = directory.getPath(containerDirectory).resolve(
											ActivePersistenceDirectory.REBUILD_INDEX_FILE_FLAG_NAME);

									boolean needsRebuild = isNeedsRebuild(directory, containerDirectory, changedFlagFile);

									if (needsRebuild || rebuildMode.isFull()) {
										long t2 = Util.startTimer();
										RebuildResult result = new RebuildResult();
										if (rebuildMode.isDelete()) {
											directory.deleteIndexes(containerDirectory, result);
										} else {
											directory.rebuildIndexes(containerDirectory, result);
										}
										containerRead.addAndGet(result.getContainerRead());
										containerIgnored.addAndGet(result.getContainerIgnored());
										indexesDeleted.addAndGet(result.getIndexesDeleted());
										try {
											Files.deleteIfExists(changedFlagFile);
										} catch (IOException e) {
											throw new IOException(
													taskName + ": '" + ActivePersistenceDirectory.REBUILD_INDEX_FILE_FLAG_NAME + "' im Verzeichnis" +
															" '" + didDir
															+ "' konnte nicht geloescht werden.", e
											);
										}
										goodDID.getAndIncrement();
										_debug.fine(taskName + ": Indexe in " + didDir + " erfolgreich neu aufgebaut (" + Util.stopTimer(t2) + ").");

									}

									// maximale Archivzeit und Container-ID ermitteln
									IndexResult<ContainerManagementInformation> headers = directory.getContainerHeaders(containerDirectory);
									if (!headers.isEmpty()) {
										long contId = headers.getMax(ContainerManagementInformation.CHP_CONT_ID);
										maxContID.updateAndGet(old -> Math.max(old, contId));

										if (directory instanceof ActivePersistenceDirectory activePersistenceDirectory) {
											long aTime = activePersistenceDirectory.maxATime(containerDirectory);
											maxArchiveTime.updateAndGet(old -> Math.max(old, aTime));
										}
									}
								} catch (Exception e) {
									badDID.getAndIncrement();
									_debug.error(taskName + ": Index der Datenidentifikation '" + didDir + "' konnte nicht neu angelegt werden", e);
								}
							}
						}
					}
			);
		}

		private boolean isNeedsRebuild(PersistenceDirectory directory, LockedContainerDirectory containerDirectory, Path changedFlagFile) {
			// Strategie: Wenn die _rebuildIndex.flag-Datei vorhanden ist, werden die Indexe neu aufgebaut.
			if (Files.exists(changedFlagFile)) {
				return true;
			}

			// Wenn die Indexdateien fehlen, werden die Indexe nur
			// neu aufgebaut, wenn sie beim Aufbau nicht eh schon gelöscht werden würden
			if (rebuildMode.isDelete()) {
				return false;
			}

			Path path = directory.getPath(containerDirectory);
			if (!Files.exists(path.resolve(ContainerManagementIndex.IDX_FILENAME))) {
				return true;
			}
			if (!Files.exists(path.resolve(ArchiveTimeIndexImpl.IDX_FILENAME))) {
				return true;
			}
			if (!Files.exists(path.resolve(DataTimeIndexImpl.IDX_FILENAME))) {
				return true;
			}
			return containerDirectory.archiveDataKind().isDelayed() && !Files.exists(path.resolve(DataIndexIndexImpl.IDX_FILENAME));
		}

		@NotNull
		private String getProgressMessage(final long goodDirs, final long badDirs, final long sum) {
			final String result;
			if (sum != 0) {
				StringBuilder sb = new StringBuilder().append("bei ")
						.append(sum)
						.append(" Index-Neuaufbau notwendig, davon ")
						.append(goodDirs)
						.append(" erfolgreich und ")
						.append(badDirs)
						.append(" nicht erfolgreich.")
						.append("\n");
				if (getContainerRead() != 0) {
					sb.append(String.format("%8d Container erfolgreich gelesen.\n", getContainerRead()));
				}
				if (getContainerIgnored() != 0) {
					sb.append(String.format("%8d defekte Container ignoriert.\n", getContainerIgnored()));
				}
				if (getIndexesDeleted() != 0) {
					sb.append(String.format("%8d inkonsistente Indexdateien gelöscht.\n", getIndexesDeleted()));
				}
				result = sb.toString();
			} else {
				result = "kein Index-Neuaufbau notwendig.";
			}
			return result;
		}
	}
}
