/*
 *
 * Copyright 2017-2019 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence;

import java.util.EnumMap;

/**
 * Einfache Implementierung von ContainerManagementData, die die Daten von einem anderen ContainerManagementData kopiert.
 * <p>
 * Sinnvoll, falls die andere Implementierung eine Containerdatei o.ä. ist, die schnell wieder geschlossen werden soll und
 * die Infos aber längere Zeit im Speicher gehalten werden sollen.
 * <p>
 * Diese Klasse ist immutable.
 *
 * @author Kappich Systemberatung
 */
public class SimpleContainerManagementData implements ContainerManagementData {

	/**
	 * Enthält Long und String-Objekte
	 */
	private final EnumMap<ContainerManagementInformation, Object> _values = new EnumMap<>(ContainerManagementInformation.class);

	/**
	 * Datenidentifikation
	 */
	private final ContainerDirectory _location;

	/**
	 * Erstellt eine neue SimpleContainerManagementData basierend auf einem anderen Objekt
	 * @param base Objekt, das kopiert werden soll
	 * @throws PersistenceException Fehler bei der Anfrage vom anderen Objekt
	 */
	public SimpleContainerManagementData(ContainerManagementData base) throws PersistenceException {
		for(ContainerManagementInformation header : ContainerManagementInformation.values()) {
			if(header.isNumeric()) {
				_values.put(header, base.getContainerHeaderParamAsLong(header));
			}
			else {
				_values.put(header, base.getContainerHeaderParamAsString(header));
			}
		}
		_location = base.getLocation();
	}

	@Override
	public String getContainerHeaderParamAsString(final ContainerManagementInformation param) {
		return (String) _values.get(param);
	}

	@Override
	public boolean getContainerHeaderParamAsBoolean(final ContainerManagementInformation param) {
		return getContainerHeaderParamAsString(param).equals("1");
	}

	@Override
	public long getContainerHeaderParamAsLong(final ContainerManagementInformation param) {
		return (Long) _values.get(param);
	}

	@Override
	public int getContainerHeaderParamAsInt(final ContainerManagementInformation param) {
		return (int) getContainerHeaderParamAsLong(param);
	}

	@Override
	public ContainerDirectory getLocation() {
		return _location;
	}
}
