/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence;

import de.bsvrz.ars.ars.persistence.iter.DataSequence;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;

import java.nio.file.Path;

/**
 * Referenz auf eine unabhängige Containerdatei außerhalb des Persistenzverzeichnisses
 *
 * @author Kappich Systemberatung
 */
public class StandaloneContainerFileHandle implements AutoCloseable, ContainerManagementData, DataSequence {

	/**
	 * Interner Konstruktor, der von abgeleiteten Klassen verwendet wird.
	 */
	protected StandaloneContainerFileHandle() {
	}

	/**
	 * Haupt-Konstruktor
	 *
	 * @param containerFile Zu öffnende Containerdatei
	 * @throws PersistenceException Fehler beim Lesen (Datei korrupt?)
	 */
	public StandaloneContainerFileHandle(Path containerFile) throws PersistenceException {
		_containerFile.accessContainerReadOnly(containerFile);
	}

	/**
	 * Interne Referenz auf Containerdatei
	 */
	final ContainerFile _containerFile = new ContainerFile();

	/**
	 * Gibt containerFile von diesem BasicContainerFileHandle-Objekt zurück.
	 *
	 * @return containerFile (Typ ContainerFile)
	 */
	public final ContainerFile getContainerFile() {
		return _containerFile;
	}

	/**
	 * Beendet das Arbeiten mit dem zuvor über eine der {@code accessContainer()}-Methoden definierten Container.
	 */
	public void leaveContainer() {
		getContainerFile().leaveContainer();
	}

	/**
	 * Nachdem mit {@link #readContainerHeader()} der Header einer Container-Datei eingelesen worden ist, kann mit dieser Methode der Wert eines
	 * Header-Parameters ermittelt werden. Wirft eine {@link PersistenceException}, falls der Parameter nicht gefunden wurde. Liefert den Wert des
	 * Parameters unabhaengig von dessen Typ immer als String zurück.
	 *
	 * @param param Container-Header-Parameter
	 * @return Parameter-Wert als String
	 * @see KeyValParam
	 * @throws PersistenceException Lesefehler
	 */
	public String getContainerHeaderParamAsString(KeyValParam param) throws PersistenceException {
		return getContainerFile().getContHdrParamAsString(param);
	}

	/**
	 * Nachdem mit {@link #readContainerHeader()} der Header einer Container-Datei eingelesen worden ist, kann mit dieser Methode der Wert eines
	 * Header-Parameters ermittelt werden. Liefert den Wert des Parameters als Datensatzart zurück. Wirft eine {@link PersistenceException}, falls der
	 * Parameter nicht gefunden wurde oder nicht in den Typ {@link ArchiveDataKind} umgeformt werden kann.
	 *
	 * @param param Container-Header-Parameter
	 * @return Datensatzart
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	public ArchiveDataKind getContainerHeaderParamAsArchiveDataKind(KeyValParam param) throws PersistenceException {
		return getContainerFile().getContHdrParamAsArchiveDataKind(param);
	}

	/**
	 * Nachdem mit {@link #readContainerHeader()} der Header einer Container-Datei eingelesen worden ist, kann mit dieser Methode der Wert eines
	 * Header-Parameters ermittelt werden. Liefert den Wert des Parameters als boolschen Wert zurück. Wirft eine {@link PersistenceException}, falls
	 * der Parameter nicht gefunden wurde oder nicht in den Typ {@code boolean} umgeformt werden kann.
	 *
	 * @param param Container-Header-Parameter
	 * @return {@code true} oder {@code false}
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	public boolean getContainerHeaderParamAsBoolean(KeyValParam param) throws PersistenceException {
		return getContainerFile().getContHdrParamAsBool(param);
	}

	/**
	 * Nachdem mit {@link #readContainerHeader()} der Header einer Container-Datei eingelesen worden ist, kann mit dieser Methode der Wert eines
	 * Header-Parameters ermittelt werden. Liefert den Wert des Parameters als numerischen Wert vom Typ {@code long} zurück. Wirft eine {@link
	 * PersistenceException}, falls der Parameter nicht gefunden wurde oder nicht in den Typ {@code long} umgeformt werden kann.
	 *
	 * @param param Container-Header-Parameter
	 * @return Ganzzahl als {@code long}
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	public long getContainerHeaderParamAsLong(KeyValParam param) throws PersistenceException {
		return getContainerFile().getContHdrParamAsLong(param);
	}

	/**
	 * Nachdem mit {@link #readContainerHeader()} der Header einer Container-Datei eingelesen worden ist, kann mit dieser Methode der Wert eines
	 * Header-Parameters ermittelt werden. Liefert den Wert des Parameters als numerischen Wert vom Typ {@code long} zurück. Wirft eine {@link
	 * PersistenceException}, falls der Parameter nicht gefunden wurde oder nicht in den Typ {@code long} umgeformt werden kann.
	 *
	 * @param param Container-Header-Parameter
	 * @return Ganzzahl als {@code long}
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	public int getContainerHeaderParamAsInt(KeyValParam param) throws PersistenceException {
		return getContainerFile().getContHdrParamAsInt(param);
	}

	/**
	 * Nachdem mit {@link #readContainerHeader()} der Header einer Container-Datei eingelesen worden ist, kann mit dieser Methode der Wert eines
	 * Header-Parameters ermittelt werden. Wirft eine {@link PersistenceException}, falls der Parameter nicht gefunden wurde. Liefert den Wert des
	 * Parameters unabhaengig von dessen Typ immer als String zurück.
	 *
	 * @param param Container-Header-Parameter
	 * @return Parameter-Wert als String
	 * @see KeyValParam
	 */
	@Override
	public String getContainerHeaderParamAsString(ContainerManagementInformation param) throws PersistenceException {
		return getContainerHeaderParamAsString(param.delegate());
	}

	/**
	 * Nachdem mit {@link #readContainerHeader()} der Header einer Container-Datei eingelesen worden ist, kann mit dieser Methode der Wert eines
	 * Header-Parameters ermittelt werden. Liefert den Wert des Parameters als boolschen Wert zurück. Wirft eine {@link PersistenceException}, falls
	 * der Parameter nicht gefunden wurde oder nicht in den Typ {@code boolean} umgeformt werden kann.
	 *
	 * @param param Container-Header-Parameter
	 * @return {@code true} oder {@code false}
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	@Override
	public boolean getContainerHeaderParamAsBoolean(ContainerManagementInformation param) throws PersistenceException {
		return getContainerHeaderParamAsBoolean(param.delegate());
	}

	/**
	 * Nachdem mit {@link #readContainerHeader()} der Header einer Container-Datei eingelesen worden ist, kann mit dieser Methode der Wert eines
	 * Header-Parameters ermittelt werden. Liefert den Wert des Parameters als numerischen Wert vom Typ {@code long} zurück. Wirft eine {@link
	 * PersistenceException}, falls der Parameter nicht gefunden wurde oder nicht in den Typ {@code long} umgeformt werden kann.
	 *
	 * @param param Container-Header-Parameter
	 * @return Ganzzahl als {@code long}
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	@Override
	public long getContainerHeaderParamAsLong(ContainerManagementInformation param) throws PersistenceException {
		return getContainerHeaderParamAsLong(param.delegate());
	}

	/**
	 * Nachdem mit {@link #readContainerHeader()} der Header einer Container-Datei eingelesen worden ist, kann mit dieser Methode der Wert eines
	 * Header-Parameters ermittelt werden. Liefert den Wert des Parameters als numerischen Wert vom Typ {@code int} zurück. Wirft eine {@link
	 * PersistenceException}, falls der Parameter nicht gefunden wurde oder nicht in den Typ {@code int} umgeformt werden kann.
	 *
	 * @param param Container-Header-Parameter
	 * @return Ganzzahl als {@code int}
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	@Override
	public int getContainerHeaderParamAsInt(ContainerManagementInformation param) throws PersistenceException {
		return getContainerHeaderParamAsInt(param.delegate());
	}

	@Override
	public ContainerDirectory getLocation() throws PersistenceException {
		return getLocationFromHeader();
	}

	/**
	 * Gibt den Ort des Containers zurück
	 *
	 * @return den Ort des Containers
	 */
	public IdContainerFileDir getLocationFromHeader() throws PersistenceException {
		ensureHeaderRead();
		return new IdContainerFileDir(getContainerHeaderParamAsLong(ContainerHdr.CHP_OBJ_ID), getContainerHeaderParamAsLong(ContainerHdr.CHP_ATG_ID),
				getContainerHeaderParamAsLong(ContainerHdr.CHP_ASP_ID), getContainerHeaderParamAsInt(ContainerHdr.CHP_SIM_VAR),
				getContainerHeaderParamAsArchiveDataKind(ContainerHdr.CHP_DATA_KIND));
	}

	/**
	 * Die Methode beendet den Zugriff auf den Container (der Container wird nicht abgeschlossen), wird benötigt für das {@link
	 * AutoCloseable}-interface.
	 * <p>
	 * Die Methode ruft nur {@link #leaveContainer()} auf.
	 */
	@Override
	public void close() {
		leaveContainer();
	}

	/**
	 * Liest den Header der Container-Datei ein und springt im Eingabestrom an die Stelle hinter dem Header. Die Container-Header Parameter stehen
	 * dann als Key/Value-Paare zur Verfuegung und können mit {@link #getContainerHeaderParamAsString(KeyValParam)} abgefragt werden.
	 *
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	public void readContainerHeader() throws PersistenceException {
		getContainerFile().readContainerHeader();
	}

	/**
	 * Erzeugt einen Iterator über Containerdaten. Falls der Container nach der alten Methode gelöscht ist (und nur aus Header und Datensatzbereich
	 * besteht) werden nur die Datenindexe zurückgegeben.
	 *
	 * <p>Dieses Verhalten wird nur für die Migration zu einem {@link DeletedContainerFile}
	 * verwendet. In der Regel ist ein BasicContainerFileHandle nicht gelöscht.
	 *
	 * @return Iterator
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	@Override
	@NotNull
	public ContainerDataIterator iterator() throws PersistenceException {
		ensureHeaderRead(); // Flusht den Cache, wichtig damit alle Containerdaten in der Datei sind, wenn gleich gelesen wird!
		return new ContainerDataIterator(this);
	}

	/**
	 * Gibt die Container-ID zurück
	 *
	 * @return Container-ID
	 */
	public long getContainerId() {
		return _containerFile.getContainerId();
	}

	/**
	 * Prüft ob der Container bereits abgeschlossen ist.
	 *
	 * @return Kennzeichen, ob Container abgeschlossen ist
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	public boolean isContainerClosed() throws PersistenceException {
		return isClosedInternal();
	}

	/**
	 * Prüft, ob der Container geschlossen ist, ohne den Zugriff auf den {@link de.bsvrz.ars.ars.mgmt.datatree.DataIdentNode} zu prüfen.
	 *
	 * @return Kennzeichen, ob Container abgeschlossen ist
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	public boolean isClosedInternal() throws PersistenceException {
		return getContainerFile().isContainerClosed();
	}

	/**
	 * Gibt die Datei zurück
	 *
	 * @return die Datei
	 */
	public Path getFile() {
		return _containerFile.getContFile();
	}

	@Override
	public String toString() {
		return getFile().toString();
	}

	/**
	 * Stellt sicher, dass der Header eingelesen wurde. Liest ihn ein, falls noch nicht geschehen.
	 *
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	public void ensureHeaderRead() throws PersistenceException {
		_containerFile.ensureHeaderRead();
	}
}
