/*
 * Copyright 2019-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence;

/**
 * Implementierung von {@link OpenContainerData}. Die Minimalwerte sind final, die Maximalwerte werden beim Schreiben eines Datensatzes geupdated.
 */
public final class StandardOpenContainerData implements OpenContainerData {
	private long _containerEntries;
	private final long _minArcTime;
	private long _maxArcTime;
	private final long _minDataTime;
	private long _maxDataTime;
	private final long _minDataIdx;
	private long _maxDataIdx;
	private final long _containerId;

	/**
	 * @param containerEntries Anzahl Datensätze im Container
	 * @param minArcTime       Minimaler Archivzeitstempel im Container
	 * @param maxArcTime       Maximaler Archivzeitstempel im Container
	 * @param minDataTime      Minimaler Datenzeitstempel im Container
	 * @param maxDataTime      Maximaler Datenzeitstempel im Container
	 * @param minDataIdx       Minimaler Datenindex im Container
	 * @param maxDataIdx       Maximaler Datenindex im Container
	 * @param containerId      Container-ID des offenen Containers
	 */
	public StandardOpenContainerData(final long containerEntries, final long minArcTime, final long maxArcTime, final long minDataTime, final long maxDataTime, final long minDataIdx, final long maxDataIdx, final long containerId) {
		_containerEntries = containerEntries;
		_minArcTime = minArcTime;
		_maxArcTime = maxArcTime;
		_minDataTime = minDataTime;
		_maxDataTime = maxDataTime;
		_minDataIdx = minDataIdx;
		_maxDataIdx = maxDataIdx;
		_containerId = containerId;
	}

	/**
	 * Gibt die Anzahl Datensätze im Container zurück. Dieser Wert ist immer &gt; 0
	 *
	 * @return die Anzahl Datensätze im Container
	 */
	public long getNumContainerEntries() {
		return _containerEntries;
	}

	/**
	 * Gibt die minimale Archivzeit zurück
	 * @return die minimale Archivzeit
	 */
	public long getMinArcTime() {
		return _minArcTime;
	}

	/**
	 * Gibt die maximale Archivzeit zurück
	 * @return die maximale Archivzeit
	 */
	public long getMaxArcTime() {
		return _maxArcTime;
	}

	/**
	 * Gibt die minimale Datenzeit zurück
	 * @return die minimale Datenzeit
	 */
	public long getMinDataTime() {
		return _minDataTime;
	}

	/**
	 * Gibt die maximale Datenzeit zurück
	 * @return die maximale Datenzeit
	 */
	public long getMaxDataTime() {
		return _maxDataTime;
	}

	/**
	 * Gibt den minimalen Datenindex zurück
	 * @return den minimalen Datenindex
	 */
	public long getMinDataIdx() {
		return _minDataIdx;
	}

	/**
	 * Gibt den maximalen Datenindex zurück
	 * @return den maximalen Datenindex
	 */
	public long getMaxDataIdx() {
		return _maxDataIdx;
	}

	/**
	 * Gibt die ID des offenen Containers zurück
	 * @return die ID des offenen Containers
	 */
	public long getContainerId() {
		return _containerId;
	}

	/**
	 * Aktualisiert die Maximalwerte beim Schreiben eines neuen Datensatzes
	 * @param dataIdx  Datenindex des Datensatzes
	 * @param arsTime  Archivzeit des Datensatzes
	 * @param dataTime Datenzeit des Datensatzes
	 */
	public void updateMax(final long dataIdx, final long arsTime, final long dataTime) {
		_maxArcTime = Math.max(arsTime, _maxArcTime);
		_maxDataIdx = Math.max(dataIdx, _maxDataIdx);
		_maxDataTime = Math.max(dataTime, _maxDataTime);
		_containerEntries++;
	}

	@Override
	public String toString() {
		return "StandardOpenContainerData{" +
				"_containerEntries=" + _containerEntries +
				", _minArcTime=" + _minArcTime +
				", _maxArcTime=" + _maxArcTime +
				", _minDataTime=" + _minDataTime +
				", _maxDataTime=" + _maxDataTime +
				", _minDataIdx=" + _minDataIdx +
				", _maxDataIdx=" + _maxDataIdx +
				", _containerId=" + _containerId +
				'}';
	}
}
