/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.persistence;

import java.io.IOException;
import java.io.InputStream;
import java.io.Writer;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import static java.nio.charset.StandardCharsets.ISO_8859_1;


/**
 * Klasse zum Lesen und Schreiben der Startup-Info, die das Archivsystem braucht, um starten zu können. Wenn die Startup-Info nicht vorliegt, muss sie neu
 * erzeugt werden (aufwendig).
 *
 * @author beck et al. projects GmbH
 * @author Alexander Schmidt
 * @version $Revision$ / $Date$ / ($Author$)
 */
public class StartupProperties extends KeyValProps {

	public static final String STARTUP_PROP_FILE_NAME = "_startUpInfo.property";

	/** Sortierte Liste aller StartupInfo-Parameter. */
	protected static final List<KeyValParam> allParams = new ArrayList<>();

	private final Path startUpPropFile;

	private static final long INVALID_CONTAINER_ID = -1;

	private static final long INVALID_TIME = -1;

	public static final KeyValParam STUP_MAX_CONT_ID = KeyValParam.createNumParam("maxContainerID", INVALID_CONTAINER_ID, BYTES5_STR_LEN, INVALID_CONTAINER_ID, BYTE5_MAXVAL, 5);
	public static final KeyValParam STUP_LAST_ATIME = KeyValParam.createNumParam("lastArchiveTime", INVALID_TIME, BYTES6_STR_LEN, INVALID_TIME, BYTE6_MAXVAL, 6);

	/** Länge des Key/Value-Blocks in Byte. */
	public static final int HDR_TXT_LEN;

	static {
		allParams.add(STUP_MAX_CONT_ID);
		allParams.add(STUP_LAST_ATIME);

		int headerLen = 0;
		for(KeyValParam param : allParams) {
			headerLen += param.getKey().length() + "=".length() + param.getValLen() + CH_DELIM.length();
		}
		HDR_TXT_LEN = headerLen;
	}


	/**
	 * Initialisiert die StartUp-Properties.
	 *
	 * @param persBaseDir Wurzel des Persistenzverzeichnisses
	 */
	public StartupProperties(Path persBaseDir) {
		super();
		startUpPropFile = persBaseDir.resolve(STARTUP_PROP_FILE_NAME);
	}

	/**
	 * Liest die StartUp-Properties ein
	 *
	 * @throws PersistenceException falls ein Fehler aufgetreten ist. In diesem Fall muss der Restore-Vorgang gestartet werden.
	 */
	public void readStartUpProperties() throws PersistenceException {
		properties.clear();
		try(InputStream fis = Files.newInputStream(startUpPropFile)) {
			Properties tmpProp = new Properties();	// zur Prüfung erst in extra Properties laden
			tmpProp.load(fis);
			for(KeyValParam param : allParams) {
				setVal(param, (String)tmpProp.get(param.getKey()));
			}
		}
		catch(IOException e) {
			throw new PersistenceException("Startup-Properties konnten nicht gelesen werden", e);
		}
	}

	public void writeStartUpProperties() throws PersistenceException {
		try {
			try (Writer fw = Files.newBufferedWriter(startUpPropFile, ISO_8859_1)) {
				StringBuilder sb = new StringBuilder(HDR_TXT_LEN);
				for (KeyValParam param : allParams) {
					sb.append(param.getKey()).append("=").append(properties.getProperty(param.getKey())).append(CH_DELIM);
				}
				fw.write(sb.toString());
			}
		}
		catch(IOException e) {
			throw new PersistenceException("Startup-Properties konnten nicht geschrieben werden", e);
		}
	}

	public void deleteStartupPropsFile() throws IOException {
		Files.deleteIfExists(startUpPropFile);
	}

	public String toString() {
		return "Startup Properties Datei '" + startUpPropFile.toAbsolutePath() + "'";
	}
}
