/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence.directories;

import com.google.common.collect.ImmutableMap;
import de.bsvrz.ars.ars.mgmt.datatree.IndexTree;
import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SyncKey;
import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SynchronizationFailedException;
import de.bsvrz.ars.ars.persistence.*;
import de.bsvrz.ars.ars.persistence.directories.cache.DataRange;
import de.bsvrz.ars.ars.persistence.directories.cache.NoData;
import de.bsvrz.ars.ars.persistence.directories.cache.SimpleDataRange;
import de.bsvrz.ars.ars.persistence.directories.cache.ValidDataRange;
import de.bsvrz.ars.ars.persistence.directories.mgmt.lock.LockFileManager;
import de.bsvrz.ars.ars.persistence.index.*;
import de.bsvrz.ars.ars.persistence.index.backend.management.BaseIndex;
import de.bsvrz.ars.ars.persistence.index.backend.management.LongRange;
import de.bsvrz.ars.ars.persistence.index.result.AggregateIndexResult;
import de.bsvrz.ars.ars.persistence.index.result.IndexResult;
import de.bsvrz.ars.ars.persistence.iter.DataIterator;
import de.bsvrz.ars.ars.persistence.layout.PersistenceDirectoryLayoutInstance;
import de.bsvrz.ars.ars.persistence.sequence.AllDataSpecification;
import de.bsvrz.ars.ars.persistence.sequence.ArchiveTimeSequenceSpecification;
import de.bsvrz.ars.ars.persistence.sequence.DataIndexSequenceSpecification;
import de.bsvrz.ars.ars.persistence.sequence.SequenceSpecification;
import de.bsvrz.ars.ars.persistence.walk.internal.CountEstimator;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import org.jetbrains.annotations.Contract;

import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.time.Instant;
import java.util.*;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.atomic.AtomicLong;

/**
 * Einzelnes Persistenzverzeichnis, das für einen bestimmten Zeitbereich oder für eine
 * bestimmte Simulationsvariante die Daten enthält.
 */
public abstract sealed class PersistenceDirectory implements CountEstimator permits ActivePersistenceDirectory, ReadonlyPersistenceDirectory {

	protected static final Debug debug = Debug.getLogger();

	/**
	 * Baum zur Verwaltung der Indexe
	 */
	protected final IndexTree indexTree;

	/**
	 * Menge von Datenidentifikationen, in denen sehr viele Containerdateien enthalten sind (für Warnmeldung)
	 */
	protected final ConcurrentSkipListSet<IdContainerFileDir> largeDataIdentifications = new ConcurrentSkipListSet<>();

	/**
	 * Klasse, die den Zugriff auf Datenidentifikationen verwaltet, im Normalbetrieb typischerweise der
	 * {@link PersistenceManager}, für Tests oder die Migration ggf. auch eine andere Implementierung.
	 */
	@NotNull
	private final DataIdentificationManager dataIdentificationManager;

	/**
	 * Layout (Struktur) dieses Persistenzverzeichnisses
	 */
	private final PersistenceDirectoryLayoutInstance layoutInstance;

	/**
	 * Geschätzte Anzahl Datenidentifikationen
	 */
	private final AtomicLong lastCount = new AtomicLong();

	/**
	 * Erstellt ein neues PersistenceDirectory.
	 *
	 * @param dataIdentificationManager Klasse zum Zugriff auf Indexe (Im Normalbetrieb der {@link PersistenceManager})
	 * @param layoutInstance            Instanz eines Verzeichnis-Layouts
	 */
	public PersistenceDirectory(@NotNull DataIdentificationManager dataIdentificationManager, @NotNull PersistenceDirectoryLayoutInstance layoutInstance) {
		this.dataIdentificationManager = dataIdentificationManager;
		var basePath = layoutInstance.basePath();
		boolean exists = Files.isDirectory(basePath);
		if (!exists) {
			debug.info("Persistenzverzeichnis '" + basePath + "' existiert nicht, wird neu angelegt.");
			try {
				Files.createDirectories(basePath);
			} catch (IOException e) {
				debug.warning("Kann Verzeichnis nicht anlegen: " + basePath, e);
			}
		}
		indexTree = new IndexTree(dataIdentificationManager.getIndexCacheMaxSize(), this);
		this.layoutInstance = layoutInstance;
	}

	/**
	 * Die Methode getHeaders gibt die Header eines Containers zurück.
	 *
	 * @param contHdrIdx  von Typ ContainerManagementIndex
	 * @param containerID von Typ long
	 * @param location    Ort des Containers
	 * @return ContainerHeaders
	 * @throws IndexException wenn ein Problem beim Zugriff auf den Index auftritt
	 */
	private static ContainerHeaders getHeaders(final ContainerManagementIndex contHdrIdx,
	                                           final long containerID,
	                                           ContainerDirectory location) throws IndexException {
		var query = Map.of(ContainerManagementInformation.CHP_CONT_ID, new LongRange(containerID, containerID));
		var result = contHdrIdx.query(query);
		// Die Anfrage könnte einen vorherigen Container zurückgeliefert haben
		if (!result.isEmpty() && result.get(0, ContainerManagementInformation.CHP_CONT_ID) == containerID) {
			return new ContainerHeaders(result, 0, location);
		}
		throw new IllegalArgumentException("Container " + containerID + " ist nicht im Index enthalten.");
	}

	/**
	 * Die Methode getLastContainerHeaders ermittelt die Header vom letzten Container (vom Container mit der größten ID)
	 *
	 * @param containerDirectory Referenz auf die gelockte Datenidentifikation und Datenart für den Zugriff auf Containerdaten
	 *                           Container-Verzeichnis
	 * @return ContainerHeaders
	 * @throws IndexException wenn der Indexzugriff fehlschlägt
	 */
	@Nullable
	public ContainerHeaders getLastContainerHeaders(final LockedContainerDirectory containerDirectory) throws IndexException {
		Optional<? extends ContainerManagementIndex> index = getIndexTree().getContainerManagementIndex(containerDirectory);
		if (index.isEmpty() || index.get().numEntries() == 0) return null;
		long contId = index.get().getLast(ContainerManagementInformation.CHP_CONT_ID);
		return getContainerHeaders(containerDirectory, contId);
	}

	/**
	 * Die Methode getMinimumDataIndex bestimmt den minimalen Datenindex aus einer Containerdatei
	 *
	 * @param containerFileHandle von Typ BasicContainerFileHandle
	 * @return long Datenindex
	 * @throws PersistenceException wenn ein Fehler beim Dateizugriff auftritt
	 */
	private static long getMinimumDataIndex(final BasicContainerFileHandle containerFileHandle)
			throws PersistenceException {
		int numDs = containerFileHandle.getContainerHeaderParamAsInt(ContainerManagementInformation.CHP_ANZ_DS);
		if (numDs == ContainerHdr.CONT_UNCLOSED) {
			// Im Container nachsehen
			try (DataIterator iterator = containerFileHandle.iterator()) {
				return iterator.peekDataIndex();
			}
		}
		return containerFileHandle.getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_DATA_IDX_MIN);
	}

	/**
	 * Führt eine Aktion auf einem Container auf und benennt den Container um, falls die Aktion eine {@link PersistenceException} geworfen hat und der
	 * Benutzer {@link PersistenceManager#shouldDeleteBrokenContainers} gesetzt hat.
	 *
	 * @param contFile Containerdatei
	 * @param action   Aktion
	 * @throws PersistenceException Beliebiger Grund
	 */
	protected void executeOnContainer(final Path contFile, ContainerCall action) throws PersistenceException, SynchronizationFailedException {
		try {
			action.call();
		} catch (PersistenceException e) {
			if (dataIdentificationManager.shouldDeleteBrokenContainers()) {
				renameContainerFile(contFile);
			} else {
				debug.error("Container " + contFile +
						" konnte nicht gelesen werden. Aufrufargument \"-defekteDateienVerschieben\" setzen, um defekte Dateien automatisch zu" +
						" entfernen.");
			}
			throw new PersistenceException(e);
		}
	}

	protected void renameContainerFile(final Path contFile) throws PersistenceException {
		Path target = contFile.resolveSibling(contFile.getFileName() + ".defect");
		try {
			Files.move(contFile, target);
			debug.fine("Container " + contFile + " konnte nicht gelesen werden und wurde in " + target.getFileName() + " umbenannt.");
		} catch (IOException e1) {
			throw new PersistenceException("Umbenennen der defekten Containerdatei fehlgeschlagen: " + contFile, e1);
		}
	}

	/**
	 * Gibt die Header eines Containers zurück. Die Daten werden aus dem Verwaltungsdatenindex gelesen.
	 *
	 * @param containerDirectory Referenz auf die gelockte Datenidentifikation und Datenart für den Zugriff auf Containerdaten
	 *                           Container-Verzeichnis
	 * @param containerID        Container-ID, dessen Header gelesen werden sollen
	 * @return die Header eines Containers
	 * @throws IndexException Daten konnten nicht ermittelt werden (z. B. Container nicht vorhanden)
	 */
	public ContainerHeaders getContainerHeaders(final LockedContainerDirectory containerDirectory, final long containerID)
			throws IndexException {
		Optional<? extends ContainerManagementIndex> cmi = getIndexTree().getContainerManagementIndex(containerDirectory);
		if (cmi.isEmpty()) {
			throw new IllegalArgumentException("Container " + containerID + " ist nicht im Index enthalten.");
		}
		return getHeaders(cmi.get(), containerID, containerDirectory);
	}

	/**
	 * Gibt die Header von allen Containern eines Verzeichnisses zurück.
	 * Im Prinzip wird der gesamte Inhalt des Verwaltungsdatenindex zurückgegeben.
	 *
	 * @param containerDirectory Referenz auf die gelockte Datenidentifikation und Datenart für den Zugriff auf Containerdaten
	 *                           Container-Verzeichnis
	 * @return die Header als {@link IndexResult}
	 * @throws IndexException Daten konnten nicht ermittelt werden
	 */
	public IndexResult<ContainerManagementInformation> getContainerHeaders(final LockedContainerDirectory containerDirectory)
			throws IndexException {
		Optional<? extends ContainerManagementIndex> containerManagementIndex = getIndexTree().getContainerManagementIndex(containerDirectory);
		if (containerManagementIndex.isEmpty()) {
			// Keine Daten vorhanden
			return new AggregateIndexResult<>();
		}
		return containerManagementIndex.get().query();
	}

	/**
	 * Greift auf einen Container zu
	 *
	 * @param containerDirectory Referenz auf die gelockte Datenidentifikation und Datenart für den Zugriff auf Containerdaten
	 *                           Containerverzeichnis
	 * @param containerId        Id des Containers
	 * @return Container-Handle
	 * @throws PersistenceException Lesefehler
	 */
	@NotNull
	public ContainerFileHandle accessContainer(final LockedContainerDirectory containerDirectory, final long containerId) throws PersistenceException {
		assert containerDirectory.lock().isValid();
		return new ContainerFileHandle(containerDirectory, containerId, false, layoutInstance);
	}

	/**
	 * Gibt das Verzeichnis auf dem Datenträger zurück.
	 *
	 * @return das Verzeichnis auf dem Datenträger
	 */
	public final Path getBasePath() {
		return layoutInstance.basePath();
	}

	/**
	 * Liefert ein File für das angegebene Verzeichnis. Wenn das Verzeichnis nicht existiert, wird es angelegt (lazy-Verhalten).
	 *
	 * @param path Verzeichnis
	 * @return Archiv-Pfad als File
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	public static Path createDirectoriesIfAbsent(Path path) throws PersistenceException {

		if (Files.isDirectory(path)) {
			// Verzeichnis existiert schon, also einfach zurückgeben.

			// Dieser vorzeitige Abbruch ist eigentlich nicht nötig, führt aber zu einer deutlichen Performance-Verbesserung.
			// Files.createDirectories ist verhältnismäßig langsam, auch wenn das Verzeichnis schon existiert.
			return path;
		}

		try {
			Files.createDirectories(path);
		} catch (IOException e) {
			throw new PersistenceException("Persistenzverzeichnis '" + path.toAbsolutePath() + "' konnte nicht angelegt werden.", e);
		}
		return path;
	}

	/**
	 * Die Methode rebuildStandardIndex erstellt einen Standard-Index aus dem Verwaltungsdatenindex
	 *
	 * @param managementIndex    Verwaltungsdatenindex
	 * @param index              Zu erzeugender Index (leer)
	 * @param containerDirectory Referenz auf die gelockte Datenidentifikation und Datenart für den Zugriff auf Containerdaten
	 * @throws IndexException Problem beim Index-Zugriff
	 */
	public final void rebuildStandardIndex(final ContainerManagementIndex managementIndex,
	                                       final BaseIndex<IndexValues> index, LockedContainerDirectory containerDirectory) throws IndexException {
		debug.fine("Baue Index neu auf", index.toString());
		IndexResult<ContainerManagementInformation> result = managementIndex.query();
		for (int i = 0; i < result.size(); i++) {
			int num = result.getSignedInt(i, ContainerManagementInformation.CHP_ANZ_DS);
			long contId = result.get(i, ContainerManagementInformation.CHP_CONT_ID);

			index.setInsertValue(IndexValues.ContainerId, contId);

			if (num == ContainerHdr.CONT_UNCLOSED) {
				// Spezialbehandlung, je nachdem ob es sich um ein offenes oder geschlossenes Persistenzverzeichnis handelt.
				handleUnclosedContainer(managementIndex, containerDirectory, contId, index);
			} else {
				index.setInsertValue(IndexValues.DataIndexMin, result.get(i, ContainerManagementInformation.CHP_DATA_IDX_MIN));
				index.setInsertValue(IndexValues.DataIndexMax, result.get(i, ContainerManagementInformation.CHP_DATA_IDX_MAX));
				index.setInsertValue(IndexValues.DataTimeMin, result.get(i, ContainerManagementInformation.CHP_DATA_TIME_MIN));
				index.setInsertValue(IndexValues.DataTimeMax, result.get(i, ContainerManagementInformation.CHP_DATA_TIME_MAX));
				index.setInsertValue(IndexValues.ArchiveTimeMin, result.get(i, ContainerManagementInformation.CHP_ARC_TIME_MIN));
				index.setInsertValue(IndexValues.ArchiveTimeMax, result.get(i, ContainerManagementInformation.CHP_ARC_TIME_MAX));
			}
			index.insert();
		}
	}

	protected void handleUnclosedContainer(ContainerManagementIndex managementIndex, LockedContainerDirectory containerDirectory, long contId, BaseIndex<IndexValues> index) throws IndexException {
		// Fehlermeldung erzeugen, offene Container sind in abgeschlossenen Verzeichnissen unzulässig.
		// Verhalten wird in ActivePersistenceDirectory überschrieben.
		throw new IndexException("Offener Container in abgeschlossenem Archivdatenverzeichnis", managementIndex.getFile());
	}

	/**
	 * Erzeugt einen neuen ContainerHeaderIndex basierend aus den Containerdateien in einem Verzeichnis
	 *
	 * @param containerFileDir     ID des Containerverzeichnisses
	 * @param containerHeaderIndex Container-Header-Index (leer)
	 * @param rebuildResult        Statistik über die Wiederherstellung
	 * @throws IndexException Lesefehler der Indexe (z. B. korrupt)
	 */
	public void rebuildContainerHeaderIndex(final IdContainerFileDir containerFileDir, final ContainerManagementIndex containerHeaderIndex,
	                                        @Nullable final RebuildResult rebuildResult) throws IndexException {

		debug.fine("Baue Verwaltungsdatenindex neu auf", containerHeaderIndex);

		Path dir = getPath(containerFileDir);
		if (!Files.exists(dir)) {
			return;
		}

		// Zeit messen, wie lange Wiederherstellung dauert, um Benutzer bei großen Verzeichnissen zu warnen.
		Duration nextWarnTimeLimit = Duration.ofMinutes(1);
		Instant startTime = Instant.now();

		try {
			final Map<Long, Long> containerIndexToMinDataIndex = new HashMap<>();
			DeletedContainerFile deletedContainerFile = new DeletedContainerFile(containerFileDir, this);
			if (deletedContainerFile.exists()) {
				try {
					deletedContainerFile.read();
				} catch (PersistenceException e) {
					debug.warning("Fehler beim Einlesen der Datei, die gelöschte Container enthält: " + deletedContainerFile.getFile() +
							". Ungültige Einträge werden ignoriert.", e);
				}
				for (long containerId : deletedContainerFile.containers()) {
					try {
						ContainerManagementData headers = deletedContainerFile.headers(containerId);
						ActivePersistenceDirectory.updateContainerIndex(headers, containerHeaderIndex);
						containerIndexToMinDataIndex.put(containerId,
								headers.getContainerHeaderParamAsLong(ContainerManagementInformation.CHP_DATA_IDX_MIN));
					} catch (PersistenceException e) {
						throw new IndexException("Fehler beim Indizieren der Daten eines gelöschten Containers (" + containerId + ")",
								deletedContainerFile.getFile(), e);
					}
				}
			}
			try (DirectoryStream<Path> paths = Files.newDirectoryStream(dir)) {
				for (Path path : paths) {
					Duration durationOfRestore = Duration.between(startTime, Instant.now());
					if (durationOfRestore.compareTo(nextWarnTimeLimit) > 0) {
						try {
							nextWarnTimeLimit = nextWarnTimeLimit.multipliedBy(2);
						} catch (ArithmeticException ignored) {
						}
						warnAboutSlowRestore(containerFileDir, containerHeaderIndex.numEntries(), durationOfRestore);
					}
					if (ContainerFile.isContainerFile(path)) {
						try {
							executeOnContainer(path, () -> {
								try (BasicContainerFileHandle handle = new BasicContainerFileHandle(path, containerFileDir)) {
									handle.readContainerHeader();
									ActivePersistenceDirectory.updateContainerIndex(handle, containerHeaderIndex);
									if (rebuildResult != null) {
										rebuildResult.containerRead++;
									}
									containerIndexToMinDataIndex.put(handle.getContainerId(), PersistenceDirectory.getMinimumDataIndex(handle));
								}
							});
						} catch (PersistenceException e) {
							debug.warning("Container " + path + " konnte nicht gelesen werden und wird ignoriert.");
						}
					}
				}
			}

			HashSet<Long> knownMinDataIndexes = new HashSet<>((int) containerHeaderIndex.numEntries());

			// Inhalte überprüfen, Fehler wie doppelte offene Container erkennen und beheben
			IndexResult<ContainerManagementInformation> contents = containerHeaderIndex.query();
			for (int i = 0; i < contents.size(); i++) {
				if (hasError(contents, i, knownMinDataIndexes, containerIndexToMinDataIndex, !containerFileDir.archiveDataKind().isRequested())) {
					long containerId = contents.get(i, ContainerManagementInformation.CHP_CONT_ID);

					Path contFile = getPath(containerFileDir).resolve(ContainerFile.getContainerFileName(containerId));

					// bekanntes Problem aus alten Archivversionen, höheres Warnlevel würde vermutlich Anwender irritieren 
					debug.fine("Container \"" + contFile + "\" enthält inkonsistente Daten und wird ignoriert.");

					if (dataIdentificationManager.shouldDeleteBrokenContainers()) {
						renameContainerFile(contFile);
					}
					if (rebuildResult != null) {
						rebuildResult.containerIgnored++;
					}

					containerHeaderIndex.removeAll(
							ImmutableMap.of(ContainerManagementInformation.CHP_CONT_ID, new LongRange(containerId, containerId)));
				}
			}
		} catch (Exception e) {
			throw new IndexException("IO-Fehler beim Erstellen des Verwaltungsdaten-Index", containerHeaderIndex.getFile(), e);
		}
	}

	/**
	 * Bestimmt, ob ein Eintrag im Verwaltungsdatenindex fehlerhaft ist
	 *
	 * @param contents             Indexinhalte
	 * @param i                    Zu prüfende Zeile im Index
	 * @param knownMinDataIndexes  Bereits bekannte Minimum-Datenindex-Werte im Container (Duplikate sind ein Fehler), dieses Set wird von der Methode
	 *                             ergänzt.
	 * @param contIdToMinDataIndex Bekannte Minimum-Datenindex-Werte je Container-ID
	 * @param checkIndexMonotonic  Soll geprüft werden, ob der Datenindex-Minimum-Wert monoton ist?
	 * @return true falls Fehler vorhanden, sonst false
	 */
	public boolean hasError(final IndexResult<ContainerManagementInformation> contents, final int i, final HashSet<Long> knownMinDataIndexes,
	                        final Map<Long, Long> contIdToMinDataIndex, boolean checkIndexMonotonic) {

		// Container ist offen, obwohl es nicht der letzte Container ist → Fehler
		if (i != contents.size() - 1 && contents.getSignedInt(i, ContainerManagementInformation.CHP_ANZ_DS) == ContainerHdr.CONT_UNCLOSED) {
			return true;
		}

		// oder Container hat einen doppelten DataIndexMin-Wert (darf nicht vorkommen)
		long myContId = contents.get(i, ContainerManagementInformation.CHP_CONT_ID);
		final Long dataIndexMinValue = contIdToMinDataIndex.get(myContId);

		if (dataIndexMinValue == null) {
			// Minimum-Datenindex nicht bekannt, Container gelöscht?
			return true;
		}

		// Datenindex nicht monoton?
		if (checkIndexMonotonic && i > 0) {
			long prevContId = contents.get(i - 1, ContainerManagementInformation.CHP_CONT_ID);
			final Long prevDataIndexMinValue = contIdToMinDataIndex.get(prevContId);
			if (prevDataIndexMinValue > 0 && dataIndexMinValue <= prevDataIndexMinValue) {
				// Datenindex ist kleiner als vorhergehender → Fehler
				return true;
			}
		}

		return !knownMinDataIndexes.add(dataIndexMinValue);
	}

	/**
	 * Die Methode warnAboutSlowRestore warnt den Benutzer, wenn die Wiederherstellung eines Verzeichnisses sehr lange dauert.
	 *
	 * @param location        von Typ IdContainerFileDir
	 * @param currentEntries  von Typ long
	 * @param currentDuration von Typ Duration
	 */
	public void warnAboutSlowRestore(final IdContainerFileDir location, final long currentEntries, Duration currentDuration) {

		if (!largeDataIdentifications.add(location)) {
			return;
		}

		debug.warning("Die Wiederherstellung des Verwaltungsdatenindexes für die Datenidentifikation " + dataIdentificationManager.formatContainerDirectory(location) +
				" im Verzeichnis '" + getPath(location) + "' benötigt länger als " + currentDuration.toMinutes() +
				" Minuten. Es wurden bereits " + currentEntries +
				" Containerdateien indiziert. Es sollte geprüft werden, ob eine Software übermäßig viele Daten generiert, ob viele " +
				"Datenzeitsprünge auftreten oder ob der Vorhaltezeitraum verkürzt werden kann. Falls fehlerhafte Daten archiviert wurden, " +
				"sollten diese ggf. endgültig gelöscht werden.");

	}

	public final PersistenceDirectoryLayoutInstance getLayoutInstance() {
		return layoutInstance;
	}

	/**
	 * Gibt zu einer Datenidentifikation das zugehörige Verzeichnis zurück. Die Simulationsvariante der
	 * Datenidentifikation muss mit {@link #getSimulationVariant()} übereinstimmen.
	 *
	 * @param dataIdentification Datenidentifikation
	 * @return Verzeichnis
	 */
	@NotNull
	@Contract(pure = true)
	public Path getPath(IdDataIdentification dataIdentification) {
		if (dataIdentification.getSimVariant() != layoutInstance.simulationVariant()) {
			throw new IllegalArgumentException("Falsche übergebene Simulationsvariante");
		}
		return layoutInstance.getPath(dataIdentification);
	}

	/**
	 * Gibt zu einem ContainerDirectory das zugehörige Verzeichnis zurück. Die Simulationsvariante der
	 * Datenidentifikation muss mit {@link #getSimulationVariant()} übereinstimmen.
	 *
	 * @param containerDirectory Datenidentifikation und Archivdatenart, die das Containerverzeichnis identifizieren
	 * @return Verzeichnis
	 */
	@NotNull
	@Contract(pure = true)
	public Path getPath(ContainerDirectory containerDirectory) {
		if (containerDirectory.getSimVariant() != layoutInstance.simulationVariant()) {
			throw new IllegalArgumentException("Falsche übergebene Simulationsvariante");
		}
		return layoutInstance.getPath(containerDirectory);
	}

	/**
	 * Gibt die Verwaltung der Indexe zurück
	 *
	 * @return Klasse zum zugriff auf Indexdateien in diesem Verzeichnis
	 */
	public IndexTree getIndexTree() {
		return indexTree;
	}

	/**
	 * Gibt ein Set zurück, dass alle bekannten Containerverzeichnisse enthält, die "übermäßig viel" Daten enthalten. Das Set ist thread-safe und
	 * änderbar.
	 *
	 * @return Set mit Containerverzeichnissen
	 */
	public Set<IdContainerFileDir> getLargeDataIdentifications() {
		return Collections.unmodifiableSortedSet(largeDataIdentifications);
	}

	/**
	 * Gibt den Pfad der Lockdatei zurück (unabhängig davon, ob diese gerade vorhanden ist).
	 *
	 * @return Potenzieller Pfad der Lockdatei
	 */
	public Path getLockFile() {
		return getBasePath().resolve(LockFileManager.ACCESS_FLAG_FILE_NAME);
	}

	/**
	 * Die Methode warnAboutHugeContainerDirectory warnt den Benutzer, wenn zu viele Container in einem Containerverzeichnis gespeichert wurden.
	 *
	 * @param location   von Typ IdContainerFileDir
	 * @param numEntries von Typ long
	 */
	public void warnAboutHugeContainerDirectory(final IdContainerFileDir location, final long numEntries) {

		if (!largeDataIdentifications.add(location)) return;

		debug.warning("Für die Datenidentifikation " + dataIdentificationManager.formatContainerDirectory(location)
				+ " im Verzeichnis '" + getPath(location) + "' sind eine große Zahl (" + numEntries +
				") Container vorhanden, was zu Problemen und schlechter Performance führen kann. Es sollte geprüft werden, ob eine Software übermäßig viele Daten generiert, ob viele Datenzeitsprünge auftreten oder ob der Vorhaltezeitraum verkürzt werden kann. Falls fehlerhafte Daten archiviert wurden, sollten diese ggf. endgültig gelöscht werden.");

	}

	/**
	 * Die Methode openDeletedContainerFile öffnet die Datei mit den Informationen zu den gelöschten Containern
	 *
	 * @param containerDirectory Referenz auf die gelockte Datenidentifikation und Datenart für den Zugriff auf Containerdaten
	 *                           Containerverzeichnis
	 * @return DeletedContainerFile
	 * @throws PersistenceException           Lesefehler
	 * @throws SynchronizationFailedException Fehler bei Index-Synchronisation
	 */
	public DeletedContainerFile openDeletedContainerFile(final LockedContainerDirectory containerDirectory)
			throws PersistenceException, SynchronizationFailedException {
		DeletedContainerFile deletedContainerFile = new DeletedContainerFile(containerDirectory, this);
		if (!deletedContainerFile.exists()) {
			try {
				initiallyBuildDeletedContainerFile(containerDirectory, deletedContainerFile);
			} catch (IndexException e) {
				throw new PersistenceException(e);
			}
		} else {
			try {
				deletedContainerFile.read();
			} catch (PersistenceException e) {
				debug.warning("Fehler beim Einlesen der Datei, die gelöschte Container enthält: " + deletedContainerFile.getFile() +
						". Ungültige Einträge werden ignoriert.", e);
			}
		}
		return deletedContainerFile;
	}

	/**
	 * Erstellt erstmalig die Datei, die die gelöschten Container mit Datenindexbereichen enthält. Verkürzte Containerdateien werden dabei konvertiert
	 * und umgewandelt.
	 *
	 * @param containerDirectory   Referenz auf die gelockte Datenidentifikation und Datenart für den Zugriff auf Containerdaten
	 *                             Containerdatenverzeichnis
	 * @param deletedContainerFile Datei, die erstellt werden soll
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 * @throws IndexException       Lesefehler der Indexe (z. B. korrupt)
	 */
	private void initiallyBuildDeletedContainerFile(final LockedContainerDirectory containerDirectory,
	                                                final DeletedContainerFile deletedContainerFile)
			throws PersistenceException, IndexException, SynchronizationFailedException {
		IndexResult<ContainerManagementInformation> headers = getContainerHeaders(containerDirectory);
		final List<Long> containersToDelete = new ArrayList<>();
		for (int i = 0; i < headers.size(); i++) {
			if (headers.getBoolean(i, ContainerManagementInformation.CHP_DELETED)) {
				long containerId = headers.get(i, ContainerManagementInformation.CHP_CONT_ID);
				try (ContainerFileHandle handle = accessContainer(containerDirectory, containerId)) {
					if (!handle.existsContainer()) {
						continue;
					}
					executeOnContainer(handle.getFile(), () -> {
						handle.readContainerHeader();
						// Datei öffnen und Datenindexbereiche in neue Datei übernehmen
						deletedContainerFile.addDeletedContainer(handle);
						containersToDelete.add(containerId);
					});
				}
			}
		}
		deletedContainerFile.write();
		for (Long contId : containersToDelete) {
			Path contFile = getPath(containerDirectory).resolve(ContainerFile.getContainerFileName(contId));
			try {
				Files.delete(contFile);
			} catch (IOException e) {
				debug.warning("Kann Containerdatei nicht löschen: " + contFile, e);
			}
		}
	}

	/**
	 * Erstellt einen Iterator über die Daten eines Containers.
	 * Der zurückgegebene Iterator muss nach Gebrauch geschlossen werden!
	 *
	 * @param adk                Archivdatenart
	 * @param containerId        Container-ID
	 * @param dataIdentification Datenidentifikation
	 * @return Iterator
	 * @throws PersistenceException Lesefehler im Persistenzverzeichnis
	 */
	@NotNull
	public DataIterator iterator(final ArchiveDataKind adk, final long containerId, final IdDataIdentification dataIdentification) throws PersistenceException {
		try (ContainerFileHandle handle = new ContainerFileHandle(dataIdentification.resolve(adk), containerId, true, layoutInstance)) {
			if (handle.existsContainer()) {
				return handle.iterator();
			}
		}

		// Container könnte gelöscht worden sein, Kompatibilitätscode mit alten Persistenz-Verzeichnissen:
		try (SyncKey<IdDataIdentification> indexLock = dataIdentificationManager.lockIndex(dataIdentification)) {
			ContainerHeaders headers = getContainerHeaders(new LockedContainerDirectory(indexLock, adk), containerId);
			return new DeletedMinMaxIterator(headers, adk, indexLock.getElement());
		} catch (SynchronizationFailedException | IndexException e) {
			throw new PersistenceException(e);
		}
	}

	/**
	 * Die Methode removeContainerFromIndex löscht einen Container aus den Indexen
	 *
	 * @param containerDirectory Referenz auf die gelockte Datenidentifikation und Datenart für den Zugriff auf Containerdaten
	 *                           Container-Verzeichnis
	 * @param containerId        von Typ long
	 * @throws IndexException wenn der Indexzugriff fehlschlägt
	 */
	public void removeContainerFromIndex(final LockedContainerDirectory containerDirectory, final long containerId) throws IndexException {
		Optional<? extends DataTimeIndex> dtIdx = getIndexTree().getDataTimeIndex(containerDirectory);
		if (dtIdx.isPresent()) {
			dtIdx.get().removeAll(ImmutableMap.of(IndexValues.ContainerId, new LongRange(containerId, containerId)));
		}

		if (containerDirectory.archiveDataKind().isDelayed()) {
			Optional<? extends DataIndexIndex> diIdx = getIndexTree().getDataIndexIndex(containerDirectory);
			if (diIdx.isPresent()) {
				diIdx.get().removeAll(ImmutableMap.of(IndexValues.ContainerId, new LongRange(containerId, containerId)));
			}
		}

		Optional<? extends ArchiveTimeIndex> aTimeMonoIdx = getIndexTree().getArchiveTimeIndex(containerDirectory);
		if (aTimeMonoIdx.isPresent()) {
			aTimeMonoIdx.get().removeAll(ImmutableMap.of(IndexValues.ContainerId, new LongRange(containerId, containerId)));
		}

		Optional<? extends ContainerManagementIndex> headerIdx = getIndexTree().getContainerManagementIndex(containerDirectory);
		if (headerIdx.isPresent()) {
			headerIdx.get().removeAll(ImmutableMap.of(ContainerManagementInformation.CHP_CONT_ID, new LongRange(containerId, containerId)));
		}
	}

	/**
	 * Die Methode getIndexResult führt eine Index-Abfrage durch (für Archivanfragen und ähnliche Operationen)
	 *
	 * @param containerDirectory    Referenz auf die gelockte Datenidentifikation und Datenart für den Zugriff auf Containerdaten
	 *                              Container-Verzeichnis
	 * @param sequenceSpecification der gewünschte Datenbereich, siehe {@link SequenceSpecification}
	 * @return IndexResult&lt;IndexValues&gt;
	 * @throws IndexException                 wenn ein Problem beim Zugriff auf den Index auftritt
	 * @throws SynchronizationFailedException wenn die Synchronisierung auf die Indexe fehlschlägt
	 */
	public IndexResult<IndexValues> getIndexResult(LockedContainerDirectory containerDirectory, SequenceSpecification sequenceSpecification)
			throws IndexException, SynchronizationFailedException {
		if (sequenceSpecification instanceof DataIndexSequenceSpecification spec) {
			Optional<? extends DataIndexIndex> dataIndexIndex = getIndexTree().getDataIndexIndex(containerDirectory);
			if (dataIndexIndex.isEmpty()) {
				return new AggregateIndexResult<>();
			}
			return dataIndexIndex.get().getContainerIDByDataIndex(spec.minimumIndex(), spec.maximumIndex());
		} else if (sequenceSpecification instanceof ArchiveTimeSequenceSpecification spec) {
			Optional<? extends ArchiveTimeIndex> archiveTimeIndex = getIndexTree().getArchiveTimeIndex(containerDirectory);
			if (archiveTimeIndex.isEmpty()) {
				return new AggregateIndexResult<>();
			}
			return archiveTimeIndex.get().getContainerIDByArchiveTime(spec.minimumTime(), spec.maximumTime());
		} else if (sequenceSpecification instanceof AllDataSpecification) {
			Optional<? extends DataIndexIndex> dataIndexIndex = getIndexTree().getDataIndexIndex(containerDirectory);
			if (dataIndexIndex.isEmpty()) {
				return new AggregateIndexResult<>();
			}
			return dataIndexIndex.get().query();
		}
		throw new AssertionError();
	}

	/**
	 * Die Methode rebuildIndexes baut die Indexe in einem Verzeichnis erneut auf.
	 *
	 * @param containerDirectory Containerverzeichnis im exklusiven Zugriff, in dem die Indexdateien aufgebaut werden sollen
	 * @param rebuildResult      Ergebnis für Statistiken, optional
	 * @throws IndexException wenn ein Problem beim Zugriff auf den Index auftritt
	 */
	public void rebuildIndexes(final LockedContainerDirectory containerDirectory, @Nullable RebuildResult rebuildResult) throws IndexException {
		getIndexTree().recreateIndex(containerDirectory, rebuildResult);
	}

	/**
	 * Die Methode deleteIndexes löscht alle Indexdateien in einem Verzeichnis.
	 *
	 * @param containerDirectory Containerverzeichnis im exklusiven Zugriff, in dem die Indexdateien gelöscht werden sollen
	 * @param rebuildResult      Ergebnis für Statistiken
	 * @throws IndexException wenn ein Problem beim Zugriff auf den Index auftritt
	 */
	public void deleteIndexes(final LockedContainerDirectory containerDirectory, @Nullable RebuildResult rebuildResult) throws IndexException {
		getIndexTree().deleteIndex(containerDirectory, rebuildResult);
	}

	/**
	 * Ermittelt maximale Container-ID in einem Verzeichnis.
	 *
	 * @param containerDirectory Referenz auf die gelockte Datenidentifikation und Datenart für den Zugriff auf Containerdaten
	 *                           Containerverzeichnis
	 * @return Größte Container-ID im Verzeichnis
	 * @throws IndexException Lesefehler der Indexe (z. B. korrupt)
	 */
	public long maxContainerId(final LockedContainerDirectory containerDirectory) throws IndexException {
		Optional<? extends ContainerManagementIndex> index = getIndexTree().getContainerManagementIndex(containerDirectory);
		if (index.isEmpty()) {
			return -1;
		}
		return index.get().getLast(ContainerManagementInformation.CHP_CONT_ID);
	}

	/**
	 * Gibt die Simulationsvariante dieses Verzeichnisses zurück. In jedem Persistenz-Wochenverzeichnis wird nur eine
	 * einzelne Simulationsvariante verwaltet.
	 *
	 * @return die Simulationsvariante dieses Verzeichnisses
	 */
	public int getSimulationVariant() {
		return getLayoutInstance().simulationVariant();
	}

	@Override
	public String toString() {
		return layoutInstance.basePath().toString();
	}

	@Override
	public long estimate() {
		return lastCount.get();
	}

	@Override
	public void setCount(long actualCount) {
		lastCount.set(actualCount);
	}

	/**
	 * Ermittelt zu einem Containerverzeichnis den gesamten (ggf. in diesem Wochenverzeichnis) vorliegenden Datenbereich.
	 *
	 * @param directory Containerdatenidentifikation
	 * @return Datenbereich, insbesondere {@link de.bsvrz.ars.ars.persistence.directories.cache.NoData#Instance}
	 * wenn gar keine Daten vorliegen, ein {@link de.bsvrz.ars.ars.persistence.directories.cache.ValidDataRange} sonst.
	 */
	public abstract DataRange getDataRange(LockedContainerDirectory directory) throws IndexException;

	protected DataRange computeDataRange(LockedContainerDirectory directory) {
		try {
			Optional<? extends DataIndexIndex> dataIndexIndex = getIndexTree().getDataIndexIndex(directory);
			Optional<? extends DataTimeIndex> dataTimeIndex = getIndexTree().getDataTimeIndex(directory);
			Optional<? extends ArchiveTimeIndex> archiveTimeIndex = getIndexTree().getArchiveTimeIndex(directory);
			if (dataTimeIndex.isEmpty() || dataIndexIndex.isEmpty() || archiveTimeIndex.isEmpty()) {
				// Keine Daten vorhanden
				return NoData.Instance;
			}
			return new SimpleDataRange(
					archiveTimeIndex.get().getFirst(IndexValues.ArchiveTimeMin),
					archiveTimeIndex.get().getLast(IndexValues.ArchiveTimeMax),
					dataTimeIndex.get().getFirst(IndexValues.DataTimeMin),
					dataTimeIndex.get().getLast(IndexValues.DataTimeMax),
					dataIndexIndex.get().getFirst(IndexValues.DataIndexMin),
					dataIndexIndex.get().getLast(IndexValues.DataIndexMax)
			);
		} catch (IndexException e) {
			// Sollte nicht auftreten, wenn doch wird sie in #getDataRange abgefangen
			throw new RuntimeException(e);
		}
	}

	/**
	 * Funktionales Interface für Container-Aktionen
	 */
	@FunctionalInterface
	public interface ContainerCall {
		/**
		 * Methode, die etwas ausführt und dabei Exceptions werfen kann
		 *
		 * @throws SynchronizationFailedException Synchronisierung auf Indexe fehlgeschlagen
		 * @throws PersistenceException           Persistenzfehler
		 */
		void call() throws PersistenceException, SynchronizationFailedException;
	}
}
