/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence.directories.mgmt;

import de.bsvrz.ars.ars.persistence.IdDataIdentification;
import de.bsvrz.ars.ars.persistence.PersistenceException;
import de.bsvrz.ars.ars.persistence.directories.ActivePersistenceDirectory;
import de.bsvrz.ars.ars.persistence.directories.PersistenceDirectory;
import de.bsvrz.ars.ars.persistence.directories.mgmt.lock.DirectoryIsLockedException;
import de.bsvrz.ars.ars.persistence.iter.DataGapManager;
import de.bsvrz.ars.ars.persistence.sequence.SequenceSpecification;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import org.jetbrains.annotations.Contract;

import java.io.IOException;
import java.nio.file.Path;
import java.util.List;

/**
 * Verwaltung von mehreren (zeitlich getrennten) Persistenzverzeichnissen
 */
public interface PersistenceDirectoryManager extends DataGapManager {

	/**
	 * Aktualisiert das aktive Persistenzverzeichnis basierend auf der aktuellen Archivzeit
	 *
	 * @param archTime aktuelle Archivzeit (Millis seit Epoch)
	 * @throws IOException          IO-Fehler
	 * @throws InterruptedException Unterbrochen beim Einrichten des neuen Persistenzverzeichnisses
	 * @throws PersistenceException Allgemeiner Persistenzfehler
	 */
	void updatePersistenceDirectories(long archTime) throws IOException, InterruptedException, PersistenceException;

	/**
	 * Gibt das Persistenzverzeichnis zurück, in das aktuell archiviert wird. Kann null zurückgeben,
	 * z. B. während der Initialisierung oder während noch nichts archiviert wurde.
	 *
	 * @return Aktives Persistenzverzeichnis
	 */
	@Nullable
	ActivePersistenceDirectory getActivePersistenceDirectory();

	/**
	 * Ermittelt alle Persistenzverzeichnisse, die zur angegebenen Zeitspezifikation die zugehörigen Archivdaten enthält.
	 * <p>
	 * Achtung: Da bei Archivanfragen ggf. ein initialer Zustand und ein Nachfolgedatensatz (Gültigkeitsdauer des letzten
	 * Datensatzes) ermittelt werden muss, muss ein Aufrufer ggf. noch zusätzlich einen vorherigen oder nachfolgenden
	 * Container aus einem anderen Verzeichnis eigenständig ermitteln!
	 *
	 * @param simVariant            Simulationsvariante
	 * @param sequenceSpecification Zeitspezifikation
	 * @return Collection mit Persistenzverzeichnissen, die zurückgegebene Liste ist immutable.
	 */
	List<? extends PersistenceDirectory> getPersistenceDirectories(int simVariant, SequenceSpecification sequenceSpecification);

	/**
	 * Gibt das Persistenzverzeichnis zurück, das von einer speziellen Simulation verwendet wird
	 *
	 * @param simVariant Simulationsvariante (> 0)
	 * @return zugehöriges Persistenzverzeichnis oder null (wenn keines existiert).
	 */
	@Nullable
	ActivePersistenceDirectory getSimulationPersistenceDirectory(int simVariant);

	/**
	 * Gibt das Wurzelverzeichnis der Persistenz zurück, unter dem die weiteren Persistenzverzeichnisse (je Zeitbereich)
	 * angelegt werden.
	 *
	 * @return Wurzelverzeichnis, nicht null
	 */
	@NotNull
	@Contract(pure = true)
	Path getRootPath();

	/**
	 * Initialisiert die Verzeichnisverwaltung und lädt z. B. die vorhandenen Verzeichnisse ein
	 *
	 * @throws IOException          Dateisystem-Lesefehler
	 * @throws InterruptedException Unterbrochen
	 * @throws PersistenceException Persistenz fehler
	 * @throws DirectoryIsLockedException Persistenzverzeichnis ist bereits gelockt (isActive-Datei existiert)
	 */
	void initialize() throws IOException, InterruptedException, PersistenceException, DirectoryIsLockedException;

	/**
	 * Terminiert die Verzeichnisverwaltung, markiert die Verzeichnisse nicht mehr als benutzt und löscht alle Lock-Dateien.
	 */
	void shutDown();

	/**
	 * Erstellt für die angegebene Simulationsvariante das Persistenzverzeichnis. Sollte bereits ein Verzeichnis
	 * existieren, wird es zurückgegeben.
	 *
	 * @param simVariant Simulationsvariante (größer 0)
	 * @return Zugehöriges Persistenzverzeichnis
	 */
	ActivePersistenceDirectory createSimulationDirectory(int simVariant);

	/**
	 * Löscht ein Verzeichnis einer Simulation
	 *
	 * @param directory Persistenzverzeichnis der Simulation
	 */
	void deleteSimulationDirectory(ActivePersistenceDirectory directory);

	/**
	 * Gibt den Pfad und Dateinamen einer Lückendatei für das Nachfordern zurück. In dieser Datei werden die
	 * bereits erhaltenen Datenlücken vermerkt, also wo erfolglos nachgefordert wurde,
	 * damit diese nicht sinnloserweise neu angefragt werden.
	 *
	 * @param dataIdentification Datenidentifikation
	 * @return Ein Dateipfad
	 */
	@NotNull
	@Contract(pure = true)
	Path getGapFilePath(IdDataIdentification dataIdentification);
}
