/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence.directories.mgmt;

import de.bsvrz.ars.ars.mgmt.tasks.base.TaskManager;
import de.bsvrz.ars.ars.persistence.ContainerCreator;
import de.bsvrz.ars.ars.persistence.IdDataIdentification;
import de.bsvrz.ars.ars.persistence.directories.ActivePersistenceDirectory;
import de.bsvrz.ars.ars.persistence.directories.PersistenceDirectory;
import de.bsvrz.ars.ars.persistence.directories.mgmt.lock.DirectoryIsLockedException;
import de.bsvrz.ars.ars.persistence.directories.mgmt.lock.LockFileManager;
import de.bsvrz.ars.ars.persistence.layout.ClassicPersistenceDirectoryLayout;
import de.bsvrz.ars.ars.persistence.layout.PersistenceDirectoryLayoutInstance;
import de.bsvrz.ars.ars.persistence.layout.ShortPersistenceDirectoryLayout;
import de.bsvrz.ars.ars.persistence.sequence.SequenceSpecification;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;

import java.io.IOException;
import java.nio.file.Path;
import java.util.List;
import java.util.NavigableMap;
import java.util.TreeMap;

/**
 * Implementierung von PersistenceDirectoryManager, die ein einzelnes Persistenz-(unter-)verzeichnis verwaltet.
 * Nur Simulationen werden in extra-Verzeichnisse ausgelagert, was nicht dem Verhalten des Alten Archivsystems vor
 * Version 5 entspricht. Dort waren Simulationen zusammen mit den normalen Daten im selben Verzeichnis.
 */
public class SingletonPersistenceDirectoryManager implements PersistenceDirectoryManager {

	private final ActivePersistenceDirectory persistenceDirectory;
	private final NavigableMap<Integer, ActivePersistenceDirectory> simulationDirectories = new TreeMap<>();
	private final LockFileManager lockFileManager = new LockFileManager();
	private static final Debug _debug = Debug.getLogger();
	private final ContainerCreator containerCreator;
	private final Path rootPath;

	/**
	 * Erstellt einen neuen SingletonPersistenceDirectoryManager.
	 *
	 * @param containerCreator Interface der persistenzschicht, z. B. ein {@link de.bsvrz.ars.ars.persistence.PersistenceManager}.
	 * @param rootPath         Wurzelverzeichnis
	 */
	public SingletonPersistenceDirectoryManager(ContainerCreator containerCreator, Path rootPath) {
		this.containerCreator = containerCreator;
		this.rootPath = rootPath;
		PersistenceDirectoryLayoutInstance layoutInstance = ClassicPersistenceDirectoryLayout.instance(rootPath, 0);
		this.persistenceDirectory = new ActivePersistenceDirectory(containerCreator, layoutInstance);
	}

	@Override
	public void updatePersistenceDirectories(long archTime) {
	}

	@Nullable
	@Override
	public ActivePersistenceDirectory getActivePersistenceDirectory() {
		return persistenceDirectory;
	}

	@Override
	public List<? extends PersistenceDirectory> getPersistenceDirectories(int simVariant, SequenceSpecification sequenceSpecification) {
		if (simVariant != 0) {
			ActivePersistenceDirectory simulationPersistenceDirectory = getSimulationPersistenceDirectory(simVariant);
			if (simulationPersistenceDirectory != null) {
				return List.of(simulationPersistenceDirectory);
			}
			return List.of();
		}
		return List.of(persistenceDirectory);
	}

	@NotNull
	@Override
	public Path getRootPath() {
		return persistenceDirectory.getBasePath();
	}

	@Override
	public void initialize() throws IOException, DirectoryIsLockedException {
		lockFileManager.open(persistenceDirectory.getBasePath());
	}

	@Nullable
	@Override
	public synchronized ActivePersistenceDirectory getSimulationPersistenceDirectory(int simVariant) {
		if (simVariant < 1) {
			throw new IllegalArgumentException("Keine gültige Simulationsvariante: " + simVariant);
		}
		return simulationDirectories.get(simVariant);
	}

	@Override
	public ActivePersistenceDirectory createSimulationDirectory(int simVariant) {
		if (simVariant < 1) {
			throw new IllegalArgumentException("Keine gültige Simulationsvariante: " + simVariant);
		}
		return simulationDirectories.computeIfAbsent(
				simVariant,
				(sv) -> new ActivePersistenceDirectory(containerCreator, ShortPersistenceDirectoryLayout.instance(rootPath.resolve("sim" + sv), sv))
		);
	}

	@Override
	public synchronized void shutDown() {
		try {
			lockFileManager.close(persistenceDirectory.getBasePath());
		} catch (IOException e) {
			_debug.warning("Persistenzverzeichnis konnte nicht geschlossen werden: " + persistenceDirectory.getBasePath().toAbsolutePath(), e);
		}
	}

	@Override
	public synchronized void deleteSimulationDirectory(ActivePersistenceDirectory directory) {
		TaskManager.runExclusively(
				"Simulationsverzeichnis für Simulation " + directory.getSimulationVariant() + " löschen",
				tpi -> simulationDirectories.remove(directory.getSimulationVariant(), directory)
		);
	}

	@NotNull
	@Override
	public Path getGapFilePath(IdDataIdentification dataIdentification) {
		return persistenceDirectory.getPath(dataIdentification).resolve("_datagaps.txt");
	}

	@Override
	public boolean isRangeUnavailable(long fromArchiveTime, long toArchiveTime) {
		return false;
	}
}
