/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence.directories.mgmt;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Range;
import de.bsvrz.ars.ars.mgmt.tasks.base.TaskManager;
import de.bsvrz.ars.ars.persistence.ContainerCreator;
import de.bsvrz.ars.ars.persistence.IdDataIdentification;
import de.bsvrz.ars.ars.persistence.PersistenceException;
import de.bsvrz.ars.ars.persistence.PersistenceManager;
import de.bsvrz.ars.ars.persistence.directories.ActivePersistenceDirectory;
import de.bsvrz.ars.ars.persistence.directories.ReadonlyPersistenceDirectory;
import de.bsvrz.ars.ars.persistence.directories.PersistenceDirectory;
import de.bsvrz.ars.ars.persistence.directories.mgmt.lock.DirectoryIsLockedException;
import de.bsvrz.ars.ars.persistence.directories.mgmt.lock.DirectoryStatus;
import de.bsvrz.ars.ars.persistence.directories.mgmt.lock.LockFileManager;
import de.bsvrz.ars.ars.persistence.directories.mgmt.range.TimeDomain;
import de.bsvrz.ars.ars.persistence.directories.mgmt.range.TimeRange;
import de.bsvrz.ars.ars.persistence.directories.mgmt.util.RangeUpdater;
import de.bsvrz.ars.ars.persistence.layout.PersistenceDirectoryLayoutInstance;
import de.bsvrz.ars.ars.persistence.layout.ShortPersistenceDirectoryLayout;
import de.bsvrz.ars.ars.persistence.sequence.ArchiveTimeSequenceSpecification;
import de.bsvrz.ars.ars.persistence.sequence.SequenceSpecification;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import org.jetbrains.annotations.Contract;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.*;
import java.util.stream.Stream;

/**
 * Implementierung des PersistenceDirectoryManager, die je Woche (oder anderer {@link TimeDomain}) ein eigenes
 * {@link PersistenceDirectory} anlegt. Simulationen werden davon getrennt verwaltet.
 *
 * @param <T> Typ des gewünschten Zeitabschnitts, z. B. {@link de.bsvrz.ars.ars.persistence.directories.mgmt.range.Week}
 * @author Kappich Systemberatung
 */
public class TimeBasedPersistenceDirectoryManager<T extends TimeRange<T>> implements PersistenceDirectoryManager {

	@Nullable
	private ActivePersistenceDirectory activePersistenceDirectory = null;
	private final NavigableMap<T, PersistenceDirectory> allPersistenceDirectories = new TreeMap<>();
	private final NavigableMap<Integer, ActivePersistenceDirectory> simulationDirectories = new TreeMap<>();
	private final ContainerCreator containerCreator;
	private final Path rootPath;
	private final RangeUpdater<T> rangeUpdater;
	private final TimeDomain<T> domain;
	private final LockFileManager lockFileManager = new LockFileManager();
	private static final Debug _debug = Debug.getLogger();
	private final ShortPersistenceDirectoryLayout layout = ShortPersistenceDirectoryLayout.Instance;

	/**
	 * Name des Ordners, der die {@linkplain de.bsvrz.ars.ars.persistence.gap.GapFileImpl Lückendateien} für das
	 * Nachfordern enthält.
	 */
	public static final String GAP_FILE_FOLDER_NAME = "_gaps";

	/**
	 * Erstellt eine neue Instanz
	 *
	 * @param containerCreator Persistenz-Verwaltung
	 * @param rootPath         Wurzelverzeichnis, unter dem die Wochenverzeichnisse angelegt werden sollen
	 * @param domain           Domain, die Pfadnamen in Zeitbereiche umwandelt, z. B.
	 *                         {@link de.bsvrz.ars.ars.persistence.directories.mgmt.range.WeekDomain}.
	 */
	public TimeBasedPersistenceDirectoryManager(ContainerCreator containerCreator, Path rootPath, TimeDomain<T> domain) {
		this.containerCreator = containerCreator;
		this.rootPath = rootPath;
		rangeUpdater = new RangeUpdater<>(domain);
		this.domain = domain;
	}

	public void initialize() throws IOException, InterruptedException, PersistenceException, DirectoryIsLockedException {
		List<PersistenceDirectory> persistenceDirectories;
		synchronized (this) {
			initializeDirectories();

			if (allPersistenceDirectories.isEmpty()) {
				return;
			}

			// Aktives Persistenzverzeichnis bestimmen
			PersistenceDirectory lastValue = allPersistenceDirectories.lastEntry().getValue();
			if (lastValue instanceof ActivePersistenceDirectory persistenceDirectory) {
				activePersistenceDirectory = persistenceDirectory;
			}
			persistenceDirectories = List.copyOf(allPersistenceDirectories.values());
		}

		// Alle Persistenzverzeichnisse außer dem aktuellen abschließen
		// Dies nicht synchronisiert ausführen, da Deadlockgefahr
		// (closePersistenceDirectory ruft indirekt PersistenceManager.getPersistenceDirectories auf)
		for (PersistenceDirectory value : persistenceDirectories) {
			if (value != activePersistenceDirectory && value instanceof ActivePersistenceDirectory apd) {
				closePersistenceDirectory(apd);
			}
		}
	}

	@Override
	public void updatePersistenceDirectories(long archTime) throws IOException {
		ActivePersistenceDirectory directory;
		T nextRange;

		synchronized (this) {
			nextRange = rangeUpdater.update(archTime);
			if (nextRange == null) {
				return;
			}
			directory = activePersistenceDirectory;
			if (directory != null && directory == allPersistenceDirectories.get(nextRange)) {
				return;
			}
			activePersistenceDirectory = createDirectory(nextRange);
			allPersistenceDirectories.put(nextRange, activePersistenceDirectory);
		}

		if (directory != null) {
			try {
				closePersistenceDirectory(directory);
			}
			catch (Exception e) {
				// Wenn das schiefgeht, ist das nur eine Warnung, da das Archivsystem auch noch mit 
				// "offenen" Verzeichnissen weiterarbeitet.
				_debug.warning("Unerwarteter Fehler beim Abschließen eines Persistenzverzeichnisses", e);
			}
		}
	}

	/**
	 * Schließt ein Persistenzverzeichnis ab. Diese Operation wird parallel zu den generellen Aktionen des Archivsystems durchgeführt,
	 * nur die Archivierung wird temporär pausiert, (da diese Funktion vom ArchiveTask aufgerufen wird).
	 * <p>
	 * Während das Persistenzverzeichnis geschlossen wird, werden daher keine weiteren Schreiboperationen
	 * in das gerade zu schließende Verzeichnis durchgeführt.
	 * <p>
	 * Parallel laufende lesende Operationen sind kein Problem, da die Funktionen, die Container und Indexe
	 * schließen normal synchronisieren.
	 *
	 * @param activeDirectory Zu schließendes Persistenzverzeichnis
	 * @throws InterruptedException Warten auf Schließen wurde unterbrochen
	 * @throws PersistenceException Schreibfehler
	 */
	private void closePersistenceDirectory(ActivePersistenceDirectory activeDirectory) throws InterruptedException, PersistenceException {
		Path basePath = activeDirectory.getBasePath();
		Path relativePath = getRelativePath(basePath);
		_debug.info("Schließe Verzeichnis ab: " + relativePath);
		T domain = this.domain.ofPath(relativePath);
		if(domain == null) {
			throw new PersistenceException("Zeitbereich des Verzeichnisses '" + relativePath + "' konnte nicht bestimmt werden.");
		}
		activeDirectory.closePermanently();
		try {
			lockFileManager.writeProtectPermanently(basePath);
		} catch (IOException e) {
			throw new PersistenceException("Fehler beim Löschen der Lock-Datei in " + activeDirectory);
		}
		synchronized (this) {
			PersistenceDirectory immutableInstance = new ReadonlyPersistenceDirectory(containerCreator, activeDirectory.getLayoutInstance());
			immutableInstance.setCount(activeDirectory.estimate());
			allPersistenceDirectories.put(domain, immutableInstance);
		}
	}

	/**
	 * Aktion für {@link #executeOnSubDirectories(PathAction)}.
	 */
	public interface PathAction {

		/**
		 * Führt für das gefundene Wochenverzeichnis eine beliebige Aktion aus
		 *
		 * @param path Wochenverzeichnis
		 * @throws PersistenceException Beliebiger Fehler, wird an die {@link #executeOnSubDirectories(PathAction)}-Methode
		 *                              weitergegeben.
		 */
		void run(Path path) throws PersistenceException, DirectoryIsLockedException;
	}

	private void initializeDirectories() throws IOException, PersistenceException, DirectoryIsLockedException {
		Files.createDirectories(rootPath);
		executeOnSubDirectories(this::initializeExistingDirectory);
	}

	/**
	 * Führt einen rekursiven Verzeichnisdurchlauf aus und führt für jedes gefundene Wochenverzeichnis den angegebenen Befehl aus
	 *
	 * @param pathAction Auszuführende Aktion
	 * @throws PersistenceException Fehler bei Aktion
	 * @throws IOException          Fehler beim Verzeichnisdurchlauf
	 */
	public void executeOnSubDirectories(PathAction pathAction) throws PersistenceException, IOException, DirectoryIsLockedException {
		executeOnSubDirectories(rootPath, domain.getMaxDepth(), pathAction);
	}

	private void executeOnSubDirectories(Path p, int maxDepth, PathAction pathAction) throws PersistenceException, IOException, DirectoryIsLockedException {
		try (Stream<Path> pathStream = Files.list(p)) {
			List<Path> paths = pathStream.sorted().toList();
			for (Path path : paths) {
				// Unterverzeichnisse vom Jahresverzeichnis suchen
				if (Files.isDirectory(path)) {
					// Unterverzeichnisse vom Jahresverzeichnis suchen
					if (path.getFileName().toString().equals(PersistenceManager.BACKUP_DIR_NAME)) {
						// Verzeichnis mit Backup-Definitionen ignorieren
						continue;
					}
					if (getId(path) != null) {
						// Es handelt sich um ein Persistenz-Unterverzeichnis
						pathAction.run(path);
						continue;
					}
					if (maxDepth > 1) {
						// Weitere Verzeichnisse rekursiv suchen
						executeOnSubDirectories(path, maxDepth - 1, pathAction);
					}
				}
			}
		}
	}

	@Contract(pure = true)
	@NotNull
	public TimeDomain<T> getDomain() {
		return domain;
	}

	/**
	 * Gibt die Woche zu einem Dateisystem-Pfad zurück, welcher ein Wochenverzeichnis darstellt
	 *
	 * @param it Dateisystem-Pfad
	 * @return Woche oder null, falls die Woche nicht bestimmbar ist.
	 */
	@Contract(pure = true)
	@Nullable
	public T getId(Path it) {
		return domain.ofPath(getRelativePath(it));
	}

	@Contract(pure = true)
	@NotNull
	private Path getRelativePath(@NotNull Path path) {
		if (path.isAbsolute() == rootPath.isAbsolute()) {
			return rootPath.relativize(path);
		}
		if(!path.isAbsolute()) {
			return path;
		}
		return rootPath.toAbsolutePath().relativize(path);
	}

	@NotNull
	private PersistenceDirectory initializeExistingDirectory(Path persistenceDirectory) throws PersistenceException, DirectoryIsLockedException {
		try {
			DirectoryStatus status = lockFileManager.open(persistenceDirectory);
			PersistenceDirectory tmp;
			PersistenceDirectoryLayoutInstance layoutInstance = layout.createInstance(persistenceDirectory, 0);
			if (status == DirectoryStatus.Writable) {
				tmp = new ActivePersistenceDirectory(containerCreator, layoutInstance);
			} else {
				tmp = new ReadonlyPersistenceDirectory(containerCreator, layoutInstance);
			}
			allPersistenceDirectories.put(getId(persistenceDirectory), tmp);
			return tmp;
		} catch (IOException e) {
			throw new PersistenceException("Fehler beim Initialisieren des Verzeichnisses: " + persistenceDirectory, e);
		}
	}

	@NotNull
	private ActivePersistenceDirectory createDirectory(T timeKey) throws IOException {
		Path persistenceDirectory = rootPath.resolve(domain.getPath(timeKey));
		lockFileManager.forceWritable(persistenceDirectory);
		ActivePersistenceDirectory tmp = new ActivePersistenceDirectory(containerCreator, layout.createInstance(persistenceDirectory, 0));
		allPersistenceDirectories.put(timeKey, tmp);
		return tmp;
	}

	@Override
	@Nullable
	public synchronized ActivePersistenceDirectory getActivePersistenceDirectory() {
		return activePersistenceDirectory;
	}

	@Override
	public synchronized List<? extends PersistenceDirectory> getPersistenceDirectories(int simVariant, SequenceSpecification sequenceSpecification) {
		if (simVariant != 0) {
			ActivePersistenceDirectory simulationPersistenceDirectory = getSimulationPersistenceDirectory(simVariant);
			if (simulationPersistenceDirectory != null) {
				return List.of(simulationPersistenceDirectory);
			}
			return List.of();
		}
		if (sequenceSpecification instanceof ArchiveTimeSequenceSpecification specification) {
			Range<Long> specificationMillisRange = specification.asMillisRange();
			return allPersistenceDirectories.entrySet().stream().filter(it -> specificationMillisRange.isConnected(it.getKey().getEpochMillisRange())).map(Map.Entry::getValue).filter(Objects::nonNull).toList();
		}
		return ImmutableList.copyOf(allPersistenceDirectories.values());
	}

	@Nullable
	@Override
	public synchronized ActivePersistenceDirectory getSimulationPersistenceDirectory(int simVariant) {
		if (simVariant < 1) {
			throw new IllegalArgumentException("Keine gültige Simulationsvariante: " + simVariant);
		}
		return simulationDirectories.get(simVariant);
	}

	public synchronized ActivePersistenceDirectory createSimulationDirectory(int simVariant) {
		if (simVariant < 1) {
			throw new IllegalArgumentException("Keine gültige Simulationsvariante: " + simVariant);
		}
		return simulationDirectories.computeIfAbsent(simVariant, (sv) -> new ActivePersistenceDirectory(containerCreator, layout.createInstance(rootPath.resolve("sim" + sv), sv)));
	}

	@NotNull
	@Override
	public Path getRootPath() {
		return rootPath;
	}

	@Override
	public synchronized void shutDown() {
		for (PersistenceDirectory value : allPersistenceDirectories.values()) {
			try {
				lockFileManager.close(value.getBasePath());
			} catch (IOException e) {
				_debug.warning("Persistenzverzeichnis konnte nicht geschlossen werden: " + value.getBasePath().toAbsolutePath(), e);
			}
		}
	}

	@Override
	public synchronized void deleteSimulationDirectory(ActivePersistenceDirectory directory) {
		TaskManager.runExclusively(
				"Simulationsverzeichnis für Simulation " + directory.getSimulationVariant() + " löschen",
				tpi -> simulationDirectories.remove(directory.getSimulationVariant(), directory)
		);
	}

	@NotNull
	@Override
	public Path getGapFilePath(IdDataIdentification dataIdentification) {
		return getRootPath().resolve(GAP_FILE_FOLDER_NAME).resolve(generateFileName(dataIdentification));
	}

	static Path generateFileName(IdDataIdentification dataIdentification) {
		// Ähnlich wie de.bsvrz.ars.ars.persistence.layout.ShortPersistenceDirectoryLayout
		// aber Datenart weglassen und die Objekt-ID gleich als Dateinamen benutzen
		String atgName = ShortPersistenceDirectoryLayout.encode(dataIdentification.getAtgId());
		String aspName = ShortPersistenceDirectoryLayout.encode(dataIdentification.getAspectId());
		String objName = ShortPersistenceDirectoryLayout.encode(dataIdentification.getObjectId());
		return Paths.get(atgName, aspName, objName + ".datagaps");
	}

	/**
	 * Gibt zu einer Woche das Persistenzverzeichnis zurück
	 * @param directoryTimeRange Woche
	 * @return Persistenzverzeichnis
	 */
	public synchronized PersistenceDirectory getPersistenceDirectory(@NotNull T directoryTimeRange) {
		return allPersistenceDirectories.get(Objects.requireNonNull(directoryTimeRange));
	}

	/**
	 * Wirft ein Persistenzverzeichnis aus (entfernt es aus dem aktiven Zugriff)
	 * @param directoryTimeRange Woche, die ausgeworfen werden soll
	 * @return true: erfolgreich, false: Verzeichnis war nicht vorhanden oder Operation wurde unterbrochen
	 */
	public synchronized boolean removePersistenceDirectory(T directoryTimeRange) {
		Path fileName = domain.getPath(directoryTimeRange);
		try {
			return TaskManager.computeExclusively("Verzeichnis " + fileName + " auswerfen", tpi -> {
				PersistenceDirectory directory = allPersistenceDirectories.get(directoryTimeRange);
				if (directory == null || directory instanceof ActivePersistenceDirectory) {
					return false;
				}
				try {
					lockFileManager.close(directory.getBasePath());
				} catch (IOException e) {
					_debug.warning("Persistenzverzeichnis konnte nicht geschlossen werden: " + directory.getBasePath().toAbsolutePath(), e);
				}
				allPersistenceDirectories.remove(directoryTimeRange);
				return true;
			});
		} catch (InterruptedException e) {
			return false;
		}
	}

	/**
	 * Integriert ein Verzeichnis in den aktiven Zugriff
	 * @param directoryTimeRange Woche
	 * @return true: Verzeichnis wurde integriert, false: Verzeichnis konnte nicht integriert werden oder war bereits im Zugriff
	 */
	public boolean addPersistenceDirectory(T directoryTimeRange) {
		try {
			PersistenceDirectory existingDirectory;
			synchronized (this) {
				PersistenceDirectory directory = allPersistenceDirectories.get(directoryTimeRange);
				if (directory != null) {
					return false;
				}
				Path path = domain.getPath(directoryTimeRange);
				existingDirectory = initializeExistingDirectory(rootPath.resolve(path));
			}
			if (existingDirectory instanceof ActivePersistenceDirectory persistenceDirectory) {
				closePersistenceDirectory(persistenceDirectory);
			}
			return true;
		} catch (InterruptedException | PersistenceException | DirectoryIsLockedException e) {
			return false;
		}
	}

	@Override
	public boolean isRangeUnavailable(long fromArchiveTime, long toArchiveTime) {
		T fromInterval = domain.ofEpochMillis(fromArchiveTime);
		T toInterval = domain.ofEpochMillis(toArchiveTime);
		if (fromInterval.stepsUntil(toInterval) <= 0) {
			return false;
		}
		T temp = fromInterval.next();
		while (!temp.equals(toInterval)) {
			if (getPersistenceDirectory(temp) == null) {
				return true;
			}
			temp = temp.next();
		}
		return false;
	}
}
