/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence.directories.mgmt.lock;

import de.bsvrz.sys.funclib.debug.Debug;

import java.io.IOException;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Diese Klasse verwaltet den Zugriff auf Persistenzverzeichnisse. Ein Persistenzverzeichnis kann entweder
 * "offen" sein, sodass weitere Archivdaten angehängt wurden, oder es ist nur lesbar. Beschreibbare Verzeichnisse werden
 * über die {@link #NOT_PROTECTED_FLAG_FILE_NAME}-Datei markiert.
 * <p>
 * Persistenzverzeichnisse können außerdem sich im Zugriff befinden oder nicht. Verzeichnisse im zugriff werden über
 * die {@link #ACCESS_FLAG_FILE_NAME}-Daten identifiziert.
 */
public class LockFileManager {

	private static final Debug _debug = Debug.getLogger();

	public static final String ACCESS_FLAG_FILE_NAME = "_isActive.flag";
	public static final String NOT_PROTECTED_FLAG_FILE_NAME = ".isOpen.flag";

	private final ConcurrentHashMap<Path, DirectoryToken> cache = new ConcurrentHashMap<>();

	public void createWritable(Path path) throws IOException {
		DirectoryToken token = openPath(path);
		token.createWritable();
	}

	public DirectoryStatus open(Path path) throws IOException, DirectoryIsLockedException {
		DirectoryToken token = openPath(path);
		try {
			token.access();
		} catch (FileAlreadyExistsException e) {
			throw new DirectoryIsLockedException("Die Sperrdatei \"" + e.getFile() + "\" zeigt an, dass bereits ein Archivsystem im Persistenzverzeichnis aktiv ist.", e);
		}
		return token.writeStatus();
	}

	public void forceWritable(Path path) throws IOException {
		DirectoryToken token = openPath(path);
		token.forceWritable();
	}

	public void writeProtectPermanently(Path path) throws IOException {
		DirectoryToken token = openPath(path);
		token.writeProtectPermanently();
	}

	public void close(Path path) throws IOException {
		DirectoryToken token = openPath(path);
		token.close();
	}

	private DirectoryToken openPath(Path path) {
		return cache.computeIfAbsent(path, DirectoryToken::new);
	}

	private static class DirectoryToken {
		private final Path path;
		private final Path accessFile;
		private final Path openFile;

		boolean accessed = false;
		DirectoryStatus status = null;

		public DirectoryToken(Path path) {
			this.path = path;
			this.accessFile = path.resolve(ACCESS_FLAG_FILE_NAME);
			this.openFile = path.resolve(NOT_PROTECTED_FLAG_FILE_NAME);
		}

		public void createWritable() throws IOException {
			if (Files.isDirectory(path) && !Files.deleteIfExists(path)) {
				throw new FileAlreadyExistsException(path.toAbsolutePath().toString());
			}
			Files.createDirectories(path);
			Files.createFile(accessFile);
			Files.createFile(openFile);
			status = DirectoryStatus.Writable;
			accessed = true;
		}

		public void forceWritable() throws IOException {
			Files.createDirectories(path);
			if (!Files.exists(accessFile)) {
				Files.createFile(accessFile);
			}
			if (!Files.exists(openFile)) {
				Files.createFile(openFile);
			}
			status = DirectoryStatus.Writable;
			accessed = true;
		}

		public void access() throws IOException {
			if (!Files.isDirectory(path)) {
				throw new NoSuchFileException(path.toAbsolutePath().toString());
			}
			if (Files.exists(openFile)) {
				Files.createFile(accessFile);
				status = DirectoryStatus.Writable;
			} else {
				try {
					Files.createFile(accessFile);
				} catch (FileAlreadyExistsException e) {
					_debug.fine("Ignoriere existierende Lock-Datei in geschlossenem Verzeichnis", e);
				}
				status = DirectoryStatus.Finalized;
			}
			accessed = true;
		}

		public DirectoryStatus writeStatus() {
			return status;
		}

		public void writeProtectPermanently() throws IOException {
			assert accessed;
			if (status != DirectoryStatus.Writable) {
				throw new IOException("Verzeichnis ist nicht beschreibbar: " + path.toAbsolutePath());
			}
			Files.deleteIfExists(openFile);
		}

		public void close() throws IOException {
			Files.deleteIfExists(accessFile);
			status = null;
			accessed = false;
		}
	}
}
