/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence.directories.mgmt.range;

import de.bsvrz.sys.funclib.kappich.annotations.NotNull;

import java.util.AbstractSequentialList;
import java.util.ListIterator;
import java.util.NoSuchElementException;

/**
 * Hilfsklasse für eine Liste von {@link TimeRange}-Objekten, die durch die Angabe eines Starts und eines
 * Endes definiert ist.
 * <p>
 * Diese Klasse ist Immutable.
 * </p>
 *
 * @param <T>
 */
public class RangeList<T extends TimeRange<T>> extends AbstractSequentialList<T> {
	private final T start;
	private final T end;

	public RangeList(T start, T end) {
		this.start = start;
		this.end = end;
		if (start.stepsUntil(end) < 0) {
			throw new IllegalArgumentException("Start ist hinter Ende");
		}
	}

	@NotNull
	@Override
	public ListIterator<T> listIterator(int index) {
		MyIterator iterator = new MyIterator();
		for (int i = 0; i < index; i++) {
			iterator.next();
		}
		return iterator;
	}

	@Override
	public int size() {
		long size = 1 + start.stepsUntil(end);
		if (size < 0 || size > Integer.MAX_VALUE) {
			return Integer.MAX_VALUE;
		}
		return (int) size;
	}

	@Override
	public boolean contains(Object o) {
		//noinspection unchecked
		return start.compareTo((T) o) >= 0 && end.compareTo((T) o) <= 0;
	}

	private class MyIterator implements ListIterator<T> {

		T nextItem = start;
		int nextIndex = 0;

		@Override
		public boolean hasNext() {
			return !nextItem.equals(end.next());
		}

		@Override
		public T next() {
			if (!hasNext()) {
				throw new NoSuchElementException();
			}
			nextIndex++;
			T tmp = nextItem;
			nextItem = tmp.next();
			return tmp;
		}

		@Override
		public boolean hasPrevious() {
			return !nextItem.equals(start);
		}

		@Override
		public T previous() {
			if (!hasPrevious()) {
				throw new NoSuchElementException();
			}
			nextIndex--;
			nextItem = nextItem.previous();
			return nextItem;
		}

		@Override
		public int nextIndex() {
			return nextIndex;
		}

		@Override
		public int previousIndex() {
			return nextIndex - 1;
		}

		@Override
		public void remove() {
			throw new UnsupportedOperationException();
		}

		@Override
		public void set(T t) {
			throw new UnsupportedOperationException();
		}

		@Override
		public void add(T t) {
			throw new UnsupportedOperationException();
		}
	}
}
