/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.persistence.directories.mgmt.range;

import com.google.common.collect.Range;

import java.time.LocalDate;

/**
 * Interface für einen Zeitbereich, in den Persistenzverzeichnisse aufgeteilt werden können (z. B. Wochen)
 *
 * @param <T> Referenz auf den eigenen Typ (für {@link Comparable} usw.)
 */
public interface TimeRange<T> extends Comparable<T> {

	/**
	 * Konvertiert den Zeitbereich in einen Bereich von Tagen.
	 *
	 * @return Zeitbereich asl Epoch Millis
	 */
	Range<LocalDate> getDayRange();

	/**
	 * Konvertiert den Zeitbereich in eine Long-Range, die Epoch-Millisekunden repräsentiert.
	 *
	 * @return Zeitbereich asl Epoch Millis
	 */
	Range<Long> getEpochMillisRange();

	/**
	 * Gibt das nachfolgende Verzeichnis zurück
	 *
	 * @return nachfolgendes Verzeichnis
	 */
	T next();

	/**
	 * Gibt das zeitlich davorliegende Verzeichnis zurück
	 *
	 * @return zeitlich direkt vorher liegendes Verzeichnis
	 */
	T previous();

	/**
	 * Gibt den Abstand zwischen diesem Zeitbereich und dem angegebenen Zeitbereich zurück.
	 * <p>
	 * Falls {@code other} gleich {@code this} ist, wird 0 zurückgegeben.
	 * </p>
	 * <p>
	 * Falls {@code other} gleich {@code this.next()} ist, wird 1 zurückgegeben.
	 * </p>
	 * <p>
	 * Falls {@code other} gleich {@code this.next().next()} ist, wird 2 zurückgegeben.
	 * </p>
	 * <p>
	 * Falls {@code other} gleich {@code this.previous()} ist, wird -1 zurückgegeben.
	 * </p>
	 * <p>und So weiter, d. h. diese Methode {@code other - this} wenn man annimmt, dass die Zeitbereiche in Richtung Zukunft
	 * laufend durchnummeriert werden.</p>
	 *
	 * @param other Anderer Zeitbereich
	 * @return Anzahl "Schritte" zwischen this und other
	 */
	long stepsUntil(T other);
}
